"""
Componente reutilizable de filtro de empleados.
Integra con Issue #402 - API de empleados con gating PRO.
"""

from dash import dcc, html
import dash_bootstrap_components as dbc
from components.base import Title
from styles.design_tokens import COLORS


def EmployeeFilter(id_prefix: str, pro_only: bool = True):
    """
    Crea un dropdown de empleados con gating PRO opcional.

    Args:
        id_prefix: Prefijo para los IDs (ej: "generics", "ventalibre")
        pro_only: Si True, muestra tooltip PRO para usuarios FREE

    Returns:
        dbc.Col con el componente de empleados

    Notes:
        - Las opciones se cargan dinámicamente via callback desde /api/v1/employees/{pharmacy_id}
        - Incluye valor especial "__sin_empleado__" para ventas sin empleado
        - Si usuario FREE y pro_only=True, dropdown está deshabilitado con tooltip

    Example:
        EmployeeFilter("generics", pro_only=True)
        # Genera IDs: "generics-employee-filter", "generics-employee-tooltip"
    """
    dropdown_component = dcc.Dropdown(
        id=f"{id_prefix}-employee-filter",
        multi=True,
        placeholder="Todos los empleados",
        options=[],  # Se carga dinámicamente via callback
        value=[],  # Default: todos los empleados
        className="w-100",
        persistence=True,
        persistence_type="session",
        clearable=True,
        searchable=True,
        style={"width": "100%"},
    )

    # Wrapper para tooltip PRO si es necesario
    dropdown_wrapper = html.Div(
        [
            dropdown_component,
            # Tooltip PRO (se muestra via callback si usuario FREE)
            dbc.Tooltip(
                [
                    html.Div([
                        html.I(className="fas fa-crown me-2"),
                        "Función disponible en plan PRO"
                    ]),
                    html.Small("Filtra ventas por empleado específico", className="d-block mt-1")
                ],
                target=f"{id_prefix}-employee-filter",
                id=f"{id_prefix}-employee-tooltip",
                placement="top",
                style={"display": "none"}  # Se muestra via callback
            )
        ],
        id=f"{id_prefix}-employee-wrapper"
    )

    return dbc.Col([
        Title("👥 Filtro de Empleados", level=6, className="mb-2"),
        dropdown_wrapper,
        # Badge informativo
        dbc.Badge(
            html.Span([
                html.I(className="fas fa-info-circle me-1"),
                "Vacío = Todos los empleados"
            ]),
            color="info",
            className="mt-2",
            pill=True,
            style={"fontSize": "0.75rem"}
        )
    ], lg=4, md=6, sm=12, className="mb-3 mb-lg-0")
