"""
Componente contenedor genérico para paneles de filtros.
Proporciona estructura consistente para /generics, /ventalibre, /prescription.
"""

from dash import html
import dash_bootstrap_components as dbc
from components.base import BaseCard, Title
from styles.design_tokens import COLORS, SPACING, SHADOWS


def CommonFilterPanel(
    filters_row: list,
    title: str = "Filtros de Análisis",
    sticky: bool = True,
    show_divider: bool = True
):
    """
    Crea un panel de filtros reutilizable con estructura consistente.

    Args:
        filters_row: Lista de dbc.Col con los componentes de filtros
        title: Título del panel (default: "Filtros de Análisis")
        sticky: Si True, panel se mantiene visible al hacer scroll
        show_divider: Si True, muestra línea divisoria inferior

    Returns:
        dbc.Row con el panel de filtros completo

    Architecture:
        - Sticky header que permanece visible al scroll
        - Responsive: 3 columnas en desktop, 2 en tablet, 1 en mobile
        - z-index 100 para overlay sobre contenido
        - Background blanco para contraste
        - Shadow para profundidad visual

    Example:
        CommonFilterPanel(
            filters_row=[
                DateRangeFilter("generics"),
                EmployeeFilter("generics"),
                ApplyButton("generics")
            ],
            title="Filtros de Genéricos",
            sticky=True
        )
    """
    # Clases para sticky positioning
    sticky_classes = "sticky-top" if sticky else ""

    # Estilos del contenedor
    container_style = {
        "zIndex": "1020",  # Por encima de navbar (1000)
        "backgroundColor": COLORS["bg_primary"],
        "paddingTop": SPACING["m"],
        "paddingBottom": SPACING["m"],
    }

    # Card contenedor
    filter_card = BaseCard(
        [
            # Header con icono
            html.Div([
                Title(
                    [
                        html.I(className="fas fa-filter me-2", style={"color": COLORS["primary"]}),
                        title
                    ],
                    level=5,
                    className="mb-0"
                )
            ], className="mb-3"),

            # Fila de filtros (responsive)
            dbc.Row(
                filters_row,
                className="g-3 align-items-end"  # Gap entre columnas, alinear al fondo
            ),

            # Línea divisoria opcional
            html.Hr(className="mt-3 mb-0") if show_divider else None
        ],
        shadow="md",
        className="mb-4"
    )

    # Contenedor principal con sticky
    return dbc.Row([
        dbc.Col([
            html.Div(
                filter_card,
                className=sticky_classes,
                style=container_style
            )
        ], lg=12)
    ])
