"""
Sistema de feature flags para testing A/B y activación gradual de funcionalidades.
Permite habilitar/deshabilitar componentes sin modificar código.
"""

import json
from typing import Dict, List, Literal, Optional

import dash_bootstrap_components as dbc
from dash import ALL, Input, Output, State, dcc, html

FeatureScope = Literal["global", "user", "pharmacy", "session"]
FeatureStatus = Literal["enabled", "disabled", "beta", "deprecated"]


class FeatureFlags:
    """Gestor centralizado de feature flags."""

    # Feature flags del sistema unificado de estado
    UNIFIED_STATUS_FLAGS = {
        "unified_status_panel": {
            "enabled": True,  # ✅ PRODUCTION: Componente unificado habilitado
            "scope": "global",
            "description": "Panel unificado de estado del sistema",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["admin", "developer"],
            "ab_test": False,
        },
        "basic_mode_enabled": {
            "enabled": True,
            "scope": "global",
            "description": "Modo básico para farmacéuticos",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["pharmacist", "technician"],
            "ab_test": False,
        },
        "detailed_mode_enabled": {
            "enabled": True,
            "scope": "global",
            "description": "Modo detallado para administradores",
            "version": "3.0",
            "rollout_percentage": 90,
            "target_users": ["admin", "manager"],
            "ab_test": True,
            "ab_variant": "detailed_v2",
        },
        "diagnostic_mode_enabled": {
            "enabled": True,
            "scope": "global",
            "description": "Modo diagnóstico para desarrolladores",
            "version": "3.0",
            "rollout_percentage": 75,
            "target_users": ["developer", "support"],
            "ab_test": False,
        },
        "legacy_fallback_enabled": {
            "enabled": True,
            "scope": "global",
            "description": "Fallback al sistema legacy en caso de error",
            "version": "2.0",
            "rollout_percentage": 100,
            "target_users": ["all"],
            "ab_test": False,
        },
        "real_time_updates": {
            "enabled": True,
            "scope": "user",
            "description": "Actualizaciones automáticas en tiempo real",
            "version": "3.0",
            "rollout_percentage": 85,
            "target_users": ["all"],
            "ab_test": True,
            "ab_variant": "realtime_optimized",
        },
    }

    # Feature flags de herramientas de mantenimiento
    MAINTENANCE_FLAGS = {
        "maintenance_tools_enabled": {
            "enabled": True,
            "scope": "global",
            "description": "Herramientas de mantenimiento seguras",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["admin", "support"],
            "ab_test": False,
        },
        "quick_actions_toolbar": {
            "enabled": True,
            "scope": "user",
            "description": "Barra de herramientas de acciones rápidas",
            "version": "3.0",
            "rollout_percentage": 80,
            "target_users": ["admin"],
            "ab_test": True,
            "ab_variant": "toolbar_compact",
        },
        "connectivity_testing": {
            "enabled": True,
            "scope": "global",
            "description": "Pruebas de conectividad automáticas",
            "version": "3.0",
            "rollout_percentage": 95,
            "target_users": ["all"],
            "ab_test": False,
        },
        "cache_management": {
            "enabled": True,
            "scope": "global",
            "description": "Gestión avanzada de cache",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["admin"],
            "ab_test": False,
        },
    }

    # Feature flags de zona de peligro
    DANGER_ZONE_FLAGS = {
        "danger_zone_enabled": {
            "enabled": True,
            "scope": "global",
            "description": "Zona de peligro para operaciones críticas",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["admin"],
            "ab_test": False,
        },
        "dangerous_operations": {
            "enabled": True,
            "scope": "global",
            "description": "Operaciones peligrosas habilitadas",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["admin"],
            "ab_test": False,
        },
        "text_confirmation_required": {
            "enabled": True,
            "scope": "global",
            "description": "Confirmación de texto obligatoria",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["all"],
            "ab_test": False,
        },
        "multi_step_confirmation": {
            "enabled": True,
            "scope": "global",
            "description": "Confirmación en múltiples pasos",
            "version": "3.0",
            "rollout_percentage": 90,
            "target_users": ["all"],
            "ab_test": True,
            "ab_variant": "enhanced_protection",
        },
        "operation_audit_log": {
            "enabled": True,
            "scope": "global",
            "description": "Log de auditoría para operaciones",
            "version": "3.0",
            "rollout_percentage": 100,
            "target_users": ["all"],
            "ab_test": False,
        },
    }

    # Feature flags experimentales
    EXPERIMENTAL_FLAGS = {
        "ai_suggestions": {
            "enabled": False,
            "scope": "user",
            "description": "Sugerencias de IA para operaciones",
            "version": "4.0-beta",
            "rollout_percentage": 25,
            "target_users": ["developer", "beta_tester"],
            "ab_test": True,
            "ab_variant": "ai_assistant_v1",
        },
        "advanced_monitoring": {
            "enabled": False,
            "scope": "global",
            "description": "Monitoreo avanzado con métricas ML",
            "version": "4.0-beta",
            "rollout_percentage": 40,
            "target_users": ["admin", "developer"],
            "ab_test": True,
            "ab_variant": "monitoring_v2",
        },
        "predictive_maintenance": {
            "enabled": False,
            "scope": "global",
            "description": "Mantenimiento predictivo automatizado",
            "version": "4.0-alpha",
            "rollout_percentage": 10,
            "target_users": ["developer"],
            "ab_test": False,
        },
    }

    # Consolidación de todos los flags
    @classmethod
    def get_all_flags(cls) -> Dict[str, Dict]:
        """Retorna todos los feature flags consolidados."""
        all_flags = {}
        all_flags.update(cls.UNIFIED_STATUS_FLAGS)
        all_flags.update(cls.MAINTENANCE_FLAGS)
        all_flags.update(cls.DANGER_ZONE_FLAGS)
        all_flags.update(cls.EXPERIMENTAL_FLAGS)
        return all_flags

    @classmethod
    def is_feature_enabled(cls, feature_name: str, user_type: str = "admin") -> bool:
        """
        Verifica si una feature está habilitada para el usuario.

        Args:
            feature_name: Nombre del feature flag
            user_type: Tipo de usuario (admin, pharmacist, developer, etc.)
        """
        all_flags = cls.get_all_flags()
        feature = all_flags.get(feature_name)

        if not feature:
            return False

        # Verificar si está habilitado globalmente
        if not feature.get("enabled", False):
            return False

        # Verificar tipo de usuario
        target_users = feature.get("target_users", ["all"])
        if "all" not in target_users and user_type not in target_users:
            return False

        # Verificar rollout percentage (simulado)
        rollout = feature.get("rollout_percentage", 100)
        if rollout < 100:
            # En producción, esto debería usar hash del usuario para consistencia
            import random

            return random.randint(1, 100) <= rollout

        return True


def create_feature_flags_store() -> dcc.Store:
    """Crea store global para todos los feature flags."""
    return dcc.Store(id="global-feature-flags-store", data=FeatureFlags.get_all_flags(), storage_type="session")


def create_feature_flags_panel(panel_id: str = "feature-flags-panel") -> html.Div:
    """
    Crea panel de administración de feature flags.
    Solo para desarrollo y administradores.

    Args:
        panel_id: ID único del panel
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H6(
                        [html.I(className="fas fa-flag me-2 text-primary"), "Feature Flags - Panel de Control"],
                        className="mb-0",
                    ),
                    dbc.Badge("SOLO DESARROLLO", color="warning", className="ms-auto"),
                ]
            ),
            dbc.CardBody(
                [
                    dbc.Alert(
                        html.Div([
                            html.I(className="fas fa-info-circle me-2"),
                            "Este panel permite habilitar/deshabilitar funcionalidades en tiempo real. "
                            "Los cambios afectan solo a la sesión actual.",
                        ]),
                        color="info",
                        className="mb-3",
                    ),
                    # Store para cambios locales
                    dcc.Store(id=f"{panel_id}-local-flags", data={}),
                    # Tabs por categoría
                    dbc.Tabs(
                        [
                            dbc.Tab(
                                create_flags_category_tab(
                                    "unified_status", "Estado Unificado", FeatureFlags.UNIFIED_STATUS_FLAGS
                                ),
                                label="Estado Sistema",
                                tab_id="tab-unified",
                            ),
                            dbc.Tab(
                                create_flags_category_tab(
                                    "maintenance", "Mantenimiento", FeatureFlags.MAINTENANCE_FLAGS
                                ),
                                label="Mantenimiento",
                                tab_id="tab-maintenance",
                            ),
                            dbc.Tab(
                                create_flags_category_tab(
                                    "danger_zone", "Zona Peligro", FeatureFlags.DANGER_ZONE_FLAGS
                                ),
                                label="Zona Peligro",
                                tab_id="tab-danger",
                            ),
                            dbc.Tab(
                                create_flags_category_tab(
                                    "experimental", "Experimental", FeatureFlags.EXPERIMENTAL_FLAGS
                                ),
                                label="Experimental",
                                tab_id="tab-experimental",
                            ),
                        ],
                        id=f"{panel_id}-tabs",
                        active_tab="tab-unified",
                    ),
                    # Acciones globales
                    html.Hr(),
                    html.Div(
                        [
                            dbc.ButtonGroup(
                                [
                                    dbc.Button(
                                        html.Span([html.I(className="fas fa-sync-alt me-2"), "Recargar"]),
                                        id=f"{panel_id}-reload",
                                        color="secondary",
                                        outline=True,
                                        size="sm",
                                    ),
                                    dbc.Button(
                                        html.Span([html.I(className="fas fa-undo me-2"), "Resetear"]),
                                        id=f"{panel_id}-reset",
                                        color="warning",
                                        outline=True,
                                        size="sm",
                                    ),
                                    dbc.Button(
                                        html.Span([html.I(className="fas fa-download me-2"), "Exportar"]),
                                        id=f"{panel_id}-export",
                                        color="info",
                                        outline=True,
                                        size="sm",
                                    ),
                                ]
                            )
                        ],
                        className="text-end",
                    ),
                ]
            ),
        ],
        className="shadow-sm",
    )


def create_flags_category_tab(category: str, title: str, flags: Dict) -> html.Div:
    """Crea tab para una categoría específica de flags."""
    flag_controls = []

    for flag_name, flag_config in flags.items():
        # Card para cada flag
        flag_card = dbc.Card(
            [
                dbc.CardBody(
                    [
                        # Header con switch
                        html.Div(
                            [
                                html.Div(
                                    [
                                        html.H6(flag_name.replace("_", " ").title(), className="mb-1"),
                                        html.Small(flag_config.get("description", ""), className="text-muted"),
                                    ],
                                    className="flex-grow-1",
                                ),
                                dbc.Switch(
                                    id={"type": "flag-switch", "flag": flag_name},
                                    value=flag_config.get("enabled", False),
                                    className="ms-3",
                                ),
                            ],
                            className="d-flex align-items-start",
                        ),
                        # Detalles del flag
                        html.Hr(),
                        dbc.Row(
                            [
                                dbc.Col(
                                    [
                                        html.Small("Versión:", className="text-muted"),
                                        html.Br(),
                                        html.Strong(flag_config.get("version", "N/A")),
                                    ],
                                    width=3,
                                ),
                                dbc.Col(
                                    [
                                        html.Small("Rollout:", className="text-muted"),
                                        html.Br(),
                                        html.Strong(f"{flag_config.get('rollout_percentage', 0)}%"),
                                    ],
                                    width=3,
                                ),
                                dbc.Col(
                                    [
                                        html.Small("Usuarios:", className="text-muted"),
                                        html.Br(),
                                        html.Strong(", ".join(flag_config.get("target_users", []))),
                                    ],
                                    width=6,
                                ),
                            ]
                        ),
                        # A/B Test info si aplica
                        html.Div(
                            (
                                [
                                    html.Hr(),
                                    dbc.Badge(
                                        [
                                            html.I(className="fas fa-flask me-1"),
                                            f"A/B Test: {flag_config.get('ab_variant', 'N/A')}",
                                        ],
                                        color="info",
                                    ),
                                ]
                                if flag_config.get("ab_test")
                                else []
                            ),
                            className="mt-2",
                        ),
                    ]
                )
            ],
            className="mb-3",
        )

        flag_controls.append(flag_card)

    return html.Div([html.H6(f"Feature Flags - {title}", className="mb-3"), html.Div(flag_controls)])


def create_ab_test_indicator(flag_name: str, variant: str) -> html.Div:
    """Crea indicador visual para tests A/B activos."""
    return dbc.Alert(
        [html.I(className="fas fa-flask me-2"), html.Strong("A/B Test Activo: "), f"{flag_name} - Variante: {variant}"],
        color="info",
        className="small mb-2",
        dismissable=True,
    )


def create_feature_status_badge(flag_config: Dict) -> dbc.Badge:
    """Crea badge de estado para un feature flag."""
    enabled = flag_config.get("enabled", False)
    rollout = flag_config.get("rollout_percentage", 100)

    if not enabled:
        return dbc.Badge("Deshabilitado", color="secondary")
    elif rollout < 100:
        return dbc.Badge(f"Beta {rollout}%", color="warning")
    elif flag_config.get("ab_test"):
        return dbc.Badge("A/B Test", color="info")
    else:
        return dbc.Badge("Activo", color="success")


# Callbacks para gestión de feature flags
def register_feature_flags_callbacks(app):
    """Registra callbacks para el sistema de feature flags."""

    @app.callback(
        Output("global-feature-flags-store", "data"),
        Input({"type": "flag-switch", "flag": ALL}, "value"),
        State("global-feature-flags-store", "data"),
        prevent_initial_call=True,
    )
    def update_feature_flags(switch_values, current_flags):
        """Actualiza feature flags cuando se cambia un switch."""
        if not switch_values:
            return current_flags

        # Mapear cambios a los flags correspondientes
        all_flag_names = list(FeatureFlags.get_all_flags().keys())
        updated_flags = current_flags.copy()

        for i, value in enumerate(switch_values):
            if i < len(all_flag_names):
                flag_name = all_flag_names[i]
                if flag_name in updated_flags:
                    updated_flags[flag_name]["enabled"] = value

        return updated_flags

    @app.callback(
        Output("toast-trigger-store", "data", allow_duplicate=True),
        Input({"type": "flag-switch", "flag": ALL}, "value"),
        prevent_initial_call=True,
    )
    def notify_flag_changes(switch_values):
        """Notifica cambios en feature flags."""
        if not switch_values:
            return {}

        from .toast_manager import info_toast

        return info_toast(
            "Feature flags actualizados. Los cambios se aplicarán al recargar componentes.", "Configuración Actualizada"
        )


# Funciones helper para uso en otros componentes
def is_feature_enabled(feature_name: str, feature_flags: Dict = None) -> bool:
    """
    Verifica si un feature está habilitado.

    Args:
        feature_name: Nombre del feature flag
        feature_flags: Diccionario de flags (opcional, usa default si no se proporciona)
    """
    if feature_flags is None:
        feature_flags = FeatureFlags.get_all_flags()

    flag_config = feature_flags.get(feature_name, {})
    return flag_config.get("enabled", False)


def get_ab_test_variant(feature_name: str, feature_flags: Dict = None) -> Optional[str]:
    """
    Obtiene la variante de A/B test para un feature.

    Args:
        feature_name: Nombre del feature flag
        feature_flags: Diccionario de flags (opcional)
    """
    if feature_flags is None:
        feature_flags = FeatureFlags.get_all_flags()

    flag_config = feature_flags.get(feature_name, {})
    if flag_config.get("ab_test", False):
        return flag_config.get("ab_variant")

    return None


def create_feature_dependent_component(feature_name: str, component, fallback_component=None):
    """
    Crea un componente que depende de un feature flag.

    Args:
        feature_name: Nombre del feature flag
        component: Componente a mostrar si el feature está habilitado
        fallback_component: Componente alternativo si está deshabilitado
    """
    return html.Div(
        [
            dcc.Store(id=f"feature-check-{feature_name}", data={"feature": feature_name}),
            html.Div(id=f"feature-dependent-{feature_name}"),
        ]
    )


# Registro automático de callback para componentes dependientes de features
def register_feature_dependent_callback(app, feature_name: str, component_func, fallback_func=None):
    """
    Registra callback para mostrar/ocultar componente según feature flag.

    Args:
        app: Aplicación Dash
        feature_name: Nombre del feature flag
        component_func: Función que retorna el componente principal
        fallback_func: Función que retorna el componente de fallback
    """

    @app.callback(
        Output(f"feature-dependent-{feature_name}", "children"),
        [Input("global-feature-flags-store", "data"), Input(f"feature-check-{feature_name}", "data")],
        prevent_initial_call=False,
    )
    def toggle_feature_component(feature_flags, check_data):
        """Toggle componente basado en feature flag."""
        if is_feature_enabled(feature_name, feature_flags):
            return component_func() if callable(component_func) else component_func
        else:
            if fallback_func:
                return fallback_func() if callable(fallback_func) else fallback_func
            return html.Div()  # Componente vacío por defecto


# Utilidades de desarrollo
def get_active_ab_tests() -> List[Dict]:
    """Retorna lista de tests A/B activos."""
    all_flags = FeatureFlags.get_all_flags()
    active_tests = []

    for flag_name, flag_config in all_flags.items():
        if flag_config.get("ab_test", False) and flag_config.get("enabled", False):
            active_tests.append(
                {
                    "flag_name": flag_name,
                    "variant": flag_config.get("ab_variant"),
                    "rollout": flag_config.get("rollout_percentage", 100),
                    "target_users": flag_config.get("target_users", []),
                }
            )

    return active_tests


def export_feature_flags_config() -> str:
    """Exporta configuración actual de feature flags como JSON."""
    return json.dumps(FeatureFlags.get_all_flags(), indent=2, ensure_ascii=False)
