# frontend/components/erp_status_banner.py
"""
ERPStatusBanner Component - Banner contextual de estado ERP (Pivot 2026)

Muestra el estado de conexion con el ERP de forma no intrusiva:
- Oculto cuando todo OK (<30 min desde ultimo sync)
- Verde cuando sync 30-60 min
- Ambar cuando sync 60-1440 min (>1h, <24h)
- Rojo cuando desconectado o sync >1440 min (>24h)

Usage:
    from components.erp_status_banner import create_erp_status_banner

    # En layout
    html.Div(id="erp-status-banner-container")

    # En callback
    @app.callback(
        Output('erp-status-banner-container', 'children'),
        Input('global-alerts-store', 'data')
    )
    def render_erp_banner(alerts_data):
        system = alerts_data.get('system', {})
        erp = system.get('erp', {})
        return create_erp_status_banner(erp)

See: docs/issues/ISSUE-601-ERP-STATUS-BANNER.md
"""

from typing import Any, Dict, Optional

import dash_bootstrap_components as dbc
from dash import html
from dash_iconify import DashIconify

from stores.alerts_store import (
    ERP_SYNC_OK_MINUTES,
    ERP_SYNC_WARNING_MINUTES,
    ERP_SYNC_CRITICAL_MINUTES,
)
from styles.design_tokens import COLORS, SPACING, BORDER_RADIUS, SHADOWS


# =============================================================================
# Status Configuration
# =============================================================================

STATUS_CONFIG = {
    "ok": {
        "bg_color": COLORS["success"],
        "bg_light": "#d4edda",  # Bootstrap success-light
        "text_color": COLORS["success_dark"],
        "border_color": COLORS["success"],
        "icon": "mdi:check-circle",
        "show_retry": False,
        "pulse": False,
    },
    "warning": {
        "bg_color": COLORS["warning"],
        "bg_light": "#fff3cd",  # Bootstrap warning-light
        "text_color": COLORS["gray_800"],  # Dark text for amber bg
        "border_color": COLORS["warning_dark"],
        "icon": "mdi:alert",
        "show_retry": False,
        "pulse": False,
    },
    "critical": {
        "bg_color": COLORS["danger"],
        "bg_light": "#f8d7da",  # Bootstrap danger-light
        "text_color": COLORS["danger_dark"],
        "border_color": COLORS["danger"],
        "icon": "mdi:alert-circle",
        "show_retry": True,
        "pulse": True,
    },
}


# =============================================================================
# CSS Styles
# =============================================================================

def get_banner_style(status: str) -> Dict[str, Any]:
    """Genera estilos del banner segun el estado."""
    config = STATUS_CONFIG.get(status, STATUS_CONFIG["warning"])

    base_style = {
        "backgroundColor": config["bg_light"],
        "borderLeft": f"4px solid {config['border_color']}",
        "borderRadius": BORDER_RADIUS["md"],
        "padding": f"{SPACING['s']} {SPACING['m']}",
        "marginBottom": SPACING["m"],
        "display": "flex",
        "alignItems": "center",
        "justifyContent": "space-between",
        "boxShadow": SHADOWS["sm"],
    }

    # Agregar animacion pulse para estado critico
    if config["pulse"]:
        base_style["animation"] = "erp-banner-pulse 2s ease-in-out infinite"

    return base_style


def get_icon_style(status: str) -> Dict[str, Any]:
    """Genera estilos del icono segun el estado."""
    config = STATUS_CONFIG.get(status, STATUS_CONFIG["warning"])

    return {
        "color": config["border_color"],
        "marginRight": SPACING["s"],
        "flexShrink": 0,
    }


def get_text_style(status: str) -> Dict[str, Any]:
    """Genera estilos del texto segun el estado."""
    config = STATUS_CONFIG.get(status, STATUS_CONFIG["warning"])

    return {
        "color": config["text_color"],
        "margin": 0,
        "fontSize": "0.9rem",
        "fontWeight": "500",
    }


# =============================================================================
# Component Functions
# =============================================================================

def create_erp_status_banner(
    erp_status: Optional[Dict[str, Any]] = None,
) -> Optional[html.Div]:
    """
    Crea el banner de estado ERP si es necesario mostrarlo.

    El banner se oculta cuando:
    - No hay datos de ERP
    - El estado es "ok" Y el sync es menor a 30 minutos

    El banner se muestra cuando:
    - El sync es >= 30 minutos (verde)
    - El sync es >= 60 minutos (ambar)
    - El sync es >= 1440 minutos o desconectado (rojo)

    Args:
        erp_status: Dict con datos del ERP desde global-alerts-store.system.erp
            {
                "status": "ok" | "warning" | "critical",
                "provider": "farmanager",
                "last_sync": "2026-01-10T15:18:00Z",
                "minutes_since_sync": 12,
                "msg": "Conectado - Sync hace 12 min"
            }

    Returns:
        html.Div con el banner o None si no es necesario mostrarlo
    """
    # Si no hay datos, no mostrar banner
    if not erp_status:
        return None

    status = erp_status.get("status", "ok")
    minutes_since_sync = erp_status.get("minutes_since_sync", 0)
    msg = erp_status.get("msg", "")
    provider = erp_status.get("provider", "")

    # Si todo OK y sync reciente (<30 min), no mostrar banner
    if status == "ok" and minutes_since_sync is not None and minutes_since_sync < ERP_SYNC_OK_MINUTES:
        return None

    # Determinar configuracion visual
    config = STATUS_CONFIG.get(status, STATUS_CONFIG["warning"])

    # Construir contenido del banner
    content_left = html.Div(
        [
            DashIconify(
                icon=config["icon"],
                width=24,
                height=24,
                style=get_icon_style(status),
            ),
            html.Div(
                [
                    html.Span(
                        _get_title_text(status, provider),
                        style={
                            "fontWeight": "600",
                            "marginRight": SPACING["s"],
                        },
                    ),
                    html.Span(
                        msg,
                        style=get_text_style(status),
                    ),
                ],
                style={"display": "flex", "alignItems": "center", "flexWrap": "wrap"},
            ),
        ],
        style={"display": "flex", "alignItems": "center", "flex": 1},
    )

    # Boton de retry solo para estado critico
    content_right = None
    if config["show_retry"]:
        content_right = dbc.Button(
            [
                DashIconify(icon="mdi:refresh", width=16, className="me-1"),
                "Reintentar",
            ],
            id="btn-erp-retry-sync",
            color="danger",
            size="sm",
            outline=True,
            style={"whiteSpace": "nowrap"},
        )

    return html.Div(
        [
            content_left,
            content_right,
        ],
        style=get_banner_style(status),
        className="erp-status-banner",
    )


def _get_title_text(status: str, provider: str) -> str:
    """Genera el texto del titulo segun el estado."""
    provider_name = provider.capitalize() if provider else "ERP"

    if status == "critical":
        return f"{provider_name}:"
    elif status == "warning":
        return f"{provider_name}:"
    else:
        return f"{provider_name}:"


# =============================================================================
# CSS Keyframes (para inyectar en assets)
# =============================================================================

ERP_BANNER_CSS = """
/* ERPStatusBanner Animations */
@keyframes erp-banner-pulse {
    0% {
        box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    }
    50% {
        box-shadow: 0 0.125rem 0.5rem rgba(220, 53, 69, 0.3);
    }
    100% {
        box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    }
}

.erp-status-banner {
    transition: all 0.3s ease-in-out;
}

.erp-status-banner:hover {
    transform: translateY(-1px);
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .erp-status-banner {
        flex-direction: column;
        align-items: flex-start !important;
        gap: 8px;
    }

    .erp-status-banner button {
        width: 100%;
    }
}
"""


# =============================================================================
# Exports
# =============================================================================

__all__ = [
    "create_erp_status_banner",
    "ERP_BANNER_CSS",
    "STATUS_CONFIG",
]
