from utils.helpers import format_number

# frontend/components/enrichment_progress.py
"""
Componente para mostrar el progreso del enriquecimiento de datos.
Incluye métricas en tiempo real y botones de acción.
"""

import logging
from typing import Dict

import dash_bootstrap_components as dbc
from components.toast_manager import error_toast, success_toast
from dash import Input, Output, html
from utils.api_client import backend_client
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


def create_enrichment_progress_card(pharmacy_id: str = None) -> html.Div:
    """
    Crear tarjeta de progreso de enriquecimiento con métricas en tiempo real.

    Args:
        pharmacy_id: ID de la farmacia (si None, usa la farmacia activa)

    Returns:
        Componente Dash con progreso de enriquecimiento
    """

    # Usar farmacia activa si no se especifica
    if pharmacy_id is None:
        pharmacy_id = get_current_pharmacy_id()

    return html.Div(
        [
            dbc.Card(
                [
                    dbc.CardHeader(
                        [
                            html.H5(
                                [
                                    html.I(className="fas fa-chart-line me-2 text-primary"),
                                    "Progreso de Enriquecimiento",
                                ],
                                className="mb-0 text-primary",
                            )
                        ]
                    ),
                    dbc.CardBody(
                        [
                            # Área de contenido dinámico
                            html.Div(
                                id="enrichment-progress-content",
                                children=[
                                    dbc.Placeholder(
                                        [
                                            dbc.Placeholder(xs=6, className="mb-2"),
                                            dbc.Placeholder(xs=8, className="mb-2"),
                                            dbc.Placeholder(xs=4),
                                        ],
                                        animation="glow",
                                    )
                                ],
                            ),
                            html.Hr(),
                            # Botones de acción
                            html.Div(
                                [
                                    dbc.Button(
                                        html.Span([html.I(className="fas fa-sync-alt me-2"), "Actualizar"]),
                                        id="refresh-enrichment-btn",
                                        color="outline-primary",
                                        size="sm",
                                        className="me-2",
                                    ),
                                    dbc.Button(
                                        html.Span([html.I(className="fas fa-play me-2"), "Enriquecer Ahora"]),
                                        id="trigger-enrichment-btn",
                                        color="primary",
                                        size="sm",
                                    ),
                                ],
                                className="d-flex justify-content-start",
                            ),
                        ]
                    ),
                ],
                className="shadow-sm",
            )
        ],
        id=f"enrichment-progress-{pharmacy_id}",
    )


def create_enrichment_metrics(progress_data: Dict) -> html.Div:
    """
    Crear métricas visuales del progreso de enriquecimiento.

    Args:
        progress_data: Datos de progreso desde la API

    Returns:
        Componente con métricas visuales
    """

    total_sales = progress_data.get("total_sales", 0)
    enriched = progress_data.get("enriched", 0)
    pending = progress_data.get("pending", 0)
    failed = progress_data.get("failed", 0)
    enrichment_rate = progress_data.get("enrichment_rate", 0)
    needs_processing = progress_data.get("needs_processing", False)

    # Determinar color del progreso
    if enrichment_rate >= 90:
        progress_color = "success"
        status_color = "success"
        status_text = "Excelente"
    elif enrichment_rate >= 70:
        progress_color = "info"
        status_color = "info"
        status_text = "Bueno"
    elif enrichment_rate >= 50:
        progress_color = "warning"
        status_color = "warning"
        status_text = "Regular"
    else:
        progress_color = "danger"
        status_color = "danger"
        status_text = "Necesita atención"

    return html.Div(
        [
            # Barra de progreso principal
            html.Div(
                [
                    html.Label("Tasa de Enriquecimiento", className="form-label small text-muted mb-1"),
                    dbc.Progress(
                        value=enrichment_rate,
                        color=progress_color,
                        striped=needs_processing,
                        animated=needs_processing,
                        className="mb-2",
                        style={"height": "8px"},
                    ),
                    html.Div(
                        [
                            html.Span(
                                f"{enrichment_rate:.1f}%".replace(".", ","), className=f"fw-bold text-{status_color}"
                            ),
                            html.Span(f" • {status_text}", className="text-muted small ms-2"),
                        ]
                    ),
                ],
                className="mb-3",
            ),
            # Métricas detalladas en filas
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-database fa-sm text-muted me-2"),
                                    html.Span("Total Registros", className="text-muted small"),
                                ],
                                className="mb-1",
                            ),
                            html.Div(format_number(total_sales), className="h6 mb-0 text-dark"),
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-check-circle fa-sm text-success me-2"),
                                    html.Span("Enriquecidos", className="text-muted small"),
                                ],
                                className="mb-1",
                            ),
                            html.Div(format_number(enriched), className="h6 mb-0 text-success"),
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-clock fa-sm text-warning me-2"),
                                    html.Span("Pendientes", className="text-muted small"),
                                ],
                                className="mb-1",
                            ),
                            html.Div(format_number(pending), className="h6 mb-0 text-warning"),
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-exclamation-triangle fa-sm text-danger me-2"),
                                    html.Span("Fallidos", className="text-muted small"),
                                ],
                                className="mb-1",
                            ),
                            html.Div(format_number(failed), className="h6 mb-0 text-danger"),
                        ],
                        width=3,
                    ),
                ]
            ),
            # Alerta si hay registros pendientes
            (
                html.Div(
                    [
                        dbc.Alert(
                            [
                                html.I(className="fas fa-info-circle me-2"),
                                format_number(pending),
                                html.Strong("Haz clic en 'Enriquecer Ahora' para procesarlos."),
                            ],
                            color="info",
                            className="small mt-3 mb-0",
                        )
                    ]
                )
                if needs_processing
                else html.Div()
            ),
        ]
    )


def create_enrichment_error(error_message: str) -> html.Div:
    """
    Crear componente de error para el enriquecimiento.

    Args:
        error_message: Mensaje de error

    Returns:
        Componente de error
    """

    return dbc.Alert(
        [
            html.I(className="fas fa-exclamation-triangle me-2"),
            html.Strong("Error obteniendo progreso de enriquecimiento: "),
            html.Span(error_message, className="small"),
        ],
        color="warning",
        className="mb-0",
    )


# Callbacks para el componente de progreso de enriquecimiento
def register_enrichment_progress_callbacks(app):
    """
    Registrar callbacks para el progreso de enriquecimiento.

    Args:
        app: Instancia de la aplicación Dash
    """

    @app.callback(
        Output("enrichment-progress-content", "children"),
        [Input("refresh-enrichment-btn", "n_clicks"), Input("dashboard-update-interval", "n_intervals")],
        prevent_initial_call=False,
    )
    def update_enrichment_progress(refresh_clicks, interval_updates):
        """
        Actualizar progreso de enriquecimiento desde la API
        """
        try:
            pharmacy_id = get_current_pharmacy_id()

            # Obtener datos de progreso de enriquecimiento
            response = backend_client.get_enrichment_progress(pharmacy_id)

            if response.success and response.data:
                return create_enrichment_metrics(response.data)
            else:
                logger.warning(f"Error obteniendo progreso de enriquecimiento: {response.message}")
                return create_enrichment_error(response.message or "Error al cargar datos")

        except Exception as e:
            logger.error(f"Error actualizando progreso de enriquecimiento: {str(e)}")
            return create_enrichment_error(str(e))

    @app.callback(
        Output("toast-trigger-store", "data", allow_duplicate=True),
        [Input("trigger-enrichment-btn", "n_clicks")],
        prevent_initial_call=True,
    )
    def trigger_enrichment_process(n_clicks):
        """
        Disparar proceso de enriquecimiento manual.

        Args:
            n_clicks: Clicks en botón de enriquecimiento

        Returns:
            Toast para notificar el resultado
        """

        if not n_clicks:
            return {}

        try:
            logger.info("Disparando re-enriquecimiento manual")

            response = backend_client.execute_reenrichment(get_current_pharmacy_id())

            if response.success:
                return success_toast(
                    "El proceso se ejecutará en segundo plano. Los resultados se actualizarán automáticamente.",
                    title="Re-enriquecimiento Iniciado",
                )
            else:
                logger.error(f"Error disparando re-enriquecimiento: {response.message}")
                return error_toast(
                    response.message or "Error desconocido al iniciar el proceso", title="Error de Enriquecimiento"
                )

        except Exception as e:
            logger.error(f"Error disparando enriquecimiento: {str(e)}")
            return error_toast(f"Error del sistema: {str(e)}", title="Error Crítico")
