# frontend/components/enrichment_kpis.py
"""
Componente especializado para KPIs de enriquecimiento de datos
Muestra métricas clave del proceso de enriquecimiento con datos CIMA/nomenclator
"""

import logging
from typing import Any, Dict, Optional

import dash_bootstrap_components as dbc
import plotly.express as px
from dash import dcc, html
from utils.helpers import format_currency, format_number

logger = logging.getLogger(__name__)


def create_enrichment_kpis(enriched_data: Optional[Dict[str, Any]] = None) -> dbc.Row:
    """
    Crear tarjetas KPI para métricas de enriquecimiento

    Args:
        enriched_data: Datos del endpoint /api/sales/enriched/summary/

    Returns:
        dbc.Row con KPIs de enriquecimiento
    """

    # Datos por defecto si no hay respuesta del backend
    if not enriched_data or not enriched_data.get("metrics"):
        return create_loading_kpis()

    metrics = enriched_data["metrics"]

    # KPI 1: Tasa de Enriquecimiento
    enrichment_rate = metrics.get("enrichment_rate", 0)
    enrichment_color = get_enrichment_color(enrichment_rate)

    kpi_enrichment = dbc.Card(
        [
            dbc.CardBody(
                [
                    html.H4(
                        [html.I(className="fas fa-database me-2"), "Datos Enriquecidos"],
                        className="card-title text-primary mb-2",
                    ),
                    html.H2(f"{enrichment_rate:.2f}%".replace(".", ","), className=f"text-{enrichment_color} mb-1"),
                    html.P(
                        [
                            html.Small(
                                f"{format_number(metrics.get('enriched_sales', 0))} de {format_number(metrics.get('total_sales', 0))} ventas",
                                className="text-muted",
                            )
                        ],
                        className="mb-2",
                    ),
                    html.Div([html.Small("🎯 Objetivo: >80%", className="text-muted")]),
                ]
            )
        ],
        className="h-100 shadow-sm",
    )

    # KPI 2: Análisis de Genéricos
    generic_data = metrics.get("generic_sales", {})
    generic_percentage = generic_data.get("percentage", 0)
    generic_amount = generic_data.get("amount", 0)

    kpi_generics = dbc.Card(
        [
            dbc.CardBody(
                [
                    html.H4(
                        [html.I(className="fas fa-pills me-2"), "Ventas Genéricos"],
                        className="card-title text-success mb-2",
                    ),
                    html.H2(f"{generic_percentage:.1f}%".replace(".", ","), className="text-success mb-1"),
                    html.P(
                        [
                            html.Small(
                                f"{format_currency(generic_amount, decimals=2)} en genéricos", className="text-muted"
                            )
                        ],
                        className="mb-2",
                    ),
                    html.Div([html.Small("💡 Oportunidades disponibles", className="text-success")]),
                ]
            )
        ],
        className="h-100 shadow-sm",
    )

    # KPI 3: Total Enriquecido
    total_amount = metrics.get("total_amount", 0)
    total_sales = metrics.get("total_sales", 0)

    kpi_total = dbc.Card(
        [
            dbc.CardBody(
                [
                    html.H4(
                        [html.I(className="fas fa-chart-line me-2"), "Volumen Total"],
                        className="card-title text-info mb-2",
                    ),
                    html.H2(f"{format_currency(total_amount, decimals=0)}", className="text-info mb-1"),
                    html.P(
                        [html.Small(f"{format_number(total_sales)} transacciones", className="text-muted")],
                        className="mb-2",
                    ),
                    html.Div([html.Small("📈 Datos oficiales CIMA", className="text-muted")]),
                ]
            )
        ],
        className="h-100 shadow-sm",
    )

    # KPI 4: Top Categoría Terapéutica
    top_category = "Sin datos"
    category_amount = 0

    therapeutic_categories = enriched_data.get("therapeutic_categories", [])
    if therapeutic_categories:
        top_cat = therapeutic_categories[0]
        top_category = (
            top_cat.get("category", "Sin datos")[:20] + "..."
            if len(top_cat.get("category", "")) > 20
            else top_cat.get("category", "Sin datos")
        )
        category_amount = top_cat.get("amount", 0)

    kpi_therapy = dbc.Card(
        [
            dbc.CardBody(
                [
                    html.H4(
                        [html.I(className="fas fa-stethoscope me-2"), "Top Terapéutica"],
                        className="card-title text-warning mb-2",
                    ),
                    html.H3(top_category, className="text-warning mb-1", style={"fontSize": "1.2rem"}),
                    html.P(
                        [html.Small(f"{format_currency(category_amount, decimals=0)}", className="text-muted")],
                        className="mb-2",
                    ),
                    html.Div([html.Small("🏥 Clasificación ATC", className="text-muted")]),
                ]
            )
        ],
        className="h-100 shadow-sm",
    )

    return dbc.Row(
        [
            dbc.Col(kpi_enrichment, width=3),
            dbc.Col(kpi_generics, width=3),
            dbc.Col(kpi_total, width=3),
            dbc.Col(kpi_therapy, width=3),
        ],
        className="g-4",
    )


def create_enrichment_status_card(status_data: Optional[Dict[str, Any]] = None) -> dbc.Card:
    """
    Crear tarjeta de estado del enriquecimiento

    Args:
        status_data: Datos del endpoint /api/sales/enrichment/status/

    Returns:
        dbc.Card con estado detallado del enriquecimiento
    """

    if not status_data:
        return create_loading_status_card()

    # Datos de procesamiento
    data_processing = status_data.get("data_processing", {})
    enrichment_coverage = status_data.get("enrichment_coverage", {})

    processing_rate = data_processing.get("processing_success_rate", 0)
    enrichment_rate = enrichment_coverage.get("enrichment_rate", 0)
    avg_confidence = enrichment_coverage.get("average_confidence", 0)

    # Indicadores de salud
    processing_health = get_health_status(processing_rate)
    enrichment_health = get_health_status(enrichment_rate)
    confidence_health = get_health_status(avg_confidence)

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H5(
                        [html.I(className="fas fa-chart-pie me-2"), "Estado del Sistema de Enriquecimiento"],
                        className="mb-0",
                    )
                ]
            ),
            dbc.CardBody(
                [
                    # Métricas principales
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H6("Procesamiento de Datos", className="text-muted"),
                                    html.H4(
                                        [
                                            f"{processing_rate:.1f}%".replace(".", ","),
                                            html.Span(
                                                f" {processing_health['icon']}",
                                                className=f"text-{processing_health['color']} ms-2",
                                            ),
                                        ]
                                    ),
                                    html.Small(
                                        f"{format_number(data_processing.get('valid_records', 0))} registros válidos",
                                        className="text-muted",
                                    ),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.H6("Cobertura Enriquecimiento", className="text-muted"),
                                    html.H4(
                                        [
                                            f"{enrichment_rate:.1f}%".replace(".", ","),
                                            html.Span(
                                                f" {enrichment_health['icon']}",
                                                className=f"text-{enrichment_health['color']} ms-2",
                                            ),
                                        ]
                                    ),
                                    html.Small(
                                        f"{format_number(enrichment_coverage.get('enriched_sales', 0))} ventas enriquecidas",
                                        className="text-muted",
                                    ),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.H6("Confianza Promedio", className="text-muted"),
                                    html.H4(
                                        [
                                            f"{avg_confidence:.1f}%".replace(".", ","),
                                            html.Span(
                                                f" {confidence_health['icon']}",
                                                className=f"text-{confidence_health['color']} ms-2",
                                            ),
                                        ]
                                    ),
                                    html.Small("Precisión del matching", className="text-muted"),
                                ],
                                width=4,
                            ),
                        ],
                        className="mb-3",
                    ),
                    html.Hr(),
                    # Métodos de matching
                    html.H6("Métodos de Matching", className="text-muted mb-2"),
                    create_matching_methods_chart(status_data.get("match_methods", [])),
                    html.Hr(),
                    # Fuentes de datos
                    html.H6("Fuentes de Enriquecimiento", className="text-muted mb-2"),
                    create_enrichment_sources_badges(status_data.get("enrichment_sources", [])),
                ]
            ),
        ],
        className="shadow-sm",
    )


def create_matching_methods_chart(match_methods: list) -> html.Div:
    """Crear gráfico de métodos de matching"""
    if not match_methods:
        return html.P("No hay datos de métodos de matching", className="text-muted")

    methods = [method.get("method", "Unknown") for method in match_methods]
    percentages = [method.get("percentage", 0) for method in match_methods]

    fig = px.bar(
        x=percentages,
        y=methods,
        orientation="h",
        color=percentages,
        color_continuous_scale="Viridis",
        title="Distribución por Método",
    )

    fig.update_layout(
        height=200, margin=dict(l=10, r=10, t=30, b=10), showlegend=False, xaxis_title="Porcentaje", yaxis_title=""
    )

    return dcc.Graph(figure=fig, style={"height": "200px"})


def create_enrichment_sources_badges(sources: list) -> html.Div:
    """Crear badges de fuentes de enriquecimiento"""
    if not sources:
        return html.P("No hay fuentes configuradas", className="text-muted")

    badges = []
    for source in sources:
        source_name = source.get("source", "Unknown") or "Unknown"
        count = source.get("count", 0)

        color = "primary" if "CIMA" in source_name else "secondary"

        badges.append(
            dbc.Badge(
                [source_name, html.Span(f" ({format_number(count)})", className="ms-1")],
                color=color,
                className="me-2 mb-2",
            )
        )

    return html.Div(badges)


# Funciones auxiliares
def create_loading_kpis() -> dbc.Row:
    """KPIs de carga mientras se obtienen datos"""
    loading_card = dbc.Card(
        [
            dbc.CardBody(
                [
                    html.H4(
                        [html.I(className="fas fa-spinner fa-spin me-2"), "Cargando..."],
                        className="card-title text-muted mb-2",
                    ),
                    html.H2("--", className="text-muted mb-1"),
                    html.P("Obteniendo datos...", className="text-muted"),
                ]
            )
        ],
        className="h-100 shadow-sm",
    )

    return dbc.Row(
        [
            dbc.Col(loading_card, width=3),
            dbc.Col(loading_card, width=3),
            dbc.Col(loading_card, width=3),
            dbc.Col(loading_card, width=3),
        ],
        className="g-4",
    )


def create_loading_status_card() -> dbc.Card:
    """Tarjeta de estado de carga"""
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H5(
                        [html.I(className="fas fa-spinner fa-spin me-2"), "Cargando Estado del Sistema..."],
                        className="mb-0",
                    )
                ]
            ),
            dbc.CardBody([html.P("Obteniendo métricas de enriquecimiento...", className="text-muted")]),
        ],
        className="shadow-sm",
    )


def get_enrichment_color(rate: float) -> str:
    """Determinar color según tasa de enriquecimiento"""
    if rate >= 80:
        return "success"
    elif rate >= 60:
        return "warning"
    else:
        return "danger"


def get_health_status(rate: float) -> Dict[str, str]:
    """Determinar estado de salud según tasa"""
    if rate >= 80:
        return {"color": "success", "icon": "✅"}
    elif rate >= 60:
        return {"color": "warning", "icon": "⚠️"}
    else:
        return {"color": "danger", "icon": "❌"}
