from utils.helpers import format_number

# frontend/components/enriched_analytics.py
"""
Componentes para análisis farmacéutico enriquecido con datos CIMA/nomenclator.
Incluye visualizaciones específicas para farmacia: genéricos, prescripciones, ATC.
"""

from typing import Any, Dict

import dash_bootstrap_components as dbc
import plotly.graph_objs as go
from dash import dcc, html


def create_enrichment_overview_card(data: Dict[str, Any]) -> dbc.Card:
    """
    Tarjeta de resumen del proceso de enriquecimiento

    Args:
        data: Datos del endpoint /enrichment/status/
    """
    if not data:
        return dbc.Card([
            dbc.CardBody([
                html.H5("⚠️ Error de Enriquecimiento", className="card-title text-warning"),
                html.P("No se pudieron cargar los datos de enriquecimiento", className="text-muted")
            ])
        ], color="light")

    enrichment_data = data.get("enrichment_coverage", {})
    total_sales = enrichment_data.get("total_sales", 0)
    enriched_sales = enrichment_data.get("enriched_sales", 0)
    enrichment_rate = enrichment_data.get("enrichment_rate", 0)
    avg_confidence = enrichment_data.get("average_confidence", 0)

    # Color según la tasa de enriquecimiento
    if enrichment_rate >= 80:
        color_class = "text-success"
        icon = "fas fa-check-circle"
    elif enrichment_rate >= 50:
        color_class = "text-warning"
        icon = "fas fa-exclamation-triangle"
    else:
        color_class = "text-danger"
        icon = "fas fa-times-circle"

    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-database me-2"),
            "Estado de Enriquecimiento CIMA"
        ]),
        dbc.CardBody([
            dbc.Row([
                dbc.Col([
                    html.H3([
                        html.I(className=f"{icon} me-2 {color_class}"),
                        f"{enrichment_rate:.1f}%".replace(".", ",")
                    ], className=f"mb-1 {color_class}"),
                    html.P("Tasa de Enriquecimiento", className="text-muted mb-0")
                ], width=6),
                dbc.Col([
                    html.H4(format_number(enriched_sales), className="mb-1 text-info"),
                    html.P(format_number(total_sales), className="text-muted mb-0")
                ], width=6)
            ], className="mb-3"),
            dbc.Row([
                dbc.Col([
                    html.Strong("Confianza Promedio: "),
                    html.Span(f"{avg_confidence:.1f}%".replace(".", ","), className="text-primary")
                ], width=12)
            ])
        ])
    ], color="light", outline=True)

def create_therapeutic_categories_chart(data: Dict[str, Any]) -> dcc.Graph:
    """
    Gráfico de categorías terapéuticas (códigos ATC)

    Args:
        data: Datos del endpoint /enriched/therapeutic-analysis/
    """
    if not data or not data.get("atc_groups"):
        return dcc.Graph(
            figure={
                'data': [],
                'layout': {
                    'title': 'Categorías Terapéuticas ATC',
                    'annotations': [{
                        'text': 'No hay datos de categorías terapéuticas disponibles',
                        'showarrow': False,
                        'font': {'size': 16, 'color': 'gray'}
                    }]
                }
            },
            style={'height': '400px'}
        )

    atc_groups = data.get("atc_groups", [])

    # Preparar datos para el gráfico
    categories = []
    amounts = []
    quantities = []
    atc_codes = []

    for group in atc_groups:
        categories.append(group.get("therapeutic_group", "N/A"))
        amounts.append(group.get("total_amount", 0))
        quantities.append(group.get("total_quantity", 0))
        atc_codes.append(group.get("atc_code", "N/A"))

    # Crear gráfico de barras horizontal
    fig = go.Figure()

    fig.add_trace(go.Bar(
        y=categories,
        x=amounts,
        orientation='h',
        name='Importe (€)',
        text=[f"€{amt:.2f}" for amt in amounts],
        textposition='auto',
        marker_color='#3498db',
        customdata=list(zip(atc_codes, quantities)),
        hovertemplate=(
            "<b>%{y}</b><br>"
            "Código ATC: %{customdata[0]}<br>"
            "Importe: €%{x:.2f}<br>"
            "Unidades: %{customdata[1]}<br>"
            "<extra></extra>"
        )
    ))

    fig.update_layout(
        title={
            'text': 'Ventas por Categoría Terapéutica (ATC)',
            'x': 0.5,
            'xanchor': 'center'
        },
        xaxis_title="Importe de Ventas (€)",
        yaxis_title="Categoría Terapéutica",
        height=400,
        margin=dict(l=20, r=20, t=60, b=20),
        plot_bgcolor='rgba(0,0,0,0)',
        showlegend=False
    )

    fig.update_xaxis(gridcolor='rgba(128,128,128,0.2)')
    fig.update_yaxis(gridcolor='rgba(128,128,128,0.2)')

    return dcc.Graph(figure=fig, style={'height': '400px'})

def create_prescription_analysis_chart(data: Dict[str, Any]) -> dcc.Graph:
    """
    Gráfico de análisis de prescripción vs OTC

    Args:
        data: Datos del endpoint /enriched/therapeutic-analysis/
    """
    if not data or not data.get("prescription_analysis"):
        return dcc.Graph(
            figure={
                'data': [],
                'layout': {
                    'title': 'Análisis Prescripción vs OTC',
                    'annotations': [{
                        'text': 'No hay datos de análisis de prescripción disponibles',
                        'showarrow': False,
                        'font': {'size': 16, 'color': 'gray'}
                    }]
                }
            },
            style={'height': '300px'}
        )

    prescription_analysis = data.get("prescription_analysis", [])

    # Preparar datos
    labels = []
    values = []
    colors = []

    for item in prescription_analysis:
        requires_prescription = item.get("requires_prescription")
        if requires_prescription:
            labels.append("Con Receta")
            colors.append("#e74c3c")  # Rojo para medicamentos con receta
        else:
            labels.append("Sin Receta (OTC)")
            colors.append("#27ae60")  # Verde para OTC

        values.append(item.get("total_amount", 0))

    # Crear gráfico de donut
    fig = go.Figure(data=[go.Pie(
        labels=labels,
        values=values,
        hole=0.4,
        marker_colors=colors,
        textinfo='label+percent+value',
        texttemplate='<b>%{label}</b><br>%{percent}<br>€%{value:.2f}',
        hovertemplate=(
            "<b>%{label}</b><br>"
            "Importe: €%{value:.2f}<br>"
            "Porcentaje: %{percent}<br>"
            "<extra></extra>"
        )
    )])

    fig.update_layout(
        title={
            'text': 'Distribución: Prescripción vs OTC',
            'x': 0.5,
            'xanchor': 'center'
        },
        height=300,
        margin=dict(l=20, r=20, t=60, b=20),
        showlegend=True,
        legend=dict(
            orientation="h",
            yanchor="bottom",
            y=-0.2,
            xanchor="center",
            x=0.5
        )
    )

    return dcc.Graph(figure=fig, style={'height': '300px'})

def create_generic_opportunities_table(data: Dict[str, Any]) -> html.Div:
    """
    Tabla de oportunidades de genéricos

    Args:
        data: Datos del endpoint /enriched/generic-opportunities/
    """
    if not data or not data.get("opportunities"):
        return html.Div([
            dbc.Alert(
                html.Div([
                    html.I(className="fas fa-info-circle me-2"),
                    "No se encontraron oportunidades de genéricos en el período seleccionado"
                ]),
                color="info"
            )
        ])

    opportunities = data.get("opportunities", [])
    summary = data.get("summary", {})

    # Crear filas de la tabla
    table_rows = []
    for opp in opportunities[:10]:  # Mostrar solo top 10
        branded = opp.get("branded_product", {})
        generic = opp.get("generic_alternative", {})
        savings = opp.get("savings", {})
        sale_info = opp.get("sale_info", {})

        table_rows.append(html.Tr([
            html.Td(branded.get("name", "N/A"), className="text-truncate"),
            html.Td(branded.get("laboratory", "N/A")),
            html.Td(f"€{branded.get('official_price', 0):.2f}", className="text-end"),
            html.Td(f"€{generic.get('official_price', 0):.2f}", className="text-end"),
            html.Td([
                html.Span(f"€{savings.get('total_potential', 0):.2f}", className="text-success fw-bold"),
                html.Small(f" ({savings.get('percentage', 0):.1f}%".replace(".", ","), className="text-muted")
            ], className="text-end"),
            html.Td(str(sale_info.get("quantity", 1)), className="text-center")
        ]))

    # Resumen de oportunidades
    total_opportunities = summary.get("total_opportunities", 0)
    total_savings = summary.get("total_potential_savings", 0)

    return html.Div([
        dbc.Row([
            dbc.Col([
                dbc.Card([
                    dbc.CardBody([
                        html.H5([
                            html.I(className="fas fa-pills me-2 text-success"),
                            "Oportunidades de Genéricos"
                        ], className="card-title"),
                        html.P([
                            f"{total_opportunities} oportunidades detectadas con ahorro potencial de ",
                            html.Strong(f"€{total_savings:.2f}", className="text-success")
                        ], className="mb-3")
                    ])
                ])
            ], width=12)
        ], className="mb-3"),

        dbc.Row([
            dbc.Col([
                dbc.Table([
                    html.Thead([
                        html.Tr([
                            html.Th("Medicamento Marca"),
                            html.Th("Laboratorio"),
                            html.Th("Precio Marca", className="text-end"),
                            html.Th("Precio Genérico", className="text-end"),
                            html.Th("Ahorro Potencial", className="text-end"),
                            html.Th("Unidades", className="text-center")
                        ])
                    ]),
                    html.Tbody(table_rows)
                ], striped=True, hover=True, responsive=True, size="sm")
            ], width=12)
        ])
    ])

def create_enrichment_sources_chart(data: Dict[str, Any]) -> dcc.Graph:
    """
    Gráfico de fuentes de enriquecimiento

    Args:
        data: Datos del endpoint /enrichment/status/
    """
    if not data or not data.get("enrichment_sources"):
        return dcc.Graph(
            figure={
                'data': [],
                'layout': {
                    'title': 'Fuentes de Enriquecimiento',
                    'annotations': [{
                        'text': 'No hay datos de fuentes disponibles',
                        'showarrow': False,
                        'font': {'size': 16, 'color': 'gray'}
                    }]
                }
            },
            style={'height': '300px'}
        )

    sources = data.get("enrichment_sources", [])

    # Preparar datos
    source_names = []
    source_counts = []
    colors = []

    color_map = {
        'CIMA': '#3498db',      # Azul para CIMA (oficial)
        'ML': '#e67e22',        # Naranja para ML (inferido)
        'nomenclator': '#27ae60', # Verde para nomenclator
        'manual': '#9b59b6'     # Púrpura para manual
    }

    for source in sources:
        source_name = source.get("source", "Desconocido")
        source_names.append(source_name)
        source_counts.append(source.get("count", 0))
        colors.append(color_map.get(source_name, '#95a5a6'))  # Gris por defecto

    # Crear gráfico de barras
    fig = go.Figure(data=[go.Bar(
        x=source_names,
        y=source_counts,
        marker_color=colors,
        text=source_counts,
        textposition='auto',
        hovertemplate=(
            "<b>%{x}</b><br>"
            "Registros: %{y}<br>"
            "<extra></extra>"
        )
    )])

    fig.update_layout(
        title={
            'text': 'Fuentes de Enriquecimiento',
            'x': 0.5,
            'xanchor': 'center'
        },
        xaxis_title="Fuente de Datos",
        yaxis_title="Número de Registros",
        height=300,
        margin=dict(l=20, r=20, t=60, b=20),
        plot_bgcolor='rgba(0,0,0,0)',
        showlegend=False
    )

    fig.update_xaxis(gridcolor='rgba(128,128,128,0.2)')
    fig.update_yaxis(gridcolor='rgba(128,128,128,0.2)')

    return dcc.Graph(figure=fig, style={'height': '300px'})
