# frontend/components/encargos/widget.py
"""
Cazador de Encargos - Main Widget Component

Dashboard widget showing pending customer orders requiring attention.
Displays KPIs, critical orders list, and quick actions.

Usage:
    from components.encargos import create_encargos_widget

    widget = create_encargos_widget()
    # Data is populated via callbacks

See: docs/CONTROL_MODULES_ARCHITECTURE.md
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
from components.base import BaseCard
from dash import html
from dash_iconify import DashIconify
from styles.design_tokens import BORDER_RADIUS, COLORS, SPACING
from utils.helpers import format_currency


# =============================================================================
# Severity Color Mapping
# =============================================================================

SEVERITY_COLORS = {
    "critical": COLORS["danger"],
    "urgent": COLORS["warning"],
    "warning": COLORS["info"],
    "info": COLORS["success"],
}

SEVERITY_ICONS = {
    "critical": "mdi:alert-circle",
    "urgent": "mdi:alert",
    "warning": "mdi:clock-alert",
    "info": "mdi:clock-outline",
}

SEVERITY_LABELS = {
    "critical": "Critico (15+ dias)",
    "urgent": "Urgente (10-14 dias)",
    "warning": "Alerta (5-9 dias)",
    "info": "Reciente (<5 dias)",
}

STATUS_LABELS = {
    "NEW": "Nuevo",
    "ALERTED": "Alertado",
    "CONTACTED": "Contactado",
    "TO_RETURN": "Para devolver",
    "RESOLVED": "Resuelto",
}


# =============================================================================
# Main Widget
# =============================================================================


def create_encargos_widget(widget_id: str = "encargos-widget") -> html.Div:
    """
    Create the main Cazador de Encargos dashboard widget.

    Args:
        widget_id: Base ID for the widget components

    Returns:
        html.Div: Complete widget with KPIs and orders list

    Note:
        Widget content is populated via callbacks using:
        - {widget_id}-summary for KPIs
        - {widget_id}-list for orders list
        - {widget_id}-sync-btn for sync action
    """
    return html.Div(
        [
            # Header with title and sync button
            html.Div(
                [
                    html.Div(
                        [
                            DashIconify(
                                icon="mdi:clipboard-text-clock",
                                width=28,
                                height=28,
                                style={
                                    "color": COLORS["warning"],
                                    "marginRight": SPACING["s"],
                                },
                            ),
                            html.H5(
                                "Cazador de Encargos",
                                className="mb-0 d-inline",
                                style={"fontWeight": "600"},
                            ),
                        ],
                        className="d-flex align-items-center",
                    ),
                    dbc.Button(
                        [
                            DashIconify(
                                icon="mdi:sync",
                                width=18,
                                height=18,
                                className="me-1",
                            ),
                            "Sincronizar",
                        ],
                        id=f"{widget_id}-sync-btn",
                        color="outline-primary",
                        size="sm",
                        className="ms-auto",
                    ),
                ],
                className="d-flex align-items-center justify-content-between mb-3",
            ),
            # Summary KPIs row
            html.Div(
                id=f"{widget_id}-summary",
                children=[_create_loading_placeholder()],
            ),
            # Divider
            html.Hr(style={"margin": f"{SPACING['m']} 0"}),
            # Section header for orders list
            html.Div(
                [
                    html.H6(
                        "Encargos Criticos",
                        className="mb-0",
                        style={"color": COLORS["text_primary"], "fontWeight": "500"},
                    ),
                    html.Small(
                        "Requieren atencion inmediata",
                        className="text-muted",
                    ),
                ],
                className="mb-3",
            ),
            # Orders list
            html.Div(
                id=f"{widget_id}-list",
                children=[_create_loading_placeholder()],
                style={
                    "maxHeight": "400px",
                    "overflowY": "auto",
                },
            ),
            # Footer with link to full page
            html.Div(
                [
                    dbc.Button(
                        [
                            "Ver todos los encargos",
                            DashIconify(
                                icon="mdi:arrow-right",
                                width=18,
                                height=18,
                                className="ms-2",
                            ),
                        ],
                        id=f"{widget_id}-view-all-btn",
                        color="link",
                        className="p-0",
                    ),
                ],
                className="text-center mt-3",
            ),
            # Toast for notifications
            dbc.Toast(
                id=f"{widget_id}-toast",
                header="Cazador de Encargos",
                icon="info",
                dismissable=True,
                is_open=False,
                duration=4000,
                style={"position": "fixed", "top": 66, "right": 10, "zIndex": 1000},
            ),
        ],
        id=widget_id,
    )


def _create_loading_placeholder() -> html.Div:
    """Create a loading placeholder for the widget."""
    return html.Div(
        [
            html.I(
                className="fas fa-spinner fa-spin me-2",
                style={"color": COLORS["text_secondary"]},
            ),
            html.Span("Cargando encargos...", className="text-muted"),
        ],
        className="text-center py-4",
    )


# =============================================================================
# Summary KPIs Component
# =============================================================================


def create_encargos_summary_kpis(summary: Dict[str, Any]) -> html.Div:
    """
    Create KPI summary cards from API response.

    Args:
        summary: API response from GET /customer-orders/summary
            {
                "total_pending": 1248,
                "total_value": 21728.12,
                "by_severity": {
                    "critical": {"count": 1129, "value": 19453.50},
                    "urgent": {"count": 50, "value": 1200.00},
                    ...
                },
                "by_status": {...}
            }

    Returns:
        html.Div: Row of KPI cards
    """
    total_pending = summary.get("total_pending", 0)
    total_value = summary.get("total_value", 0)
    by_severity = summary.get("by_severity", {})

    critical_count = by_severity.get("critical", {}).get("count", 0)
    critical_value = by_severity.get("critical", {}).get("value", 0)

    # Calculate percentage critical
    critical_pct = (critical_count / total_pending * 100) if total_pending > 0 else 0

    kpi_cards = [
        # Total Pending
        _create_mini_kpi(
            label="Pendientes",
            value=f"{total_pending:,}".replace(",", "."),
            icon="mdi:clipboard-list",
            color=COLORS["primary"],
        ),
        # Critical Orders
        _create_mini_kpi(
            label="Criticos",
            value=f"{critical_count:,}".replace(",", "."),
            icon="mdi:alert-circle",
            color=COLORS["danger"],
            subtitle=f"{critical_pct:.0f}% del total",
        ),
        # Total Value
        _create_mini_kpi(
            label="Valor Total",
            value=format_currency(total_value),
            icon="mdi:currency-eur",
            color=COLORS["success"],
        ),
    ]

    return html.Div(
        [
            dbc.Row(
                [dbc.Col(kpi, width=4) for kpi in kpi_cards],
                className="g-2",
            )
        ]
    )


def _create_mini_kpi(
    label: str,
    value: str,
    icon: str,
    color: str,
    subtitle: Optional[str] = None,
) -> html.Div:
    """Create a compact KPI card for the widget."""
    return html.Div(
        [
            html.Div(
                [
                    DashIconify(
                        icon=icon,
                        width=24,
                        height=24,
                        style={"color": color},
                    ),
                ],
                className="mb-1",
            ),
            html.Div(
                value,
                style={
                    "fontSize": "1.25rem",
                    "fontWeight": "700",
                    "color": COLORS["text_primary"],
                    "lineHeight": "1.2",
                },
            ),
            html.Div(
                label,
                style={
                    "fontSize": "0.75rem",
                    "color": COLORS["text_secondary"],
                },
            ),
            html.Div(
                subtitle,
                style={
                    "fontSize": "0.7rem",
                    "color": COLORS["text_muted"],
                },
            )
            if subtitle
            else None,
        ],
        style={
            "padding": SPACING["s"],
            "backgroundColor": COLORS["bg_secondary"],
            "borderRadius": BORDER_RADIUS["md"],
            "textAlign": "center",
        },
    )


# =============================================================================
# Orders List Component
# =============================================================================


def create_orders_list(
    orders: List[Dict[str, Any]],
    widget_id: str = "encargos-widget",
    max_display: int = 10,
) -> html.Div:
    """
    Create a list of order rows from API response.

    Args:
        orders: List of orders from API (GET /customer-orders/pending)
        widget_id: Base ID for action buttons
        max_display: Maximum number of orders to display

    Returns:
        html.Div: Container with order rows
    """
    if not orders:
        return html.Div(
            [
                html.Div(
                    [
                        DashIconify(
                            icon="mdi:check-circle",
                            width=48,
                            height=48,
                            style={"color": COLORS["success"]},
                        ),
                    ],
                    className="text-center mb-2",
                ),
                html.P(
                    "No hay encargos criticos pendientes",
                    className="text-muted text-center mb-0",
                ),
            ],
            className="py-4",
        )

    # Display only top N orders
    display_orders = orders[:max_display]

    order_rows = [
        create_order_row(order, idx, widget_id)
        for idx, order in enumerate(display_orders)
    ]

    # Add "more" indicator if there are more orders
    if len(orders) > max_display:
        order_rows.append(
            html.Div(
                [
                    html.Span(
                        f"+ {len(orders) - max_display} encargos mas...",
                        className="text-muted",
                        style={"fontSize": "0.875rem"},
                    ),
                ],
                className="text-center py-2",
            )
        )

    return html.Div(order_rows)


def create_order_row(
    order: Dict[str, Any],
    index: int,
    widget_id: str = "encargos-widget",
) -> html.Div:
    """
    Create a single order row with customer info and quick actions.

    Args:
        order: Order data from API
        index: Row index (for unique IDs)
        widget_id: Base ID for action buttons

    Returns:
        html.Div: Order row component
    """
    erp_order_id = order.get("erp_order_id", "")
    customer_name = order.get("customer_name", "Cliente desconocido")
    customer_phone = order.get("customer_phone", "")
    product_desc = order.get("product_description", "Producto")
    product_cn = order.get("product_cn", "")
    days_pending = order.get("days_pending", 0)
    severity = order.get("severity", "info")
    status = order.get("status", "NEW")
    economic_value = order.get("economic_value", 0)

    severity_color = SEVERITY_COLORS.get(severity, COLORS["secondary"])
    severity_icon = SEVERITY_ICONS.get(severity, "mdi:circle")
    status_label = STATUS_LABELS.get(status, status)

    # Phone display - clickable if available
    phone_element = None
    if customer_phone:
        phone_element = html.A(
            [
                DashIconify(
                    icon="mdi:phone",
                    width=14,
                    height=14,
                    className="me-1",
                ),
                customer_phone,
            ],
            href=f"tel:{customer_phone}",
            style={
                "fontSize": "0.75rem",
                "color": COLORS["primary"],
                "textDecoration": "none",
            },
        )

    return html.Div(
        [
            # Left: Severity indicator
            html.Div(
                [
                    DashIconify(
                        icon=severity_icon,
                        width=24,
                        height=24,
                        style={"color": severity_color},
                    ),
                ],
                style={
                    "minWidth": "36px",
                    "display": "flex",
                    "alignItems": "center",
                    "justifyContent": "center",
                },
            ),
            # Middle: Order info
            html.Div(
                [
                    # Customer name + phone
                    html.Div(
                        [
                            html.Span(
                                customer_name,
                                style={
                                    "fontWeight": "600",
                                    "color": COLORS["text_primary"],
                                    "fontSize": "0.9rem",
                                },
                            ),
                            html.Span(
                                f" - {days_pending} dias",
                                style={
                                    "color": severity_color,
                                    "fontSize": "0.8rem",
                                    "fontWeight": "500",
                                },
                            ),
                        ],
                        className="mb-1",
                    ),
                    # Product description
                    html.Div(
                        [
                            html.Span(
                                product_desc[:50] + "..."
                                if len(product_desc) > 50
                                else product_desc,
                                style={
                                    "fontSize": "0.8rem",
                                    "color": COLORS["text_secondary"],
                                },
                            ),
                            html.Span(
                                f" ({product_cn})" if product_cn else "",
                                style={
                                    "fontSize": "0.75rem",
                                    "color": COLORS["text_muted"],
                                },
                            ),
                        ],
                    ),
                    # Phone + Value
                    html.Div(
                        [
                            phone_element,
                            html.Span(
                                f" | {format_currency(economic_value)}",
                                style={
                                    "fontSize": "0.75rem",
                                    "color": COLORS["text_secondary"],
                                },
                            )
                            if phone_element
                            else html.Span(
                                format_currency(economic_value),
                                style={
                                    "fontSize": "0.75rem",
                                    "color": COLORS["text_secondary"],
                                },
                            ),
                        ],
                        className="mt-1",
                    ),
                ],
                style={"flex": "1", "paddingLeft": SPACING["s"]},
            ),
            # Right: Quick actions
            html.Div(
                [
                    # Status badge
                    dbc.Badge(
                        status_label,
                        color="secondary" if status == "NEW" else "primary",
                        className="mb-2",
                        style={"fontSize": "0.65rem"},
                    ),
                    # Action buttons
                    html.Div(
                        [
                            dbc.Button(
                                [
                                    DashIconify(
                                        icon="mdi:phone-check",
                                        width=14,
                                        height=14,
                                    ),
                                ],
                                id={
                                    "type": f"{widget_id}-contact-btn",
                                    "index": erp_order_id,
                                },
                                color="outline-success",
                                size="sm",
                                className="me-1",
                                title="Marcar contactado",
                                style={"padding": "2px 6px"},
                            ),
                            dbc.Button(
                                [
                                    DashIconify(
                                        icon="mdi:package-variant-closed-remove",
                                        width=14,
                                        height=14,
                                    ),
                                ],
                                id={
                                    "type": f"{widget_id}-return-btn",
                                    "index": erp_order_id,
                                },
                                color="outline-warning",
                                size="sm",
                                title="Marcar para devolver",
                                style={"padding": "2px 6px"},
                            ),
                        ],
                        className="d-flex",
                    ),
                ],
                style={
                    "minWidth": "80px",
                    "display": "flex",
                    "flexDirection": "column",
                    "alignItems": "flex-end",
                },
            ),
        ],
        className="d-flex align-items-start",
        style={
            "padding": f"{SPACING['s']} {SPACING['xs']}",
            "borderBottom": f"1px solid {COLORS['border_light']}",
        },
    )


# =============================================================================
# Card Wrapper (for standalone use)
# =============================================================================


def create_encargos_card(widget_id: str = "encargos-widget") -> BaseCard:
    """
    Create the encargos widget wrapped in a BaseCard.

    Use this for standalone placement in dashboards.

    Args:
        widget_id: Base ID for the widget

    Returns:
        BaseCard: Card containing the encargos widget
    """
    return BaseCard(
        children=[create_encargos_widget(widget_id)],
        shadow="sm",
        padding=SPACING["l"],
        role="region",
        aria_label="Panel de encargos pendientes",
    )
