# frontend/components/empty_state.py
"""
Componente reutilizable de estado vacío para xFarma.
Proporciona mensajes amigables cuando no hay datos disponibles.
"""

import html as html_escape
from typing import Optional

import dash_bootstrap_components as dbc
from dash import html
from dash_iconify import DashIconify
from styles.design_tokens import COLORS, BORDER_RADIUS


def create_empty_state(
    icon: str,
    title: str,
    description: str,
    cta_text: Optional[str] = None,
    cta_href: Optional[str] = None,
    cta_id: Optional[str] = None,
    help_text: Optional[str] = None,
    help_href: Optional[str] = None,
    size: str = "md",
) -> html.Div:
    """
    Crear componente de estado vacío reutilizable.

    Args:
        icon: Nombre del icono de Iconify (ej: "mdi:file-document-outline")
        title: Título principal del mensaje
        description: Descripción detallada
        cta_text: Texto del botón de acción (opcional)
        cta_href: URL del botón de acción (opcional)
        cta_id: ID del botón para callbacks (opcional, usado en lugar de href)
        help_text: Texto del enlace de ayuda (opcional)
        help_href: URL del enlace de ayuda (opcional)
        size: Tamaño del icono - "sm" (60px), "md" (80px), "lg" (100px)

    Returns:
        Componente html.Div con el estado vacío
    """

    # Definir tamaños de icono
    icon_sizes = {"sm": 60, "md": 80, "lg": 100}
    icon_size = icon_sizes.get(size, 80)

    # Construir componente de icono
    icon_component = DashIconify(
        icon=icon,
        width=icon_size,
        height=icon_size,
        color=COLORS["text_muted"],  # Usar constante de colores
        className="mb-3",
    )

    # Construir título
    title_component = html.H4(title, className="text-muted mb-2", style={"fontSize": "1.25rem", "fontWeight": "600"})

    # Construir descripción
    description_component = html.P(
        description,
        className="text-muted mb-4",
        style={"fontSize": "0.95rem", "maxWidth": "400px", "margin": "0 auto 1.5rem auto"},
    )

    # Construir botón CTA si se proporciona
    cta_component = None
    if cta_text and (cta_href or cta_id):
        if cta_id:
            # Botón con ID para callbacks
            cta_component = dbc.Button(
                [html.I(className="fas fa-arrow-right me-2"), cta_text],
                id=cta_id,
                color="primary",
                size="lg",
                className="mb-3",
                style={"minWidth": "200px"},
            )
        else:
            # Botón con href para navegación
            cta_component = dbc.Button(
                [html.I(className="fas fa-arrow-right me-2"), cta_text],
                href=cta_href,
                color="primary",
                size="lg",
                className="mb-3",
                style={"minWidth": "200px"},
            )

    # Construir enlace de ayuda si se proporciona
    help_component = None
    if help_text and help_href:
        help_component = html.A(
            [html.I(className="fas fa-question-circle me-1"), help_text],
            href=help_href,
            className="text-muted",
            style={"fontSize": "0.875rem", "textDecoration": "none"},
        )

    # Ensamblar componente completo
    children = [icon_component, title_component, description_component]

    if cta_component:
        children.append(cta_component)

    if help_component:
        children.append(help_component)

    return html.Div(
        children=children,
        className="text-center py-5",
        style={
            "minHeight": "400px",
            "display": "flex",
            "flexDirection": "column",
            "justifyContent": "center",
            "alignItems": "center",
            "padding": "3rem 1.5rem",
        },
    )


# Componentes predefinidos para casos comunes


def empty_sales_state() -> html.Div:
    """Estado vacío para página de ventas sin datos."""
    return create_empty_state(
        icon="mdi:chart-line-variant",
        title="Aún no hay datos de ventas",
        description="Para comenzar a analizar las ventas de tu farmacia, carga un archivo de tu sistema ERP.",
        cta_text="Cargar Archivo de Ventas",
        cta_href="/upload",
    )


def empty_upload_history_state() -> html.Div:
    """Estado vacío para historial de cargas vacío."""
    return create_empty_state(
        icon="mdi:file-upload-outline",
        title="No hay archivos cargados",
        description="El historial de cargas aparecerá aquí una vez subas tu primer archivo de ventas.",
        help_text="Ver formatos soportados",
        help_href="/docs/formatos",
        size="sm",
    )


def empty_generic_study_state() -> html.Div:
    """Estado vacío para estudios genéricos sin datos."""
    return create_empty_state(
        icon="mdi:flask-outline",
        title="No hay datos para generar estudio",
        description="Necesitas cargar datos de ventas antes de generar un estudio de oportunidades de genéricos.",
        cta_text="Cargar Datos de Ventas",
        cta_href="/upload",
    )


def empty_admin_users_state() -> html.Div:
    """Estado vacío para lista de usuarios vacía."""
    return create_empty_state(
        icon="mdi:account-multiple-outline",
        title="No hay usuarios adicionales",
        description="Tu farmacia aún no tiene usuarios adicionales. Crea una invitación para añadir colaboradores.",
        cta_text="Crear Invitación",
        cta_id="create-invitation-btn",  # Para callback
    )


def empty_admin_invitations_state() -> html.Div:
    """Estado vacío para lista de invitaciones vacía."""
    return create_empty_state(
        icon="mdi:email-outline",
        title="No hay invitaciones pendientes",
        description="Todas las invitaciones han sido aceptadas o no hay invitaciones activas.",
        size="sm",
    )


def empty_search_results_state(query: str) -> html.Div:
    """Estado vacío para resultados de búsqueda sin coincidencias."""
    # Sanitizar input del usuario para prevenir XSS
    sanitized_query = html_escape.escape(query)
    return create_empty_state(
        icon="mdi:magnify",
        title="No se encontraron resultados",
        description=f"No hay resultados para '{sanitized_query}'. Intenta con otros términos de búsqueda.",
        size="sm",
    )


def empty_table_state(entity_name: str = "elementos") -> html.Div:
    """Estado vacío genérico para tablas sin datos."""
    return create_empty_state(
        icon="mdi:table",
        title=f"No hay {entity_name}",
        description=f"La tabla de {entity_name} está vacía. Los datos aparecerán aquí cuando estén disponibles.",
        size="sm",
    )


# =========================================================================
# ESTADOS VACÍOS PARA ANÁLISIS (Issue #436 - Design Review)
# Estados accionables con sugerencias para secciones de análisis
# =========================================================================

def create_analysis_empty_state(
    message: str,
    icon: str = "mdi:chart-bar",
    suggestions: Optional[list] = None,
    min_height: str = "300px",
    section_name: Optional[str] = None
) -> html.Div:
    """
    Crea un estado vacío accionable para secciones de análisis.

    Mejora UX con:
    - Icono visual para identificación rápida
    - Mensaje claro explicando la situación
    - Sugerencias accionables (opcional)
    - Accesibilidad con role="status" y aria-live

    Args:
        message: Mensaje principal del estado vacío
        icon: Icono MDI (ej: "mdi:chart-line", "mdi:table")
        suggestions: Lista de sugerencias para resolver el problema
        min_height: Altura mínima del contenedor
        section_name: Nombre de la sección para accesibilidad

    Returns:
        html.Div con el estado vacío estilizado

    Example:
        >>> create_analysis_empty_state(
        ...     message="No hay datos de evolución",
        ...     icon="mdi:chart-timeline-variant",
        ...     suggestions=[
        ...         "Ajusta el rango de fechas",
        ...         "Verifica la cobertura ATC de los productos"
        ...     ]
        ... )
    """
    children = []

    # Icono principal
    children.append(
        html.Div(
            DashIconify(
                icon=icon,
                width=48,
                height=48,
                color=COLORS["text_muted"]
            ),
            className="mb-3"
        )
    )

    # Mensaje principal
    children.append(
        html.P(
            message,
            className="text-muted mb-2",
            style={
                "fontSize": "1rem",
                "fontWeight": "500"
            }
        )
    )

    # Sugerencias (si las hay)
    if suggestions:
        children.append(
            html.Div(
                [
                    html.Small(
                        "Posibles soluciones:",
                        className="text-secondary fw-bold d-block mb-2"
                    ),
                    html.Ul(
                        [html.Li(s, className="text-muted small") for s in suggestions],
                        className="list-unstyled mb-0",
                        style={"textAlign": "left", "maxWidth": "400px", "margin": "0 auto"}
                    )
                ]
            )
        )

    # Construir el aria-label
    aria_label = "Estado vacío"
    if section_name:
        aria_label = f"Estado vacío en {section_name}"

    return html.Div(
        children,
        className="text-center py-4",
        style={
            "backgroundColor": COLORS["gray_100"],
            "borderRadius": BORDER_RADIUS["md"],
            "border": f"2px dashed {COLORS['border_light']}",
            "minHeight": min_height,
            "display": "flex",
            "flexDirection": "column",
            "justifyContent": "center",
            "alignItems": "center",
            "padding": "2rem"
        },
        role="status",
        **{"aria-live": "polite", "aria-label": aria_label}
    )


def create_error_state(
    error_message: str,
    icon: str = "mdi:alert-circle-outline",
    min_height: str = "300px"
) -> html.Div:
    """
    Crea un estado de error cuando algo falla.

    Args:
        error_message: Mensaje de error
        icon: Icono MDI
        min_height: Altura mínima del contenedor

    Returns:
        html.Div con estado de error
    """
    return html.Div(
        [
            DashIconify(
                icon=icon,
                width=48,
                height=48,
                color=COLORS["danger"]
            ),
            html.P(
                error_message,
                className="text-danger mt-3 mb-0",
                style={"fontSize": "0.95rem"}
            )
        ],
        className="text-center py-4 d-flex flex-column justify-content-center align-items-center",
        style={
            "minHeight": min_height,
            "backgroundColor": "#FEF2F2",  # Subtle danger background (not in tokens)
            "borderRadius": BORDER_RADIUS["md"],
            "border": f"1px solid {COLORS['danger_light']}",
        },
        role="alert",
        **{"aria-live": "assertive"}
    )
