"""
Trend Chart Component - Dashboard xFarma
Gráfico de tendencia de ventas para el dashboard principal (Issue #152)

Usage:
    from components.dashboard import create_trend_chart

    chart = create_trend_chart()
    # El gráfico se actualiza mediante callback con datos reales
"""


import plotly.graph_objects as go
from components.base import BaseCard
from dash import dcc, html
from dash_iconify import DashIconify
from styles.design_tokens import COLORS, SPACING


def create_trend_chart(chart_id: str = "sales-trend-chart") -> BaseCard:
    """
    Crea un gráfico de tendencia de ventas para los últimos 30 días.

    Args:
        chart_id: ID del componente dcc.Graph (para callbacks)

    Returns:
        BaseCard: Card con el gráfico de tendencia

    Note:
        El gráfico se inicializa vacío. Los datos reales se cargan
        mediante callback desde /api/sales/summary?group_by=day
    """
    # Gráfico placeholder (se actualizará con callback)
    empty_figure = go.Figure()
    empty_figure.update_layout(
        template="plotly_white",
        height=300,
        margin=dict(l=40, r=20, t=40, b=40),
        xaxis=dict(
            title="Fecha",
            showgrid=True,
            gridcolor=COLORS["border_light"],
        ),
        yaxis=dict(
            title="Ventas (€)",
            showgrid=True,
            gridcolor=COLORS["border_light"],
        ),
        hovermode="x unified",
        font=dict(family="system-ui, -apple-system, sans-serif", size=12),
    )

    # Mensaje de carga inicial
    empty_figure.add_annotation(
        text="Cargando datos de ventas...",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color=COLORS["text_secondary"]),
    )

    chart_content = [
        # Header con título e icono
        html.Div(
            [
                html.Div(
                    [
                        DashIconify(
                            icon="mdi:chart-line",
                            width=24,
                            height=24,
                            style={
                                "color": COLORS["primary"],
                                "marginRight": SPACING["xs"],
                            },
                        ),
                        html.H5(
                            "Tendencia de Ventas",
                            className="mb-0",
                            style={"display": "inline-block"},
                        ),
                    ],
                    className="d-flex align-items-center mb-1",
                ),
                html.P(
                    "Evolución de ventas en los últimos 30 días",
                    className="text-muted mb-3",
                    style={"fontSize": "0.875rem"},
                ),
            ]
        ),
        # Gráfico
        dcc.Graph(
            id=chart_id,
            figure=empty_figure,
            config={
                "responsive": True,
                "displayModeBar": False,  # Ocultar toolbar para diseño limpio
                "displaylogo": False,
            },
            style={"height": "300px"},
        ),
    ]

    return BaseCard(
        children=chart_content,
        shadow="sm",
        padding=SPACING["l"],
        role="region",
        aria_label="Gráfico de tendencia de ventas de los últimos 30 días",
    )


def create_trend_figure(sales_data: list[dict]) -> go.Figure:
    """
    Crea la figura de Plotly con datos reales de ventas.

    Args:
        sales_data: Lista de diccionarios con 'date' y 'total_sales'
            Ejemplo: [
                {"date": "2025-01-01", "total_sales": 1234.56},
                {"date": "2025-01-02", "total_sales": 2345.67},
                ...
            ]

    Returns:
        go.Figure: Figura configurada para mostrar tendencia

    Note:
        Esta función se usa en el callback para actualizar el gráfico
        con datos reales desde la API.
    """
    if not sales_data or len(sales_data) == 0:
        # Mostrar mensaje cuando no hay datos
        fig = go.Figure()
        fig.add_annotation(
            text="No hay datos de ventas disponibles",
            xref="paper",
            yref="paper",
            x=0.5,
            y=0.5,
            showarrow=False,
            font=dict(size=14, color=COLORS["text_secondary"]),
        )
        fig.update_layout(
            template="plotly_white",
            height=300,
            margin=dict(l=40, r=20, t=40, b=40),
        )
        return fig

    # Extraer fechas y valores
    dates = [item["date"] for item in sales_data]
    values = [item["total_sales"] for item in sales_data]

    # Crear gráfico de línea
    fig = go.Figure()

    # Línea principal
    fig.add_trace(
        go.Scatter(
            x=dates,
            y=values,
            mode="lines+markers",
            name="Ventas",
            line=dict(
                color=COLORS["primary"],
                width=3,
            ),
            marker=dict(
                size=6,
                color=COLORS["primary"],
                line=dict(color="white", width=2),
            ),
            hovertemplate="<b>%{x|%d/%m/%Y}</b><br>"
            + "Ventas: €%{y:,.2f}<br>"
            + "<extra></extra>",  # Ocultar nombre de serie en hover
        )
    )

    # Área bajo la curva (relleno sutil)
    fig.add_trace(
        go.Scatter(
            x=dates,
            y=values,
            fill="tozeroy",
            fillcolor=COLORS["primary"].replace(")", ", 0.1)").replace(
                "rgb", "rgba"
            ),  # Transparencia 10%
            line=dict(width=0),
            showlegend=False,
            hoverinfo="skip",
        )
    )

    # Configuración del layout
    fig.update_layout(
        template="plotly_white",
        height=300,
        margin=dict(l=40, r=20, t=40, b=40),
        xaxis=dict(
            title="",
            showgrid=True,
            gridcolor=COLORS["border_light"],
            tickformat="%d/%m",  # Formato español: día/mes
        ),
        yaxis=dict(
            title="Ventas (€)",
            showgrid=True,
            gridcolor=COLORS["border_light"],
            tickformat=",.0f",  # Formato con separador de miles
            separatethousands=True,
        ),
        hovermode="x unified",
        font=dict(
            family="system-ui, -apple-system, sans-serif",
            size=12,
            color=COLORS["text_primary"],
        ),
        plot_bgcolor="white",
        paper_bgcolor="white",
    )

    return fig
