"""
KPI Card Component - Dashboard xFarma
Tarjeta reutilizable para mostrar KPIs con iconografía (Issue #152)

Usage:
    from components.dashboard import create_kpi_card

    kpi = create_kpi_card(
        title="Ventas Totales",
        value="€45,678",
        icon="mdi:cash-multiple",
        color="success",
        subtitle="Últimos 30 días",
        trend="+15%"
    )
"""

from components.base import BaseCard
from dash import html
from dash_iconify import DashIconify
from styles.design_tokens import COLORS, SPACING


def create_kpi_card(
    title: str,
    value: str,
    icon: str,
    color: str = "primary",
    subtitle: str = None,
    trend: str = None,
    trend_positive: bool = True,
    component_id: str = None,
) -> BaseCard:
    """
    Crea una tarjeta KPI con icono y valor destacado.

    Args:
        title: Título del KPI (ej: "Ventas Totales")
        value: Valor principal a mostrar (ej: "€45,678")
        icon: Icono de dash-iconify (ej: "mdi:cash-multiple")
        color: Color del icono - "primary", "success", "warning", "info", "danger"
        subtitle: Texto secundario opcional (ej: "Últimos 30 días")
        trend: Indicador de tendencia opcional (ej: "+15%", "-5%")
        trend_positive: Si True, muestra tendencia en verde; si False, en rojo
        component_id: ID opcional para el valor (útil para callbacks)

    Returns:
        BaseCard: Card configurada con el KPI

    Examples:
        # KPI básico
        create_kpi_card(
            title="Productos Únicos",
            value="1,234",
            icon="mdi:package-variant"
        )

        # KPI con tendencia
        create_kpi_card(
            title="Ventas del Mes",
            value="€45,678",
            icon="mdi:cash-multiple",
            color="success",
            subtitle="Enero 2025",
            trend="+15%",
            trend_positive=True
        )
    """
    # Mapeo de colores a clases Bootstrap
    color_map = {
        "primary": COLORS["primary"],
        "success": COLORS["success"],
        "warning": COLORS["warning"],
        "info": COLORS["info"],
        "danger": COLORS["danger"],
    }

    icon_color = color_map.get(color, COLORS["primary"])
    trend_color = COLORS["success"] if trend_positive else COLORS["danger"]
    trend_icon = "mdi:trending-up" if trend_positive else "mdi:trending-down"

    # Contenido del KPI
    kpi_content = [
        # Fila superior: Icono + Título
        html.Div(
            [
                html.Div(
                    [
                        DashIconify(
                            icon=icon,
                            width=40,
                            height=40,
                            style={"color": icon_color},
                        ),
                    ],
                    style={"marginBottom": SPACING["s"]},
                ),
                html.H6(
                    title,
                    className="text-muted mb-0",
                    style={"fontSize": "0.875rem", "fontWeight": "500"},
                ),
            ],
            className="mb-3",
        ),
        # Valor principal
        html.Div(
            [
                html.H3(
                    value,
                    id=component_id,
                    className="mb-0",
                    style={
                        "fontSize": "2rem",
                        "fontWeight": "700",
                        "color": COLORS["text_primary"],
                    },
                ),
            ],
            className="mb-2",
        ),
    ]

    # Fila inferior: Subtitle + Trend (si existen)
    bottom_row_content = []

    if subtitle:
        bottom_row_content.append(
            html.Span(
                subtitle,
                className="text-muted",
                style={"fontSize": "0.875rem"},
            )
        )

    if trend:
        if bottom_row_content:
            bottom_row_content.append(html.Span(" • ", className="text-muted mx-1"))

        bottom_row_content.append(
            html.Span(
                [
                    DashIconify(
                        icon=trend_icon,
                        width=16,
                        height=16,
                        style={"color": trend_color, "marginRight": "4px"},
                    ),
                    html.Span(
                        trend,
                        style={
                            "color": trend_color,
                            "fontWeight": "600",
                            "fontSize": "0.875rem",
                        },
                    ),
                ]
            )
        )

    if bottom_row_content:
        kpi_content.append(
            html.Div(
                bottom_row_content,
                className="d-flex align-items-center",
            )
        )

    # Crear card usando Design System
    return BaseCard(
        children=kpi_content,
        shadow="sm",
        padding=SPACING["l"],
        role="region",
        aria_label=f"{title}: {value}",
    )


def create_yoy_kpi_card(
    title: str,
    change_percent: float,
    subtitle: str = None,
    icon: str = "mdi:chart-line-variant",
    component_id: str = None,
) -> BaseCard:
    """
    Crea una tarjeta KPI para variación Year-over-Year (YoY).

    El valor principal es el porcentaje de cambio, con colores y iconos
    que indican si es positivo (verde) o negativo (rojo).

    Args:
        title: Título del KPI (ej: "Trimestre YoY", "Mes YoY")
        change_percent: Porcentaje de cambio (-100 a +∞)
        subtitle: Texto descriptivo opcional (ej: "vs mismo período 2024")
        icon: Icono de dash-iconify (por defecto chart-line)
        component_id: ID opcional para el valor (útil para callbacks)

    Returns:
        BaseCard: Card configurada con el KPI YoY

    Examples:
        create_yoy_kpi_card(
            title="Q4 YoY",
            change_percent=12.5,
            subtitle="Oct-Dic vs 2023"
        )
    """
    # Determinar si es positivo, negativo o neutro
    is_positive = change_percent > 0
    is_negative = change_percent < 0
    is_neutral = change_percent == 0

    # Formatear el valor con signo
    if is_positive:
        formatted_value = f"+{change_percent:.1f}%"
        value_color = COLORS["success"]
        trend_icon = "mdi:trending-up"
        bg_color = "rgba(40, 167, 69, 0.1)"  # Verde claro
    elif is_negative:
        formatted_value = f"{change_percent:.1f}%"
        value_color = COLORS["danger"]
        trend_icon = "mdi:trending-down"
        bg_color = "rgba(220, 53, 69, 0.1)"  # Rojo claro
    else:
        formatted_value = "0.0%"
        value_color = COLORS["text_secondary"]
        trend_icon = "mdi:trending-neutral"
        bg_color = "rgba(108, 117, 125, 0.1)"  # Gris claro

    # Contenido del KPI YoY
    kpi_content = [
        # Icono de tendencia grande
        html.Div(
            [
                html.Div(
                    [
                        DashIconify(
                            icon=trend_icon,
                            width=32,
                            height=32,
                            style={"color": value_color},
                        ),
                    ],
                    style={
                        "backgroundColor": bg_color,
                        "borderRadius": "8px",
                        "padding": SPACING["s"],
                        "display": "inline-block",
                    },
                ),
            ],
            className="mb-2",
        ),
        # Título
        html.H6(
            title,
            className="text-muted mb-1",
            style={"fontSize": "0.85rem", "fontWeight": "500"},
        ),
        # Valor principal (porcentaje de cambio)
        html.Div(
            [
                html.Span(
                    formatted_value,
                    id=component_id,
                    style={
                        "fontSize": "1.75rem",
                        "fontWeight": "700",
                        "color": value_color,
                    },
                ),
            ],
            className="mb-1",
        ),
    ]

    # Subtitle opcional
    if subtitle:
        kpi_content.append(
            html.Small(
                subtitle,
                className="text-muted",
                style={"fontSize": "0.75rem"},
            )
        )

    # Crear card
    return BaseCard(
        children=kpi_content,
        shadow="sm",
        padding=SPACING["m"],
        role="region",
        aria_label=f"{title}: {formatted_value}",
    )
