"""
Activity Feed Component - Dashboard xFarma
Feed de actividad reciente para el dashboard principal (Issue #152)

Usage:
    from components.dashboard import create_activity_feed

    feed = create_activity_feed()
    # El feed se actualiza mediante callback con actividades reales
"""

from datetime import datetime

from components.base import BaseCard
from dash import html
from dash_iconify import DashIconify
from styles.design_tokens import COLORS, SPACING
from utils.helpers import format_date


def create_activity_feed(feed_id: str = "activity-feed-content") -> BaseCard:
    """
    Crea un feed de actividad reciente del sistema.

    Args:
        feed_id: ID del contenedor de actividades (para callbacks)

    Returns:
        BaseCard: Card con el feed de actividad

    Note:
        El feed se inicializa con mensaje de carga.
        Los datos reales se cargan mediante callback.
    """
    feed_content = [
        # Header con título e icono
        html.Div(
            [
                html.Div(
                    [
                        DashIconify(
                            icon="mdi:timeline-clock",
                            width=24,
                            height=24,
                            style={
                                "color": COLORS["primary"],
                                "marginRight": SPACING["xs"],
                            },
                        ),
                        html.H5(
                            "Actividad Reciente",
                            className="mb-0",
                            style={"display": "inline-block"},
                        ),
                    ],
                    className="d-flex align-items-center mb-1",
                ),
                html.P(
                    "Últimas acciones en el sistema",
                    className="text-muted mb-3",
                    style={"fontSize": "0.875rem"},
                ),
            ]
        ),
        # Contenedor de actividades (se actualiza con callback)
        html.Div(
            id=feed_id,
            children=[
                _create_activity_placeholder()
            ],  # Placeholder inicial
        ),
    ]

    return BaseCard(
        children=feed_content,
        shadow="sm",
        padding=SPACING["l"],
        role="region",
        aria_label="Feed de actividad reciente del sistema",
    )


def _create_activity_placeholder():
    """Placeholder mientras se cargan las actividades"""
    return html.Div(
        [
            html.I(className="fas fa-spinner fa-spin me-2", style={"color": COLORS["text_secondary"]}),
            html.Span("Cargando actividad reciente...", className="text-muted"),
        ],
        className="text-center py-3",
    )


def create_activity_list(activities: list[dict]) -> html.Div:
    """
    Crea la lista de actividades con datos reales.

    Args:
        activities: Lista de diccionarios con actividades
            Formato esperado:
            [
                {
                    "type": "upload" | "sync" | "enrichment" | "system",
                    "message": "Descripción de la actividad",
                    "timestamp": "2025-01-15T10:30:00Z",
                    "status": "success" | "warning" | "error" | "info"
                },
                ...
            ]

    Returns:
        html.Div: Contenedor con lista de actividades

    Note:
        Esta función se usa en el callback para actualizar el feed
        con actividades reales desde diferentes fuentes.
    """
    if not activities or len(activities) == 0:
        return html.Div(
            [
                html.Div(
                    [
                        DashIconify(
                            icon="mdi:information-outline",
                            width=48,
                            height=48,
                            style={"color": COLORS["text_secondary"]},
                        ),
                    ],
                    className="text-center mb-2",
                ),
                html.P(
                    "No hay actividad reciente",
                    className="text-muted text-center mb-0",
                    style={"fontSize": "0.875rem"},
                ),
            ],
            className="py-4",
        )

    # Mapeo de tipos a iconos
    icon_map = {
        "upload": "mdi:cloud-upload",
        "sync": "mdi:sync",
        "enrichment": "mdi:database-check",
        "system": "mdi:cog",
    }

    # Mapeo de status a colores
    color_map = {
        "success": COLORS["success"],
        "warning": COLORS["warning"],
        "error": COLORS["danger"],
        "info": COLORS["info"],
    }

    activity_items = []

    for idx, activity in enumerate(activities):
        activity_type = activity.get("type", "system")
        message = activity.get("message", "Actividad sin descripción")
        timestamp = activity.get("timestamp")
        status = activity.get("status", "info")

        icon = icon_map.get(activity_type, "mdi:circle")
        color = color_map.get(status, COLORS["text_secondary"])

        # Formatear timestamp
        time_str = ""
        if timestamp:
            try:
                # Intentar parsear timestamp ISO
                dt = datetime.fromisoformat(timestamp.replace("Z", "+00:00"))
                now = datetime.now(dt.tzinfo)
                diff = now - dt

                # Mostrar tiempo relativo
                if diff.days > 0:
                    time_str = f"Hace {diff.days} día{'s' if diff.days > 1 else ''}"
                elif diff.seconds >= 3600:
                    hours = diff.seconds // 3600
                    time_str = f"Hace {hours} hora{'s' if hours > 1 else ''}"
                elif diff.seconds >= 60:
                    minutes = diff.seconds // 60
                    time_str = f"Hace {minutes} minuto{'s' if minutes > 1 else ''}"
                else:
                    time_str = "Hace unos segundos"
            except Exception:
                # Fallback a formato simple
                time_str = format_date(timestamp, "display")

        activity_item = html.Div(
            [
                # Icono
                html.Div(
                    [
                        DashIconify(
                            icon=icon,
                            width=20,
                            height=20,
                            style={"color": color},
                        ),
                    ],
                    style={
                        "minWidth": "32px",
                        "marginRight": SPACING["s"],
                    },
                ),
                # Contenido
                html.Div(
                    [
                        html.Div(
                            message,
                            style={
                                "fontSize": "0.875rem",
                                "color": COLORS["text_primary"],
                                "marginBottom": "2px",
                            },
                        ),
                        html.Div(
                            time_str,
                            style={
                                "fontSize": "0.75rem",
                                "color": COLORS["text_secondary"],
                            },
                        ),
                    ],
                    style={"flex": "1"},
                ),
            ],
            className="d-flex align-items-start",
            style={
                "paddingTop": SPACING["s"],
                "paddingBottom": SPACING["s"],
                "borderBottom": f"1px solid {COLORS['border_light']}"
                if idx < len(activities) - 1
                else "none",
            },
        )

        activity_items.append(activity_item)

    return html.Div(activity_items)


def extract_activities_from_data(dashboard_data: dict) -> list[dict]:
    """
    Extrae actividades desde los datos del dashboard.

    Args:
        dashboard_data: Datos del dashboard (desde dashboard-data-store)

    Returns:
        list[dict]: Lista de actividades ordenadas por timestamp (más reciente primero)

    Note:
        Esta función se usa en el callback para transformar datos del
        sistema en actividades para el feed.
    """
    activities = []

    # Actividad: Última carga de ventas
    sales = dashboard_data.get("sales", {})
    last_upload = sales.get("last_upload")
    if last_upload and isinstance(last_upload, dict):
        filename = last_upload.get("filename", "archivo")
        upload_date = last_upload.get("date")
        if upload_date:
            activities.append(
                {
                    "type": "upload",
                    "message": f"Archivo cargado: {filename}",
                    "timestamp": upload_date,
                    "status": "success",
                }
            )

    # Actividad: Sincronización CIMA
    system_status = dashboard_data.get("system", {})
    if system_status and system_status.get("components"):
        components = system_status.get("components", {})
        cima_info = components.get("cima", {})
        cima_sync_date = cima_info.get("last_success_at")
        cima_products = cima_info.get("products", 0)

        if cima_sync_date:
            activities.append(
                {
                    "type": "sync",
                    "message": f"Sincronización CIMA completada ({cima_products:,} productos)".replace(
                        ",", "."
                    ),
                    "timestamp": cima_sync_date,
                    "status": "success",
                }
            )

        # Actividad: Sincronización Nomenclátor
        nomen_info = components.get("nomenclator", {})
        nomen_sync_date = nomen_info.get("last_success_at")
        nomen_products = nomen_info.get("products", 0)

        if nomen_sync_date:
            activities.append(
                {
                    "type": "sync",
                    "message": f"Nomenclátor actualizado ({nomen_products:,} productos)".replace(
                        ",", "."
                    ),
                    "timestamp": nomen_sync_date,
                    "status": "success",
                }
            )

    # Actividad: Enriquecimiento (si hay datos)
    enrichment_rate = sales.get("enrichment_rate", 0)
    if enrichment_rate > 0:
        enriched_count = sales.get("enriched_records", 0)
        if enriched_count > 0:
            # Usar timestamp del último upload como referencia
            activities.append(
                {
                    "type": "enrichment",
                    "message": f"Enriquecimiento completado: {enriched_count:,} productos".replace(
                        ",", "."
                    ),
                    "timestamp": last_upload.get("date") if last_upload else None,
                    "status": "success",
                }
            )

    # Ordenar por timestamp (más reciente primero)
    activities.sort(
        key=lambda x: x.get("timestamp", ""), reverse=True
    )

    # Limitar a 5 actividades más recientes
    return activities[:5]
