"""
Sistema de modales de confirmación para acciones críticas.
Proporciona confirmación visual antes de ejecutar operaciones destructivas o importantes.
"""

from typing import Dict, Literal, Optional

import dash_bootstrap_components as dbc
from dash import Input, Output, html

ModalType = Literal["danger", "warning", "info", "primary"]


class ConfirmationModal:
    """Gestor de modales de confirmación."""

    MODAL_CONFIGS = {
        "danger": {
            "color": "danger",
            "icon": "fas fa-exclamation-triangle",
            "confirm_text": "Eliminar",
            "cancel_text": "Cancelar",
        },
        "warning": {
            "color": "warning",
            "icon": "fas fa-exclamation-circle",
            "confirm_text": "Continuar",
            "cancel_text": "Cancelar",
        },
        "info": {"color": "info", "icon": "fas fa-info-circle", "confirm_text": "Aceptar", "cancel_text": "Cancelar"},
        "primary": {
            "color": "primary",
            "icon": "fas fa-check-circle",
            "confirm_text": "Confirmar",
            "cancel_text": "Cancelar",
        },
    }


def create_confirmation_modal(modal_id: str = "confirmation-modal", modal_type: ModalType = "warning") -> dbc.Modal:
    """
    Crea un modal de confirmación reutilizable.

    Args:
        modal_id: ID único del modal
        modal_type: Tipo de modal (danger, warning, info, primary)
    """
    config = ConfirmationModal.MODAL_CONFIGS.get(modal_type, ConfirmationModal.MODAL_CONFIGS["warning"])

    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(
                    [
                        html.I(className=f"{config['icon']} me-2 text-{config['color']}"),
                        html.Span(id=f"{modal_id}-title", children="Confirmar acción"),
                    ],
                    className="d-flex align-items-center",
                ),
                close_button=True,
            ),
            dbc.ModalBody(
                id=f"{modal_id}-body",
                children=[
                    html.P("¿Está seguro de que desea continuar con esta acción?", id=f"{modal_id}-message"),
                    html.Div(id=f"{modal_id}-details", className="alert alert-light mt-3", style={"display": "none"}),
                ],
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        config["cancel_text"],
                        id=f"{modal_id}-cancel",
                        color="secondary",
                        outline=True,
                        className="me-2",
                    ),
                    dbc.Button(
                        config["confirm_text"], id=f"{modal_id}-confirm", color=config["color"], className="ms-2"
                    ),
                ]
            ),
        ],
        id=modal_id,
        centered=True,
        backdrop="static",
        keyboard=False,
        size="md",
    )


def create_catalog_update_modal() -> dbc.Modal:
    """
    Modal específico para confirmar actualización del catálogo.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(
                    [
                        html.I(className="fas fa-sync-alt me-2 text-primary"),
                        html.Span("Actualizar Catálogo de Productos"),
                    ],
                    className="d-flex align-items-center",
                ),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    html.P(
                        [
                            html.Strong("Esta operación actualizará el catálogo completo de productos."),
                            html.Br(),
                            "El proceso puede tardar varios minutos dependiendo del volumen de datos.",
                        ]
                    ),
                    dbc.Alert(
                        [
                            html.H6("ℹ️ Información del proceso:", className="alert-heading"),
                            html.Ul(
                                [
                                    html.Li("Se verificarán cambios en CIMA y Nomenclátor"),
                                    html.Li("Se actualizarán solo los productos modificados"),
                                    html.Li("El sistema permanecerá operativo durante la actualización"),
                                    html.Li("Recibirá notificaciones del progreso"),
                                ],
                                className="mb-0",
                            ),
                        ],
                        color="info",
                        className="mt-3",
                    ),
                    html.Div(
                        [
                            html.P("Última actualización:", className="text-muted mb-1"),
                            html.Strong(id="last-catalog-update-time", children="Cargando..."),
                            html.Br(),
                            html.Small(id="catalog-update-status", className="text-muted"),
                        ],
                        className="mt-3 p-3 bg-light rounded",
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Cancelar", id="catalog-update-cancel", color="secondary", outline=True, className="me-2"
                    ),
                    dbc.Button(
                        html.Div([html.I(className="fas fa-sync-alt me-2"), "Actualizar Ahora"]),
                        id="catalog-update-confirm",
                        color="primary",
                        className="ms-2",
                    ),
                ]
            ),
        ],
        id="catalog-update-modal",
        is_open=False,  # Asegurar que empieza cerrado
        centered=True,
        backdrop="static",
        keyboard=False,
        size="lg",
    )


def create_delete_confirmation_modal(item_type: str = "elemento", item_name: str = "") -> dbc.Modal:
    """
    Modal específico para confirmar eliminación de elementos.

    Args:
        item_type: Tipo de elemento a eliminar (archivo, datos, etc.)
        item_name: Nombre del elemento a eliminar
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(
                    [html.I(className="fas fa-trash-alt me-2 text-danger"), html.Span(f"Eliminar {item_type}")],
                    className="d-flex align-items-center",
                ),
                close_button=True,
                className="bg-danger bg-opacity-10",
            ),
            dbc.ModalBody(
                [
                    dbc.Alert(
                        html.Div(
                            [
                                html.I(className="fas fa-exclamation-triangle me-2"),
                                html.Strong("¡Atención! "),
                                "Esta acción no se puede deshacer.",
                            ]
                        ),
                        color="danger",
                    ),
                    html.P(
                        [
                            "Está a punto de eliminar permanentemente: ",
                            html.Br(),
                            html.Strong(item_name, className="text-danger"),
                        ]
                    ),
                    html.Hr(),
                    html.P("Para confirmar, escriba 'ELIMINAR' en el campo siguiente:", className="text-muted"),
                    dbc.Input(
                        id="delete-confirmation-input", placeholder="Escriba ELIMINAR para confirmar", className="mb-3"
                    ),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button("Cancelar", id="delete-cancel", color="secondary", outline=True, className="me-2"),
                    dbc.Button(
                        html.Div([html.I(className="fas fa-trash-alt me-2"), "Eliminar Permanentemente"]),
                        id="delete-confirm",
                        variant="danger",
                        className="ms-2",
                        disabled=True,
                    ),
                ]
            ),
        ],
        id="delete-confirmation-modal",
        centered=True,
        backdrop="static",
        keyboard=False,
        size="md",
    )


def create_action_confirmation_modal() -> dbc.Modal:
    """
    Modal genérico para confirmar cualquier acción.
    Los valores se configuran dinámicamente via callbacks.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(id="action-modal-header", className="d-flex align-items-center"), close_button=True
            ),
            dbc.ModalBody(id="action-modal-body"),
            dbc.ModalFooter(id="action-modal-footer"),
        ],
        id="action-confirmation-modal",
        centered=True,
        backdrop="static",
        keyboard=False,
        size="md",
    )


# Funciones helper para disparar modales
def show_confirmation(
    title: str,
    message: str,
    confirm_text: str = "Confirmar",
    cancel_text: str = "Cancelar",
    modal_type: ModalType = "warning",
    details: Optional[str] = None,
) -> Dict:
    """
    Prepara los datos para mostrar un modal de confirmación.

    Ejemplo de uso en callback:
        return {
            "confirmation-modal": True,
            "confirmation-modal-data": show_confirmation(
                title="Actualizar Catálogo",
                message="¿Desea actualizar el catálogo ahora?",
                modal_type="primary"
            )
        }
    """
    return {
        "title": title,
        "message": message,
        "confirm_text": confirm_text,
        "cancel_text": cancel_text,
        "type": modal_type,
        "details": details,
    }


def register_confirmation_callbacks(app):
    """Registra los callbacks para el sistema de modales de confirmación."""

    # Callback para habilitar/deshabilitar botón de eliminar basado en input
    @app.callback(
        Output("delete-confirm", "disabled"), Input("delete-confirmation-input", "value"), prevent_initial_call=True
    )
    def validate_delete_confirmation(input_value):
        """Habilita el botón de eliminar solo si se escribe 'ELIMINAR'."""
        if input_value and input_value.upper() == "ELIMINAR":
            return False
        return True

    # Callback genérico para actualizar contenido del modal dinámicamente
    @app.callback(
        [
            Output("action-modal-header", "children"),
            Output("action-modal-body", "children"),
            Output("action-modal-footer", "children"),
        ],
        Input("action-modal-data-store", "data"),
        prevent_initial_call=True,
    )
    def update_action_modal_content(modal_data):
        """Actualiza el contenido del modal de acción basado en los datos."""
        if not modal_data:
            return [], [], []

        modal_type = modal_data.get("type", "warning")
        config = ConfirmationModal.MODAL_CONFIGS.get(modal_type)

        # Header
        header = [
            html.I(className=f"{config['icon']} me-2 text-{config['color']}"),
            html.Span(modal_data.get("title", "Confirmar acción")),
        ]

        # Body
        body_content = [html.P(modal_data.get("message", "¿Está seguro?"))]

        if modal_data.get("details"):
            body_content.append(dbc.Alert(modal_data["details"], color="light", className="mt-3"))

        # Footer
        footer = [
            dbc.Button(
                modal_data.get("cancel_text", "Cancelar"),
                id="action-modal-cancel",
                color="secondary",
                outline=True,
                className="me-2",
            ),
            dbc.Button(
                modal_data.get("confirm_text", config["confirm_text"]),
                id="action-modal-confirm",
                color=config["color"],
                className="ms-2",
            ),
        ]

        return header, body_content, footer


# Estilos CSS para los modales
MODAL_STYLES = """
<style>
.modal-backdrop {
    backdrop-filter: blur(2px);
}

.modal-content {
    border: none;
    box-shadow: 0 10px 40px rgba(0,0,0,0.2);
}

.modal-header {
    border-bottom: 2px solid #dee2e6;
}

.modal-footer {
    border-top: 1px solid #dee2e6;
    background-color: #f8f9fa;
}

/* Animación de entrada */
.modal.show .modal-dialog {
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        transform: translateY(-50px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

/* Responsive */
@media (max-width: 576px) {
    .modal-dialog {
        margin: 0.5rem;
        max-width: calc(100% - 1rem);
    }
}
</style>
"""
