# frontend/components/common.py
"""
Componentes comunes reutilizables para xfarma.
"""

from typing import Any, List, Optional, Union

import dash_bootstrap_components as dbc
from dash import html
from utils.constants import COLORS


def create_loading_spinner(size: str = "md", color: str = "primary", text: Optional[str] = None) -> html.Div:
    """
    Crear spinner de carga con texto opcional.

    Args:
        size: Tamaño del spinner ("sm", "md", "lg")
        color: Color del spinner
        text: Texto descriptivo

    Returns:
        Componente html.Div con spinner
    """

    spinner = dbc.Spinner(color=color, size=size)

    if text:
        return html.Div([spinner, html.P(text, className="text-muted mt-2 mb-0")], className="text-center py-4")
    else:
        return html.Div(spinner, className="text-center py-4")


def create_alert(
    message: str,
    alert_type: str = "info",
    dismissable: bool = False,
    icon: Optional[str] = None,
    duration: Optional[int] = None,
) -> dbc.Alert:
    """
    Crear alerta con mensaje y tipo.

    Args:
        message: Mensaje de la alerta
        alert_type: Tipo de alerta ("success", "danger", "warning", "info")
        dismissable: Si se puede cerrar
        icon: Icono FontAwesome opcional
        duration: Duración en ms para auto-cerrar

    Returns:
        Componente dbc.Alert
    """

    # Mapear iconos por defecto según el tipo
    default_icons = {
        "success": "fas fa-check-circle",
        "danger": "fas fa-exclamation-triangle",
        "warning": "fas fa-exclamation-circle",
        "info": "fas fa-info-circle",
    }

    # Usar icono por defecto si no se especifica
    if not icon and alert_type in default_icons:
        icon = default_icons[alert_type]

    # Construir contenido
    content = []
    if icon:
        content.extend([html.I(className=f"{icon} me-2"), message])
    else:
        content.append(message)

    return dbc.Alert(
        content, color=alert_type, dismissable=dismissable, duration=duration, className="d-flex align-items-center"
    )


def create_status_badge(status: str, custom_colors: Optional[dict] = None) -> dbc.Badge:
    """
    Crear badge de estado con colores predefinidos.

    Args:
        status: Estado a mostrar
        custom_colors: Mapeo personalizado de estados a colores

    Returns:
        Componente dbc.Badge
    """

    # Colores por defecto para estados comunes
    status_colors = {
        # Estados de conexión
        "online": "success",
        "offline": "danger",
        "connecting": "warning",
        # Estados de procesamiento
        "completed": "success",
        "processing": "warning",
        "pending": "secondary",
        "failed": "danger",
        "error": "danger",
        # Estados generales
        "active": "success",
        "inactive": "secondary",
        "enabled": "success",
        "disabled": "secondary",
    }

    # Usar colores personalizados si se proporcionan
    if custom_colors:
        status_colors.update(custom_colors)

    color = status_colors.get(status.lower(), "secondary")

    return dbc.Badge(status.title(), color=color, className="badge-status")


def create_info_card(
    title: str,
    content: Union[str, List[Any]],
    icon: Optional[str] = None,
    color: str = COLORS["primary"],
    footer: Optional[str] = None,
) -> dbc.Card:
    """
    Crear tarjeta informativa simple.

    Args:
        title: Título de la tarjeta
        content: Contenido (texto o lista de componentes)
        icon: Icono FontAwesome opcional
        color: Color del tema
        footer: Texto del footer opcional

    Returns:
        Componente dbc.Card
    """

    # Header con icono
    header_content = []
    if icon:
        header_content.extend([html.I(className=f"{icon} me-2", style={"color": color}), title])
    else:
        header_content.append(title)

    # Construir tarjeta
    card_components = [dbc.CardHeader(header_content, className="bg-transparent border-0 pb-0")]

    # Body
    if isinstance(content, str):
        card_components.append(dbc.CardBody(content))
    else:
        card_components.append(dbc.CardBody(content))

    # Footer
    if footer:
        card_components.append(dbc.CardFooter(footer, className="bg-transparent border-0 pt-0 text-muted small"))

    return dbc.Card(card_components, className="h-100 shadow-sm")


def create_empty_state(
    title: str = "No hay datos",
    message: str = "No hay información disponible para mostrar",
    icon: str = "fas fa-inbox",
    action_button: Optional[dict] = None,
) -> html.Div:
    """
    Crear estado vacío con mensaje y acción opcional.

    Args:
        title: Título del estado vacío
        message: Mensaje descriptivo
        icon: Icono FontAwesome
        action_button: Diccionario con 'text', 'href' o 'id' para botón de acción

    Returns:
        Componente html.Div
    """

    elements = [
        html.I(className=f"{icon} fa-3x text-muted mb-3"),
        html.H4(title, className="text-muted mb-2"),
        html.P(message, className="text-muted mb-3"),
    ]

    # Añadir botón de acción si se proporciona
    if action_button:
        button_props = {"children": action_button.get("text", "Acción"), "color": "primary", "outline": True}

        if "href" in action_button:
            button_props["href"] = action_button["href"]
        if "id" in action_button:
            button_props["id"] = action_button["id"]

        elements.append(dbc.Button(**button_props))

    return html.Div(elements, className="empty-state text-center py-5")


def create_table_wrapper(
    table_component: Any,
    title: Optional[str] = None,
    actions: Optional[List[Any]] = None,
    footer_text: Optional[str] = None,
) -> dbc.Card:
    """
    Crear wrapper para tablas con título, acciones y footer.

    Args:
        table_component: Componente de tabla
        title: Título opcional
        actions: Lista de botones/acciones
        footer_text: Texto del footer

    Returns:
        Componente dbc.Card con tabla
    """

    card_components = []

    # Header con título y acciones
    if title or actions:
        header_content = []

        if title:
            header_content.append(html.Div(html.H5(title, className="mb-0"), className="col"))

        if actions:
            header_content.append(html.Div(actions, className="col-auto d-flex gap-2"))

        card_components.append(
            dbc.CardHeader(dbc.Row(header_content, className="align-items-center"), className="bg-transparent")
        )

    # Body con tabla
    card_components.append(dbc.CardBody(table_component, className="p-0"))

    # Footer opcional
    if footer_text:
        card_components.append(dbc.CardFooter(footer_text, className="text-muted small"))

    return dbc.Card(card_components, className="shadow-sm")


def create_error_alert(message: str, dismissable: bool = True) -> dbc.Alert:
    """
    Crear alerta de error con estilo predefinido.

    Args:
        message: Mensaje de error
        dismissable: Si se puede cerrar

    Returns:
        Componente dbc.Alert de error
    """
    return create_alert(message, alert_type="danger", dismissable=dismissable, icon="fas fa-exclamation-triangle")


def create_success_alert(message: str, dismissable: bool = True) -> dbc.Alert:
    """
    Crear alerta de éxito con estilo predefinido.

    Args:
        message: Mensaje de éxito
        dismissable: Si se puede cerrar

    Returns:
        Componente dbc.Alert de éxito
    """
    return create_alert(message, alert_type="success", dismissable=dismissable, icon="fas fa-check-circle")


def create_connection_status(is_connected: bool, service_name: str = "Servidor", show_details: bool = True) -> html.Div:
    """
    Crear indicador de estado de conexión.

    Args:
        is_connected: Si está conectado
        service_name: Nombre del servicio
        show_details: Si mostrar detalles adicionales

    Returns:
        Componente html.Div con estado
    """

    if is_connected:
        color = "success"
        icon = "fas fa-circle"
        status_text = "Conectado"
        message = f"{service_name} funcionando correctamente"
    else:
        color = "danger"
        icon = "fas fa-circle"
        status_text = "Desconectado"
        message = f"No se puede conectar a {service_name}"

    elements = [html.Span([html.I(className=f"{icon} me-2", style={"color": COLORS[color]}), html.Strong(status_text)])]

    if show_details:
        elements.append(html.Small(message, className="text-muted ms-2"))

    return html.Div(elements, className="connection-status")
