"""
UMAP Scatter Plot Component (Issue #458)

Interactive 2D scatter plot for product clustering visualization.
Uses WebGL (Scattergl) for performance with ~5k products.

Features:
- Color by NECESIDAD category
- Ghost points for moved products
- Hover with product details
- Click to select product
- Accessibility mode with high contrast colors
"""

from typing import Dict, List, Optional

import plotly.graph_objects as go
from dash import dcc, html
import dash_bootstrap_components as dbc

# =============================================================================
# COLOR PALETTES
# =============================================================================

# Standard palette for NECESIDAD categories
NECESIDAD_COLORS = {
    "higiene_bucal": "#3498db",
    "higiene_corporal": "#2980b9",
    "higiene_capilar": "#1abc9c",
    "higiene_intima": "#16a085",
    "proteccion_solar": "#f39c12",
    "proteccion_solar_infantil": "#e67e22",
    "antiedad": "#e91e63",
    "antimanchas": "#c2185b",
    "hidratacion_facial": "#9c27b0",
    "hidratacion_corporal": "#7b1fa2",
    "acne": "#673ab7",
    "piel_sensible": "#512da8",
    "piel_atopica": "#3f51b5",
    "vitaminas_general": "#27ae60",
    "probioticos": "#2ecc71",
    "energia": "#00bcd4",
    "defensas": "#009688",
    "articulaciones": "#4caf50",
    "digestion": "#8bc34a",
    "alimentacion_bebe": "#ff9800",
    "higiene_bebe": "#ff5722",
    "ojos_secos": "#795548",
    "alergia_ocular": "#607d8b",
    "mucosidad_respiratoria": "#78909c",
    "dolor_muscular": "#f44336",
    "circulacion": "#e53935",
    "control_peso": "#cddc39",
    "sexual_masculino": "#ff5252",
    "sexual_femenino": "#ff4081",
    "otros": "#95a5a6",
    "sin_clasificar": "#bdc3c7",
}

# Accessibility palette (deuteranopia-friendly, high contrast)
NECESIDAD_COLORS_A11Y = {
    "higiene_bucal": "#0077BB",
    "higiene_corporal": "#0077BB",
    "higiene_capilar": "#33BBEE",
    "higiene_intima": "#33BBEE",
    "proteccion_solar": "#EE7733",
    "proteccion_solar_infantil": "#EE7733",
    "antiedad": "#CC3311",
    "antimanchas": "#CC3311",
    "hidratacion_facial": "#EE3377",
    "hidratacion_corporal": "#EE3377",
    "acne": "#AA3377",
    "piel_sensible": "#AA3377",
    "piel_atopica": "#AA3377",
    "vitaminas_general": "#009988",
    "probioticos": "#009988",
    "energia": "#009988",
    "defensas": "#009988",
    "articulaciones": "#009988",
    "digestion": "#009988",
    "alimentacion_bebe": "#CCBB44",
    "higiene_bebe": "#CCBB44",
    "ojos_secos": "#882255",
    "alergia_ocular": "#882255",
    "mucosidad_respiratoria": "#88CCEE",
    "dolor_muscular": "#CC3311",
    "circulacion": "#CC3311",
    "control_peso": "#CCBB44",
    "sexual_masculino": "#EE3377",
    "sexual_femenino": "#EE3377",
    "otros": "#BBBBBB",
    "sin_clasificar": "#DDDDDD",
}


def get_color(necesidad: str, a11y_mode: bool = False) -> str:
    """Get color for a NECESIDAD category."""
    palette = NECESIDAD_COLORS_A11Y if a11y_mode else NECESIDAD_COLORS
    if not necesidad:
        return palette.get("otros", "#95a5a6")
    return palette.get(necesidad.lower(), palette.get("otros", "#95a5a6"))


# =============================================================================
# COMPONENT
# =============================================================================

def create_umap_scatter(
    points: List[Dict],
    ghost_points: List[Dict] = None,
    color_by: str = "necesidad",
    selected_id: Optional[str] = None,
    a11y_mode: bool = False,
    height: int = 500,
    graph_id: str = "umap-scatter",
) -> dcc.Graph:
    """
    Create UMAP 2D scatter plot.

    Args:
        points: List of products with umap_x, umap_y, necesidad, etc.
        ghost_points: Products moved in session (shown as gray shadows)
        color_by: "necesidad" | "risk_level" | "verified"
        selected_id: ID of selected product (highlighted)
        a11y_mode: Use high-contrast accessibility colors
        height: Chart height in pixels
        graph_id: ID for the dcc.Graph component (default: "umap-scatter")

    Returns:
        dcc.Graph component with Plotly figure
    """
    fig = go.Figure()

    if not points:
        # Empty state
        fig.add_annotation(
            text="No hay productos con coordenadas UMAP.<br>Ejecuta el script de entrenamiento primero.",
            xref="paper",
            yref="paper",
            x=0.5,
            y=0.5,
            showarrow=False,
            font=dict(size=14, color="#6c757d"),
        )
        fig.update_layout(
            height=height,
            xaxis=dict(visible=False),
            yaxis=dict(visible=False),
        )
        return dcc.Graph(id=graph_id, figure=fig)

    # Prepare data
    x_coords = [p["umap_x"] for p in points]
    y_coords = [p["umap_y"] for p in points]
    names = [p.get("product_name", "Unknown") for p in points]
    ids = [p.get("id", "") for p in points]

    # Color based on mode
    if color_by == "necesidad":
        colors = [get_color(p.get("necesidad"), a11y_mode) for p in points]
    elif color_by == "risk_level":
        risk_colors = {"high": "#e74c3c", "medium": "#f39c12", "low": "#27ae60", "unknown": "#95a5a6"}
        colors = [risk_colors.get(p.get("risk_level", "unknown"), "#95a5a6") for p in points]
    elif color_by == "verified":
        colors = ["#27ae60" if p.get("human_verified") else "#3498db" for p in points]
    else:
        colors = ["#3498db"] * len(points)

    # Custom data for hover
    customdata = [
        [
            p.get("id", ""),
            p.get("necesidad", "Sin clasificar"),
            p.get("detected_brand", "Sin marca"),
            f"{p.get('ml_confidence', 0) * 100:.0f}%" if p.get("ml_confidence") else "N/A",
            f"{p.get('centroid_distance_zscore', 0):.2f}" if p.get("centroid_distance_zscore") else "N/A",
            "Verificado" if p.get("human_verified") else "Pendiente",
        ]
        for p in points
    ]

    # Main scatter trace (WebGL for performance)
    fig.add_trace(go.Scattergl(
        x=x_coords,
        y=y_coords,
        mode="markers",
        marker=dict(
            size=8,
            color=colors,
            line=dict(width=1, color="white"),
            opacity=0.8,
        ),
        text=names,
        customdata=customdata,
        hovertemplate=(
            "<b>%{text}</b><br>"
            "<br>"
            "Categor\u00eda: %{customdata[1]}<br>"
            "Marca: %{customdata[2]}<br>"
            "Confianza: %{customdata[3]}<br>"
            "Z-Score: %{customdata[4]}<br>"
            "Estado: %{customdata[5]}"
            "<extra></extra>"
        ),
        name="Productos",
    ))

    # Ghost points trace
    if ghost_points:
        ghost_x = [p["original_umap_x"] for p in ghost_points]
        ghost_y = [p["original_umap_y"] for p in ghost_points]
        ghost_names = [f"Movido: {p.get('product_name', 'Unknown')}" for p in ghost_points]

        fig.add_trace(go.Scattergl(
            x=ghost_x,
            y=ghost_y,
            mode="markers",
            marker=dict(
                size=6,
                color="lightgray",
                opacity=0.4,
                symbol="x",
                line=dict(width=1, color="gray"),
            ),
            text=ghost_names,
            hovertemplate="<b>%{text}</b><extra>Posici\u00f3n original</extra>",
            name="Movidos (original)",
        ))

    # Highlight selected point
    if selected_id:
        selected_points = [p for p in points if p.get("id") == selected_id]
        if selected_points:
            sp = selected_points[0]
            fig.add_trace(go.Scatter(
                x=[sp["umap_x"]],
                y=[sp["umap_y"]],
                mode="markers",
                marker=dict(
                    size=16,
                    color="rgba(0,0,0,0)",
                    line=dict(width=3, color="#e74c3c"),
                ),
                hoverinfo="skip",
                name="Seleccionado",
                showlegend=False,
            ))

    # Layout
    fig.update_layout(
        height=height,
        margin=dict(l=20, r=20, t=40, b=20),
        legend=dict(
            orientation="h",
            yanchor="bottom",
            y=1.02,
            xanchor="left",
            x=0,
        ),
        xaxis=dict(
            showgrid=False,
            zeroline=False,
            showticklabels=False,
            title="",
        ),
        yaxis=dict(
            showgrid=False,
            zeroline=False,
            showticklabels=False,
            title="",
        ),
        plot_bgcolor="white",
        paper_bgcolor="white",
        hovermode="closest",
        dragmode="pan",  # Default to pan for exploration
    )

    return dcc.Graph(
        id=graph_id,
        figure=fig,
        config={
            "responsive": True,
            "displayModeBar": True,
            "modeBarButtonsToAdd": ["zoom2d", "pan2d", "resetScale2d"],
            "modeBarButtonsToRemove": ["lasso2d", "select2d"],
            "scrollZoom": True,
        },
        style={"height": f"{height}px"},
    )


def create_umap_legend(a11y_mode: bool = False) -> html.Div:
    """Create color legend for NECESIDAD categories."""
    # Group categories by high-level groups
    groups = {
        "Higiene": ["higiene_bucal", "higiene_corporal", "higiene_capilar", "higiene_intima"],
        "Solar": ["proteccion_solar", "proteccion_solar_infantil"],
        "Facial": ["antiedad", "antimanchas", "hidratacion_facial", "acne", "piel_sensible"],
        "Corporal": ["hidratacion_corporal", "piel_atopica"],
        "Suplementos": ["vitaminas_general", "probioticos", "energia", "defensas"],
        "Beb\u00e9": ["alimentacion_bebe", "higiene_bebe"],
        "Otros": ["otros", "sin_clasificar"],
    }

    legend_items = []
    palette = NECESIDAD_COLORS_A11Y if a11y_mode else NECESIDAD_COLORS

    for group_name, categories in groups.items():
        # Get representative color (first category in group)
        color = palette.get(categories[0], "#95a5a6")

        legend_items.append(
            html.Div([
                html.Span(
                    style={
                        "display": "inline-block",
                        "width": "12px",
                        "height": "12px",
                        "backgroundColor": color,
                        "borderRadius": "50%",
                        "marginRight": "6px",
                    }
                ),
                html.Span(group_name, style={"fontSize": "12px"}),
            ], style={"display": "inline-block", "marginRight": "16px", "marginBottom": "4px"})
        )

    return html.Div(
        legend_items,
        style={
            "padding": "8px",
            "backgroundColor": "#f8f9fa",
            "borderRadius": "4px",
            "marginTop": "8px",
        }
    )


def create_umap_controls() -> html.Div:
    """Create control panel for UMAP scatter."""
    return html.Div([
        dbc.Row([
            dbc.Col([
                dbc.Label("Color por:", className="small"),
                dbc.RadioItems(
                    id="umap-color-by",
                    options=[
                        {"label": "Categor\u00eda", "value": "necesidad"},
                        {"label": "Riesgo (Z-Score)", "value": "risk_level"},
                        {"label": "Verificado", "value": "verified"},
                    ],
                    value="necesidad",
                    inline=True,
                    className="small",
                ),
            ], width=6),
            dbc.Col([
                dbc.Label("Opciones:", className="small"),
                dbc.Checklist(
                    id="umap-options",
                    options=[
                        {"label": "Alto contraste", "value": "a11y"},
                        {"label": "Solo verificados", "value": "verified_only"},
                    ],
                    value=[],
                    inline=True,
                    className="small",
                ),
            ], width=6),
        ], className="mb-2"),
    ], style={"padding": "8px", "backgroundColor": "#f8f9fa", "borderRadius": "4px"})
