# frontend/components/charts.py
"""
Componentes de gráficos Plotly con temas xfarma.
"""

from typing import Any, Dict, List, Optional, Union

import pandas as pd
import plotly.express as px
import plotly.graph_objects as go
from dash import dcc
from utils.constants import CHART_THEMES, COLORS
from utils.helpers import format_currency, format_percentage


def create_line_chart(
    data: List[Dict[str, Any]],
    x_field: str,
    y_field: str,
    title: str,
    x_title: Optional[str] = None,
    y_title: Optional[str] = None,
    color_field: Optional[str] = None,
    height: int = 400,
    show_markers: bool = True,
) -> dcc.Graph:
    """
    Crear gráfico de líneas con tema xfarma.

    Args:
        data: Lista de diccionarios con los datos
        x_field: Campo para eje X
        y_field: Campo para eje Y
        title: Título del gráfico
        x_title: Título del eje X
        y_title: Título del eje Y
        color_field: Campo para colorear líneas (opcional)
        height: Altura del gráfico
        show_markers: Si mostrar marcadores en puntos

    Returns:
        Componente dcc.Graph
    """

    if not data:
        return _create_empty_chart(title, height)

    df = pd.DataFrame(data)

    # Crear gráfico base
    if color_field and color_field in df.columns:
        fig = px.line(
            df,
            x=x_field,
            y=y_field,
            color=color_field,
            title=title,
            markers=show_markers,
            color_discrete_sequence=COLORS["chart_primary"],
        )
    else:
        fig = go.Figure()
        fig.add_trace(
            go.Scatter(
                x=df[x_field],
                y=df[y_field],
                mode="lines+markers" if show_markers else "lines",
                line=dict(color=COLORS["primary"], width=3),
                marker=dict(size=6, color=COLORS["primary"]),
                name=y_title or y_field,
            )
        )
        fig.update_layout(title=title)

    # Aplicar tema
    _apply_chart_theme(fig, height)

    # Configurar ejes
    fig.update_xaxes(title_text=x_title or x_field.replace("_", " ").title())
    fig.update_yaxes(title_text=y_title or y_field.replace("_", " ").title())

    return dcc.Graph(figure=fig, config={"displayModeBar": False}, className="chart-container")


def create_pie_chart(
    data: List[Dict[str, Any]],
    values_field: str,
    names_field: str,
    title: str,
    height: int = 400,
    hole_size: float = 0.3,
) -> dcc.Graph:
    """
    Crear gráfico de sectores (pie/donut) con tema xfarma.

    Args:
        data: Lista de diccionarios con los datos
        values_field: Campo con los valores numéricos
        names_field: Campo con los nombres/categorías
        title: Título del gráfico
        height: Altura del gráfico
        hole_size: Tamaño del agujero central (0 = pie, >0 = donut)

    Returns:
        Componente dcc.Graph
    """

    if not data:
        return _create_empty_chart(title, height)

    df = pd.DataFrame(data)

    # Crear gráfico
    fig = px.pie(
        df,
        values=values_field,
        names=names_field,
        title=title,
        hole=hole_size,
        color_discrete_sequence=COLORS["chart_primary"],
    )

    # Configurar apariencia
    fig.update_traces(
        textposition="inside",
        textinfo="percent+label",
        textfont_size=12,
        marker=dict(line=dict(color="#FFFFFF", width=2)),
    )

    # Aplicar tema
    _apply_chart_theme(fig, height)

    return dcc.Graph(figure=fig, config={"displayModeBar": False}, className="chart-container")


def create_bar_chart(
    data: List[Dict[str, Any]],
    x_field: str,
    y_field: str,
    title: str,
    x_title: Optional[str] = None,
    y_title: Optional[str] = None,
    color_field: Optional[str] = None,
    orientation: str = "v",
    height: int = 400,
) -> dcc.Graph:
    """
    Crear gráfico de barras con tema xfarma.

    Args:
        data: Lista de diccionarios con los datos
        x_field: Campo para eje X
        y_field: Campo para eje Y
        title: Título del gráfico
        x_title: Título del eje X
        y_title: Título del eje Y
        color_field: Campo para colorear barras
        orientation: Orientación ('v' vertical, 'h' horizontal)
        height: Altura del gráfico

    Returns:
        Componente dcc.Graph
    """

    if not data:
        return _create_empty_chart(title, height)

    df = pd.DataFrame(data)

    # Crear gráfico base
    if color_field and color_field in df.columns:
        fig = px.bar(
            df,
            x=x_field if orientation == "v" else y_field,
            y=y_field if orientation == "v" else x_field,
            color=color_field,
            title=title,
            orientation=orientation,
            color_discrete_sequence=COLORS["chart_primary"],
        )
    else:
        fig = go.Figure()
        fig.add_trace(
            go.Bar(
                x=df[x_field] if orientation == "v" else df[y_field],
                y=df[y_field] if orientation == "v" else df[x_field],
                orientation=orientation,
                marker_color=COLORS["primary"],
                name=y_title or y_field,
            )
        )
        fig.update_layout(title=title)

    # Aplicar tema
    _apply_chart_theme(fig, height)

    # Configurar ejes
    if orientation == "v":
        fig.update_xaxes(title_text=x_title or x_field.replace("_", " ").title())
        fig.update_yaxes(title_text=y_title or y_field.replace("_", " ").title())
    else:
        fig.update_xaxes(title_text=y_title or y_field.replace("_", " ").title())
        fig.update_yaxes(title_text=x_title or x_field.replace("_", " ").title())

    return dcc.Graph(figure=fig, config={"displayModeBar": False}, className="chart-container")


def create_metric_chart(
    current_value: Union[int, float],
    previous_value: Union[int, float],
    title: str,
    format_type: str = "number",
    height: int = 200,
) -> dcc.Graph:
    """
    Crear gráfico de métrica simple con indicador de cambio.

    Args:
        current_value: Valor actual
        previous_value: Valor anterior
        title: Título de la métrica
        format_type: Tipo de formato ('currency', 'percentage', 'number')
        height: Altura del gráfico

    Returns:
        Componente dcc.Graph
    """

    # Calcular cambio
    if previous_value != 0:
        change = ((current_value - previous_value) / previous_value) * 100
    else:
        change = 0

    # Determinar color según el cambio
    if change > 0:
        color = COLORS["success"]
    elif change < 0:
        color = COLORS["danger"]
    else:
        color = COLORS["secondary"]

    # Formatear valores
    if format_type == "currency":
        current_text = format_currency(current_value)
        previous_text = format_currency(previous_value)
    elif format_type == "percentage":
        current_text = format_percentage(current_value)
        previous_text = format_percentage(previous_value)
    else:
        current_text = str(int(current_value))
        previous_text = str(int(previous_value))

    # Crear gráfico de indicador
    fig = go.Figure(
        go.Indicator(
            mode="number+delta",
            value=current_value,
            delta={
                "reference": previous_value,
                "valueformat": ".1f" if format_type == "percentage" else ".0f",
                "relative": True,
                "position": "bottom",
            },
            title={"text": title},
            number={"valueformat": ",.0f" if format_type == "number" else None, "font": {"size": 32}},
            domain={"x": [0, 1], "y": [0, 1]},
        )
    )

    # Aplicar tema para KPI
    theme = CHART_THEMES["kpi_card"].copy()
    theme["height"] = height
    fig.update_layout(**theme)

    return dcc.Graph(figure=fig, config={"displayModeBar": False}, className="metric-chart")


def _apply_chart_theme(fig: go.Figure, height: int = 400):
    """
    Aplicar tema xfarma a un gráfico.

    Args:
        fig: Figura de Plotly
        height: Altura del gráfico
    """
    theme = CHART_THEMES["default"].copy()
    theme["height"] = height

    fig.update_layout(**theme)

    # Configurar grilla
    fig.update_xaxes(showgrid=True, gridwidth=1, gridcolor=theme["grid_color"])
    fig.update_yaxes(showgrid=True, gridwidth=1, gridcolor=theme["grid_color"])


def _create_empty_chart(title: str, height: int = 400) -> dcc.Graph:
    """
    Crear gráfico vacío cuando no hay datos.

    Args:
        title: Título del gráfico
        height: Altura del gráfico

    Returns:
        Componente dcc.Graph vacío
    """
    fig = go.Figure()
    fig.add_annotation(
        text="No hay datos disponibles",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        xanchor="center",
        yanchor="middle",
        font=dict(size=16, color=COLORS["secondary"]),
    )

    theme = CHART_THEMES["default"].copy()
    theme["height"] = height
    theme["title"] = title
    fig.update_layout(**theme)

    return dcc.Graph(figure=fig, config={"displayModeBar": False}, className="chart-container empty-chart")
