"""
Catalog Sync Progress Component - Issue #283 Phase 3.

Componente contextual de progreso de sincronización CIMA/Nomenclátor.
Reemplaza la barra de progreso inconsistente del sidebar con una versión
contextual que SOLO aparece durante sincronización activa en el panel de administración.

Responsabilidad:
- Mostrar progreso detallado durante sincronización
- Ocultar automáticamente cuando no hay sync activa
- Información contextual: fase, items procesados, tiempo estimado
"""

import logging

import dash_bootstrap_components as dbc
from dash import html
from utils.helpers import format_number

logger = logging.getLogger(__name__)

# Mapeo de tipos de sincronización a labels español
SYNC_TYPE_LABELS = {
    "cima": "CIMA",
    "nomenclator": "Nomenclátor",
    "all": "Completo (CIMA + Nomenclátor)",
    "unknown": "Desconocido",
}

# Mapeo de fases a labels descriptivos
SYNC_PHASE_LABELS = {
    "initializing": "Fase 1/3: Inicializando conexión",
    "downloading": "Fase 2/3: Descargando datos",
    "processing": "Fase 3/3: Procesando y almacenando",
    "unknown": "En progreso...",
}


def create_catalog_sync_progress_card(sync_info=None):
    """
    Crea una tarjeta contextual de progreso de sincronización (Issue #283 - Phase 3).

    Args:
        sync_info: Dict con información de la sincronización
            {
                'type': 'cima'|'nomenclator'|'all',
                'progress_pct': 0-100,
                'current_items': int,
                'total_items': int,
                'current_phase': 'initializing'|'downloading'|'processing',
                'estimated_time_remaining': int (seconds)
            }

    Returns:
        dbc.Card: Tarjeta de progreso contextual o div vacío si no hay sync

    Comportamiento:
        - Solo visible durante sincronización activa
        - Se oculta automáticamente al completar
        - Información detallada: fase, items, tiempo estimado
        - Diseño consistente con Design System
    """
    # Si no hay información de sync, retornar div vacío (componente oculto)
    if not sync_info:
        return html.Div(id="catalog-sync-progress-card", style={"display": "none"})

    # Extraer información de la sincronización
    sync_type = sync_info.get("type", "unknown")
    progress_pct = sync_info.get("progress_pct", 0)
    current_items = sync_info.get("current_items", 0)
    total_items = sync_info.get("total_items", 0)
    current_phase = sync_info.get("current_phase", "unknown")
    estimated_time = sync_info.get("estimated_time_remaining", 0)

    # Formatear labels
    sync_label = SYNC_TYPE_LABELS.get(sync_type.lower(), SYNC_TYPE_LABELS["unknown"])
    phase_label = SYNC_PHASE_LABELS.get(current_phase.lower(), SYNC_PHASE_LABELS["unknown"])

    # Formatear números con separador de miles
    current_formatted = format_number(current_items)
    total_formatted = format_number(total_items)

    # Calcular tiempo estimado en formato legible
    time_estimate_text = _format_time_estimate(estimated_time)

    # Determinar color de la progress bar según progreso
    progress_color = _get_progress_color(progress_pct)

    # Crear tarjeta de progreso contextual
    return dbc.Card(
        [
            dbc.CardBody(
                [
                    # Header con icono animado
                    html.Div(
                        [
                            html.I(className="fas fa-sync fa-spin me-2", style={"color": "#0d6efd"}),
                            html.Strong(f"Sincronización {sync_label} en Progreso", className="text-primary"),
                        ],
                        className="mb-3",
                    ),
                    # Progress bar animada
                    dbc.Progress(
                        value=progress_pct,
                        label=f"{progress_pct:.1f}%",
                        color=progress_color,
                        animated=True,
                        striped=True,
                        className="mb-3",
                        style={"height": "25px", "fontSize": "14px"},
                    ),
                    # Información detallada
                    html.Div(
                        [
                            # Fase actual
                            html.Div(
                                [
                                    html.I(className="fas fa-tasks me-2 text-info"),
                                    html.Span(phase_label, className="fw-bold"),
                                ],
                                className="mb-2",
                            ),
                            # Items procesados
                            html.Div(
                                [
                                    html.I(className="fas fa-database me-2 text-success"),
                                    html.Span(
                                        f"{current_formatted} / {total_formatted} registros procesados",
                                        className="text-muted",
                                    ),
                                ],
                                className="mb-2",
                            ),
                            # Tiempo estimado
                            (
                                html.Div(
                                    [
                                        html.I(className="fas fa-clock me-2 text-warning"),
                                        html.Span(f"Tiempo estimado: {time_estimate_text}", className="text-muted"),
                                    ]
                                )
                                if estimated_time > 0
                                else html.Div()
                            ),
                        ],
                        className="small",
                    ),
                ]
            )
        ],
        id="catalog-sync-progress-card",
        className="shadow-sm border-start border-primary border-4 mb-4",
        style={"display": "block"},  # Visible cuando hay sync activa
    )


def _format_time_estimate(seconds):
    """
    Formatea segundos a formato legible (min, horas).

    Args:
        seconds: int - Segundos estimados

    Returns:
        str: Tiempo formateado ("2 min", "1h 15min", etc.)
    """
    if seconds <= 0:
        return "calculando..."

    if seconds < 60:
        return f"{int(seconds)} seg"
    elif seconds < 3600:
        minutes = int(seconds / 60)
        return f"{minutes} min"
    else:
        hours = int(seconds / 3600)
        remaining_minutes = int((seconds % 3600) / 60)
        if remaining_minutes > 0:
            return f"{hours}h {remaining_minutes}min"
        return f"{hours}h"


def _get_progress_color(progress_pct):
    """
    Determina el color de la progress bar según el porcentaje.

    Args:
        progress_pct: float - Porcentaje de progreso (0-100)

    Returns:
        str: Color Bootstrap ('info', 'primary', 'success')
    """
    if progress_pct < 33:
        return "info"  # Azul claro para inicio
    elif progress_pct < 90:
        return "primary"  # Azul para progreso medio
    else:
        return "success"  # Verde para casi completado


def create_sync_completion_alert(sync_result):
    """
    Crea una alerta de finalización de sincronización (éxito o error).

    Args:
        sync_result: Dict con resultado de la sincronización
            {
                'success': bool,
                'message': str,
                'items_processed': int,
                'duration_seconds': int
            }

    Returns:
        dbc.Alert: Alerta de resultado de sincronización
    """
    if not sync_result:
        return html.Div()

    success = sync_result.get("success", False)
    message = sync_result.get("message", "Sincronización finalizada")
    items_processed = sync_result.get("items_processed", 0)
    duration = sync_result.get("duration_seconds", 0)

    # Formatear duración
    duration_text = _format_time_estimate(duration)

    # Formatear items procesados
    items_formatted = format_number(items_processed)

    if success:
        return dbc.Alert(
            [
                html.I(className="fas fa-check-circle me-2"),
                html.Strong("Sincronización Completada", className="me-2"),
                html.Br(),
                html.Small(
                    [f"✓ {items_formatted} registros procesados en {duration_text}", html.Br(), f"✓ {message}"],
                    className="text-muted",
                ),
            ],
            color="success",
            dismissable=True,
            className="mb-3",
        )
    else:
        return dbc.Alert(
            [
                html.I(className="fas fa-exclamation-triangle me-2"),
                html.Strong("Error en Sincronización", className="me-2"),
                html.Br(),
                html.Small(
                    [f"✗ {message}", html.Br(), "Consultar logs del sistema para más detalles"], className="text-muted"
                ),
            ],
            color="danger",
            dismissable=True,
            className="mb-3",
        )
