"""
Sistema de seguimiento de progreso para operaciones del catálogo.
Adaptado del sistema de progreso de enriquecimiento para operaciones de sincronización.
"""

from datetime import datetime
from typing import Dict, Optional

import dash_bootstrap_components as dbc
from dash import dcc, html


class CatalogProgressTracker:
    """Seguimiento de progreso para operaciones del catálogo."""

    PHASES = {
        "checking": {
            "name": "Verificando cambios",
            "icon": "fas fa-search",
            "color": "info",
            "description": "Comparando con fuentes externas...",
        },
        "downloading_cima": {
            "name": "Descargando CIMA",
            "icon": "fas fa-download",
            "color": "primary",
            "description": "Obteniendo datos de medicamentos...",
        },
        "processing_nomenclator": {
            "name": "Procesando Nomenclátor",
            "icon": "fas fa-file-medical",
            "color": "warning",
            "description": "Actualizando precios oficiales...",
        },
        "updating_database": {
            "name": "Actualizando BD",
            "icon": "fas fa-database",
            "color": "success",
            "description": "Guardando cambios...",
        },
        "completed": {
            "name": "Completado",
            "icon": "fas fa-check-circle",
            "color": "success",
            "description": "Catálogo actualizado correctamente",
        },
        "error": {
            "name": "Error",
            "icon": "fas fa-times-circle",
            "color": "danger",
            "description": "Ha ocurrido un error",
        },
        "error_recoverable": {
            "name": "Error (recuperable)",
            "icon": "fas fa-exclamation-triangle",
            "color": "warning",
            "description": "Error temporal - reintentando automáticamente",
        },
        "error_critical": {
            "name": "Error crítico",
            "icon": "fas fa-times-circle",
            "color": "danger",
            "description": "Error que requiere intervención manual",
        },
    }


def create_catalog_progress_component(
    progress_data: Optional[Dict] = None, show_details: bool = True, error_data: Optional[Dict] = None
) -> html.Div:
    """
    Crea el componente de progreso para operaciones del catálogo.

    Args:
        progress_data: Datos del progreso con estructura:
            {
                "current_phase": "downloading_cima",
                "progress": 45,
                "message": "Descargando página 5 de 10...",
                "start_time": "2025-01-13T10:00:00",
                "phases_completed": ["checking"],
                "stats": {
                    "products_checked": 15000,
                    "products_updated": 250,
                    "products_added": 10,
                    "products_removed": 2
                }
            }
        show_details: Si mostrar detalles adicionales
        error_data: Datos de errores del endpoint /api/v1/system/sync/real-progress
    """

    if not progress_data:
        return html.Div(dbc.Alert("No hay operaciones de catálogo en progreso", color="info", className="text-center"))

    current_phase = progress_data.get("current_phase", "checking")

    # Verificar si hay errores y ajustar fase
    if error_data and error_data.get("last_error"):
        if error_data.get("recoverable", True):
            current_phase = "error_recoverable"
        else:
            current_phase = "error_critical"

    phase_config = CatalogProgressTracker.PHASES.get(current_phase, CatalogProgressTracker.PHASES["checking"])
    progress_value = progress_data.get("progress", 0)

    # Mensaje actualizado con información de error si existe
    if error_data and error_data.get("last_error"):
        user_message = error_data.get("user_message", error_data.get("last_error", ""))
        error_count = error_data.get("error_count", 1)
        if error_count > 1:
            message = f"{user_message} (Error #{error_count})"
        else:
            message = user_message
    else:
        message = progress_data.get("message", phase_config["description"])

    # Calcular tiempo transcurrido
    elapsed_time = "00:00"
    if progress_data.get("start_time"):
        try:
            start = datetime.fromisoformat(progress_data["start_time"])
            elapsed = datetime.now() - start
            minutes, seconds = divmod(int(elapsed.total_seconds()), 60)
            elapsed_time = f"{minutes:02d}:{seconds:02d}"
        except:
            pass

    # Crear indicadores de fases
    phase_indicators = []
    phases_order = ["checking", "downloading_cima", "processing_nomenclator", "updating_database"]
    phases_completed = progress_data.get("phases_completed", [])

    for i, phase_key in enumerate(phases_order):
        phase = CatalogProgressTracker.PHASES[phase_key]

        # Determinar estado de la fase
        if phase_key in phases_completed:
            phase_color = "success"
            phase_icon = "fas fa-check-circle"
            phase_class = "text-success"
        elif phase_key == current_phase:
            phase_color = phase["color"]
            phase_icon = f"{phase['icon']} fa-pulse"
            phase_class = f"text-{phase_color}"
        else:
            phase_color = "secondary"
            phase_icon = phase["icon"]
            phase_class = "text-muted"

        phase_indicators.append(
            html.Div(
                [
                    html.Div(
                        [
                            html.I(className=f"{phase_icon} {phase_class}", style={"fontSize": "1.5rem"}),
                            html.Small(
                                phase["name"], className=f"d-block mt-1 {phase_class}", style={"fontSize": "0.7rem"}
                            ),
                        ],
                        className="text-center",
                    )
                ],
                className="col-3",
            )
        )

    # Estadísticas si están disponibles
    stats_cards = []
    if progress_data.get("stats"):
        stats = progress_data["stats"]

        if stats.get("products_checked") is not None:
            stats_cards.append(
                dbc.Col(
                    [
                        html.Div(
                            [
                                html.H5(f"{stats['products_checked']:,}", className="mb-0 text-info"),
                                html.Small("Verificados", className="text-muted"),
                            ],
                            className="text-center",
                        )
                    ],
                    width=6,
                    md=3,
                )
            )

        if stats.get("products_updated") is not None:
            stats_cards.append(
                dbc.Col(
                    [
                        html.Div(
                            [
                                html.H5(f"{stats['products_updated']:,}", className="mb-0 text-warning"),
                                html.Small("Actualizados", className="text-muted"),
                            ],
                            className="text-center",
                        )
                    ],
                    width=6,
                    md=3,
                )
            )

        if stats.get("products_added") is not None:
            stats_cards.append(
                dbc.Col(
                    [
                        html.Div(
                            [
                                html.H5(f"{stats['products_added']:,}", className="mb-0 text-success"),
                                html.Small("Añadidos", className="text-muted"),
                            ],
                            className="text-center",
                        )
                    ],
                    width=6,
                    md=3,
                )
            )

        if stats.get("products_removed") is not None:
            stats_cards.append(
                dbc.Col(
                    [
                        html.Div(
                            [
                                html.H5(f"{stats['products_removed']:,}", className="mb-0 text-danger"),
                                html.Small("Eliminados", className="text-muted"),
                            ],
                            className="text-center",
                        )
                    ],
                    width=6,
                    md=3,
                )
            )

    return html.Div(
        [
            # Header con icono y título
            html.Div(
                [
                    html.I(
                        className=f"{phase_config['icon']} text-{phase_config['color']} me-2",
                        style={"fontSize": "1.5rem"},
                    ),
                    html.H5("Actualización del Catálogo", className="mb-0 d-inline"),
                    html.Small(f" - {elapsed_time}", className="text-muted ms-2"),
                ],
                className="mb-3",
            ),
            # Indicadores de fases
            dbc.Row(phase_indicators, className="mb-3"),
            # Barra de progreso principal
            html.Div(
                [
                    dbc.Progress(
                        value=progress_value,
                        label=f"{progress_value}%",
                        color=phase_config["color"],
                        striped=True,
                        animated=True,
                        className="mb-2",
                        style={"height": "25px"},
                    ),
                    html.P(message, className="text-center text-muted small mb-0"),
                ]
            ),
            # Alerta de error si existe
            (
                dbc.Alert(
                    [
                        html.Div(
                            [
                                html.I(className=f"{phase_config['icon']} me-2"),
                                html.Strong("Estado del error:"),
                                html.Br(),
                                html.Span(error_data.get("user_message", error_data.get("last_error", ""))),
                            ]
                        ),
                        html.Hr(className="my-2"),
                        html.Div(
                            [
                                html.Small(
                                    [
                                        html.Strong(f"Errores detectados: {error_data.get('error_count', 1)}"),
                                        html.Br(),
                                        f"Tipo: {'Recuperable automáticamente' if error_data.get('recoverable', True) else 'Requiere intervención'}",
                                        html.Br(),
                                        f"Acciones sugeridas: {', '.join(error_data.get('suggested_actions', ['Contactar soporte']))}",
                                    ],
                                    className="text-muted",
                                )
                            ]
                        ),
                        # Botón para ver detalles técnicos
                        (
                            html.Div(
                                [
                                    dbc.Button(
                                        html.Div([html.I(className="fas fa-code me-2"), "Ver detalles técnicos"]),
                                        id="progress-error-details-btn",
                                        color="light",
                                        outline=True,
                                        size="sm",
                                        className="mt-2",
                                    ),
                                    dbc.Collapse(
                                        [
                                            html.Hr(),
                                            html.Code(
                                                error_data.get("technical_details", error_data.get("last_error", "")),
                                                className="d-block bg-dark text-light p-2 rounded small",
                                            ),
                                        ],
                                        id="progress-error-details-collapse",
                                    ),
                                ]
                            )
                            if error_data.get("technical_details")
                            else None
                        ),
                    ],
                    color=phase_config["color"],
                    className="mt-3",
                )
                if error_data and error_data.get("last_error")
                else None
            ),
            # Estadísticas si están disponibles y se solicitan detalles
            (
                html.Div(
                    [
                        html.Hr(className="my-3"),
                        html.H6("Resumen de cambios", className="text-muted mb-3"),
                        dbc.Row(stats_cards),
                    ]
                )
                if show_details and stats_cards
                else None
            ),
            # NOTA: catalog-progress-store está en app.py skeleton global (REGLA #0.5)
            # NO crear aquí - causa error DASH002 (Duplicate ID)
        ]
    )


def create_catalog_progress_modal() -> dbc.Modal:
    """
    Crea un modal para mostrar el progreso de actualización del catálogo.
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(
                    [
                        html.I(className="fas fa-sync-alt fa-spin me-2 text-primary"),
                        html.Span("Actualizando Catálogo de Productos"),
                    ],
                    className="d-flex align-items-center",
                ),
                close_button=False,  # No permitir cerrar durante actualización
            ),
            dbc.ModalBody(
                [
                    html.Div(id="catalog-progress-content"),
                    # Alertas dinámicas
                    html.Div(id="catalog-progress-alerts", className="mt-3"),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Ejecutar en segundo plano",
                        id="catalog-progress-background-btn",
                        color="secondary",
                        outline=True,
                        className="me-2",
                    ),
                    dbc.Button(
                        html.Div([html.I(className="fas fa-stop me-2"), "Cancelar"]),
                        id="catalog-progress-cancel-btn",
                        color="danger",
                        outline=True,
                    ),
                ]
            ),
        ],
        id="catalog-progress-modal",
        size="lg",
        centered=True,
        backdrop="static",
        keyboard=False,
        is_open=False,
    )


def create_catalog_progress_mini(progress_data: Optional[Dict] = None) -> html.Div:
    """
    Versión compacta del indicador de progreso para el sidebar.
    """
    if not progress_data or not progress_data.get("active"):
        return html.Div()

    current_phase = progress_data.get("current_phase", "checking")
    phase_config = CatalogProgressTracker.PHASES.get(current_phase)
    progress_value = progress_data.get("progress", 0)

    return html.Div(
        [
            html.Div(
                [
                    html.I(
                        className=f"{phase_config['icon']} fa-pulse me-2 text-{phase_config['color']}",
                        style={"fontSize": "0.9rem"},
                    ),
                    html.Small(phase_config["name"], className="fw-bold"),
                ],
                className="d-flex align-items-center mb-1",
            ),
            dbc.Progress(
                value=progress_value,
                color=phase_config["color"],
                striped=True,
                animated=True,
                style={"height": "6px"},
                className="mb-1",
            ),
            html.Small(f"{progress_value}%", className="text-muted d-block text-center", style={"fontSize": "0.7rem"}),
        ],
        className="px-3 py-2",
        style={
            "backgroundColor": f"var(--bs-{phase_config['color']}-bg-subtle)",
            "borderRadius": "8px",
            "borderLeft": f"3px solid var(--bs-{phase_config['color']})",
        },
    )


# Funciones helper para gestionar el progreso
def start_catalog_update() -> Dict:
    """Inicia una actualización del catálogo."""
    return {
        "current_phase": "checking",
        "progress": 0,
        "message": "Iniciando verificación de cambios...",
        "start_time": datetime.now().isoformat(),
        "phases_completed": [],
        "active": True,
        "stats": {"products_checked": 0, "products_updated": 0, "products_added": 0, "products_removed": 0},
    }


def update_catalog_progress(
    current_data: Dict,
    phase: Optional[str] = None,
    progress: Optional[int] = None,
    message: Optional[str] = None,
    stats_update: Optional[Dict] = None,
    complete_current_phase: bool = False,
) -> Dict:
    """
    Actualiza los datos de progreso del catálogo.

    Args:
        current_data: Datos actuales del progreso
        phase: Nueva fase (si cambia)
        progress: Nuevo valor de progreso (0-100)
        message: Nuevo mensaje
        stats_update: Actualización de estadísticas
        complete_current_phase: Si marcar la fase actual como completada
    """
    updated_data = current_data.copy()

    if complete_current_phase and current_data.get("current_phase"):
        if "phases_completed" not in updated_data:
            updated_data["phases_completed"] = []
        if current_data["current_phase"] not in updated_data["phases_completed"]:
            updated_data["phases_completed"].append(current_data["current_phase"])

    if phase:
        updated_data["current_phase"] = phase

    if progress is not None:
        updated_data["progress"] = min(100, max(0, progress))

    if message:
        updated_data["message"] = message

    if stats_update:
        if "stats" not in updated_data:
            updated_data["stats"] = {}
        updated_data["stats"].update(stats_update)

    return updated_data
