"""
Integración de componentes UX del catálogo en la aplicación principal.
Este módulo facilita la integración de todos los nuevos componentes.
"""

from typing import Dict, Optional

import dash_bootstrap_components as dbc
from components.catalog_control import create_catalog_control_panel, create_catalog_mini_card
from components.catalog_progress import create_catalog_progress_modal
from components.confirmation_modal import (
    create_action_confirmation_modal,
    create_catalog_update_modal,
    register_confirmation_callbacks,
)
from components.toast_manager import create_toast_container
from dash import dcc, html


def add_catalog_ux_components(app_layout: html.Div) -> html.Div:
    """
    Añade todos los componentes UX del catálogo al layout principal.

    Args:
        app_layout: Layout existente de la aplicación

    Returns:
        Layout actualizado con los nuevos componentes
    """

    # Crear contenedor principal si no existe
    if not isinstance(app_layout, html.Div):
        app_layout = html.Div(app_layout)

    # Obtener children actual o crear lista vacía
    children = list(app_layout.children) if app_layout.children else []

    # Añadir componentes globales al final del layout
    catalog_components = [
        # Sistema de toasts
        create_toast_container(),
        # Modales de confirmación
        create_catalog_update_modal(),
        create_action_confirmation_modal(),
        create_catalog_progress_modal(),
        # Stores para gestión de estado
        dcc.Store(id="catalog-info-store", storage_type="session"),
        # NOTA: catalog-progress-store está en app.py skeleton global (REGLA #0.5)
        dcc.Store(id="catalog-sync-complete-store"),
        dcc.Store(id="auto-update-trigger"),
        dcc.Store(id="action-modal-data-store"),
        # NOTA: dashboard-data-store está en app.py skeleton global (REGLA #0.5)
        # Componentes para manejo transparente de errores
        html.Div(id="catalog-error-display"),  # Display inline de errores
        html.Div(id="catalog-sync-status-indicator"),  # Indicador de estado en sidebar
        dbc.Modal(id="catalog-error-modal", is_open=False),  # Modal de error detallado
        # NOTA: dashboard-update-interval está en app.py skeleton global (REGLA #0.5)
        # NUEVO: Store unificado para el panel de control del catálogo
        # Evita duplicación de outputs siguiendo mejores prácticas
        dcc.Store(
            id="catalog-panel-unified-store",
            data={
                "source": "initial",  # "dashboard", "interval", "init"
                "catalog_info": {
                    "status": "CARGANDO",
                    "total_products": 0,  # Se actualizará en 500ms
                    "with_cima": 0,
                    "with_nomenclator": 0,
                    "last_update": None,
                    "message": "Conectando con el servidor...",
                },
                "dashboard_data": {},
                "catalog_status": {},
                "last_update": None,
            },
        ),
        # Intervals para actualizaciones periódicas
        # Interval inteligente: activo solo durante sincronización
        dcc.Interval(
            id="catalog-info-interval",
            interval=5000,  # 5 segundos durante sincronización
            n_intervals=0,
            max_intervals=-1,  # Sin límite
            disabled=True,  # Deshabilitado por defecto - se activa solo durante sync
        ),
        # NOTA: catalog-initial-load-interval está en app.py skeleton global (REGLA #0.5)
        dcc.Interval(id="auto-update-check-interval", interval=3600000, n_intervals=0),  # 1 hora
        # Interval para actualización durante sincronización
        dcc.Interval(
            id="catalog-update-interval",
            interval=15000,  # 15 segundos (reducido de 5s para evitar sobrecarga)
            n_intervals=0,
        ),
        # Los estilos CSS ahora están en assets/catalog-ux.css
    ]

    # Añadir componentes al layout
    children.extend(catalog_components)
    app_layout.children = children

    return app_layout


def register_catalog_callbacks(app):
    """
    Registra todos los callbacks relacionados con el catálogo.

    Args:
        app: Instancia de la aplicación Dash
    """
    # Registrar callbacks de componentes
    # Nota: register_toast_callbacks ahora se registra en callbacks/__init__.py (Issue #137)
    register_confirmation_callbacks(app)

    # ============================================================================
    # CRÍTICO (Issue #145): NO importar módulos con @callback aquí
    # ============================================================================
    #
    # PROBLEMA IDENTIFICADO:
    # Los módulos callbacks.catalog_callbacks y callbacks.catalog_error_handling
    # usan el decorador @callback, lo que significa que Dash los registra
    # AUTOMÁTICAMENTE en el momento de la importación.
    #
    # CAUSA DEL ERROR:
    # 1. app.py importa este módulo (catalog_integration) al inicio
    # 2. app.py llama register_catalog_callbacks(app) en línea 219
    # 3. Si aquí importamos callbacks.catalog_callbacks, se ejecuta el módulo
    # 4. Los callbacks con @callback se registran automáticamente
    # 5. Luego register_all_callbacks() intenta registrarlos otra vez
    # 6. Resultado: DuplicateCallback "admin-sync-progress-container already assigned"
    #
    # SOLUCIÓN:
    # Los callbacks se registran automáticamente cuando Python importa el módulo
    # al inicio de app.py. Esta función solo debe registrar callbacks de
    # componentes que SÍ necesitan la instancia app explícitamente.
    #
    # Ver documentación completa: frontend/docs/CALLBACK_DUPLICATION_SOLUTION.md
    #
    # ANTES (causaba error):
    #   import callbacks.catalog_callbacks
    #   import callbacks.catalog_error_handling
    #
    # AHORA (correcto):
    #   [Sin importaciones - callbacks auto-registrados por @callback decorator]
    # ============================================================================


def get_catalog_status_for_sidebar() -> Dict:
    """
    Obtiene el estado actual del catálogo para mostrar en el sidebar.

    Returns:
        Diccionario con información del catálogo
    """
    # En producción, esto obtendría datos reales de la API
    # Por ahora retornamos datos de ejemplo
    return {"last_update": "2025-01-10T10:00:00", "status": "SINCRONIZADO", "is_syncing": False}


def create_catalog_dashboard_card(catalog_info: Optional[Dict] = None, show_full: bool = True) -> dbc.Card:
    """
    Crea la tarjeta del catálogo para el dashboard.

    Args:
        catalog_info: Información del catálogo
        show_full: Si mostrar versión completa o mini

    Returns:
        Card del catálogo
    """
    if show_full:
        return create_catalog_control_panel(catalog_info)
    else:
        return create_catalog_mini_card(catalog_info)


# Los estilos CSS están en assets/catalog-ux.css
# Dash carga automáticamente todos los archivos CSS de la carpeta assets/


# Configuración de ejemplo para testing
DEMO_CATALOG_INFO = {
    "last_update": "2025-01-10T14:30:00",
    "total_products": 28543,
    "with_cima": 27890,
    "with_nomenclator": 14567,
    "status": "SINCRONIZADO",
    "is_syncing": False,
    "sync_history": [
        {
            "timestamp": "2025-01-10T14:30:00",
            "success": True,
            "duration": 45,
            "message": "Actualización incremental completada",
        },
        {"timestamp": "2025-01-03T10:15:00", "success": True, "duration": 120, "message": "Sincronización completa"},
        {"timestamp": "2024-12-28T09:00:00", "success": False, "duration": 15, "message": "Error de conexión con CIMA"},
    ],
}


def create_integration_example_page() -> html.Div:
    """
    Crea una página de ejemplo mostrando todos los componentes integrados.
    Útil para testing y demostración.
    """
    return html.Div(
        [
            # Header
            dbc.Container(
                [
                    html.H1("🎯 Componentes UX del Catálogo", className="mb-4"),
                    html.P(
                        "Demostración de todos los nuevos componentes de UX implementados para el Issue #15",
                        className="lead text-muted mb-4",
                    ),
                    # Panel de control del catálogo
                    html.H3("📊 Panel de Control del Catálogo", className="mb-3"),
                    create_catalog_control_panel(DEMO_CATALOG_INFO),
                    html.Hr(className="my-5"),
                    # Botones de demostración
                    html.H3("🔧 Acciones de Demostración", className="mb-3"),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dbc.Button(
                                        "Mostrar Toast Success",
                                        id="demo-toast-success",
                                        color="success",
                                        className="w-100 mb-2",
                                    ),
                                    dbc.Button(
                                        "Mostrar Toast Error",
                                        id="demo-toast-error",
                                        color="danger",
                                        className="w-100 mb-2",
                                    ),
                                    dbc.Button(
                                        "Mostrar Toast Warning",
                                        id="demo-toast-warning",
                                        color="warning",
                                        className="w-100 mb-2",
                                    ),
                                    dbc.Button(
                                        "Mostrar Toast Info", id="demo-toast-info", color="info", className="w-100"
                                    ),
                                ],
                                width=12,
                                md=6,
                            ),
                            dbc.Col(
                                [
                                    dbc.Button(
                                        "Abrir Modal Actualización",
                                        id="demo-modal-update",
                                        color="primary",
                                        className="w-100 mb-2",
                                    ),
                                    dbc.Button(
                                        "Simular Progreso",
                                        id="demo-progress",
                                        color="secondary",
                                        className="w-100 mb-2",
                                    ),
                                    dbc.Button(
                                        "Ver Mini Card", id="demo-mini-card", color="light", className="w-100 mb-2"
                                    ),
                                    dbc.Button(
                                        "Toggle Sincronización",
                                        id="demo-toggle-sync",
                                        color="warning",
                                        className="w-100",
                                    ),
                                ],
                                width=12,
                                md=6,
                            ),
                        ]
                    ),
                    html.Hr(className="my-5"),
                    # Versión mini
                    html.H3("📦 Versión Compacta", className="mb-3"),
                    dbc.Row([dbc.Col([create_catalog_mini_card(DEMO_CATALOG_INFO)], width=12, md=4)]),
                ],
                className="py-5",
            ),
            # Componentes globales
            # NOTA: create_toast_container() NO debe llamarse aquí
            # porque ya existe en el layout principal (add_catalog_ux_components)
            # Llamarlo aquí duplicaría el componente y causaría "Duplicate callback outputs"
            # create_toast_container(),  # COMENTADO: Evita duplicación de toast-trigger-store
            create_catalog_update_modal(),
            create_catalog_progress_modal(),
            # Stores de demostración
            dcc.Store(id="demo-catalog-state", data=DEMO_CATALOG_INFO),
        ]
    )


# Exportar todo lo necesario para la integración
__all__ = [
    "add_catalog_ux_components",
    "register_catalog_callbacks",
    "get_catalog_status_for_sidebar",
    "create_catalog_dashboard_card",
    "create_integration_example_page",
    "DEMO_CATALOG_INFO",
]
