"""
Panel de control del catálogo de productos.
Proporciona información del estado y control de actualización del catálogo.
"""

from datetime import datetime, timedelta
from typing import Dict, Optional

import dash_bootstrap_components as dbc
from dash import html
from utils.helpers import format_number


def create_catalog_control_panel(catalog_info: Optional[Dict] = None) -> dbc.Card:
    """
    Crea el panel de control del catálogo con estado y controles.

    Args:
        catalog_info: Información del catálogo con estructura:
            {
                "last_update": "2025-01-13T10:00:00",
                "total_products": 30000,
                "with_cima": 28500,
                "with_nomenclator": 15000,
                "status": "SINCRONIZADO",
                "is_syncing": False,
                "sync_history": [...]
            }
    """

    # Valores por defecto si no hay información
    if not catalog_info:
        catalog_info = {
            "status": "NO_INICIALIZADO",
            "total_products": 0,
            "with_cima": 0,
            "with_nomenclator": 0,
            "last_update": None,
        }

    # Determinar estado visual del catálogo
    status = catalog_info.get("status", "DESCONOCIDO")
    is_syncing = catalog_info.get("is_syncing", False)

    if is_syncing:
        status_color = "info"
        status_icon = "fas fa-sync-alt fa-spin"
        status_text = "Sincronizando..."
        status_bg = "bg-info bg-opacity-10 border-info border-2"
    elif status == "CARGANDO":
        status_color = "info"
        status_icon = "fas fa-spinner fa-spin"
        status_text = "Cargando..."
        status_bg = "bg-light"
    elif status == "SINCRONIZADO":
        status_color = "success"
        status_icon = "fas fa-check-circle"
        status_text = "Sincronizado"
        status_bg = "bg-success bg-opacity-10"
    elif status == "OPERATIVO":
        status_color = "success"
        status_icon = "fas fa-check-circle"
        status_text = "Operativo"
        status_bg = "bg-success bg-opacity-10"
    elif status == "DATOS_LIMITADOS":
        status_color = "warning"
        status_icon = "fas fa-exclamation-triangle"
        status_text = "Datos limitados"
        status_bg = "bg-warning bg-opacity-10"
    elif status == "DESACTUALIZADO":
        status_color = "warning"
        status_icon = "fas fa-exclamation-triangle"
        status_text = "Desactualizado"
        status_bg = "bg-warning bg-opacity-10"
    elif status == "ERROR":
        status_color = "danger"
        status_icon = "fas fa-times-circle"
        status_text = "Error"
        status_bg = "bg-danger bg-opacity-10"
    else:
        status_color = "secondary"
        status_icon = "fas fa-question-circle"
        status_text = "Sin inicializar"
        status_bg = "bg-secondary bg-opacity-10"

    # Calcular tiempo desde última actualización
    last_update_text = "No inicializado"
    days_since_update = None
    if catalog_info.get("last_update") and catalog_info.get("total_products", 0) > 0:
        try:
            # Manejar timezone correctamente
            from datetime import timezone as tz

            last_update_str = catalog_info["last_update"]

            # Si es string, parsearlo
            if isinstance(last_update_str, str):
                # Intentar parsear con diferentes formatos
                if "T" in last_update_str:
                    # Formato ISO
                    if last_update_str.endswith("Z"):
                        last_update_dt = datetime.fromisoformat(last_update_str.replace("Z", "+00:00"))
                    else:
                        last_update_dt = datetime.fromisoformat(last_update_str)
                else:
                    # Formato simple
                    last_update_dt = datetime.strptime(last_update_str, "%Y-%m-%d %H:%M:%S")
            else:
                last_update_dt = last_update_str

            # Asegurar que ambas fechas tengan timezone
            if last_update_dt.tzinfo is None:
                last_update_dt = last_update_dt.replace(tzinfo=tz.utc)

            now = datetime.now(tz.utc)

            days_since_update = (now - last_update_dt).days
            hours_since_update = (now - last_update_dt).total_seconds() / 3600

            if hours_since_update < 1:
                minutes = int((now - last_update_dt).total_seconds() / 60)
                last_update_text = f"Hace {minutes} minutos"
            elif hours_since_update < 24:
                last_update_text = f"Hace {int(hours_since_update)} horas"
            elif days_since_update == 1:
                last_update_text = "Ayer"
            elif days_since_update < 7:
                last_update_text = f"Hace {days_since_update} días"
            else:
                last_update_text = last_update_dt.strftime("%d/%m/%Y %H:%M")
        except Exception as e:
            # Log error for debugging but show friendly message
            import logging

            logging.error(f"Error parsing date: {catalog_info.get('last_update')}, error: {str(e)}")
            last_update_text = "No disponible"

    # Preparar historial de sincronizaciones
    sync_history_items = []
    if catalog_info.get("sync_history"):
        for sync in catalog_info["sync_history"][:5]:  # Últimas 5
            sync_time = datetime.fromisoformat(sync["timestamp"])
            sync_status_icon = "fas fa-check text-success" if sync["success"] else "fas fa-times text-danger"
            sync_history_items.append(
                html.Li(
                    [
                        html.I(className=f"{sync_status_icon} me-2", style={"fontSize": "0.8rem"}),
                        html.Small(sync_time.strftime("%d/%m %H:%M"), className="text-muted me-2"),
                        html.Small(f"({sync.get('duration', 'N/A')}s)", className="text-muted"),
                        (
                            html.Small(f" - {sync.get('message', '')}", className="text-muted ms-1")
                            if sync.get("message")
                            else None
                        ),
                    ],
                    className="mb-1",
                    style={"fontSize": "0.85rem"},
                )
            )

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.Div(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-database me-2 text-primary"),
                                    html.H5("Control del Catálogo", className="mb-0 d-inline"),
                                ]
                            ),
                            dbc.Badge(
                                html.Div([html.I(className=f"{status_icon} me-1"), status_text]),
                                color=status_color,
                                pill=True,
                                className="ms-auto",
                            ),
                        ],
                        className="d-flex justify-content-between align-items-center",
                    )
                ],
                className=status_bg,
            ),
            dbc.CardBody(
                [
                    # Estado general
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Div(
                                        [
                                            html.H6("Estado del Sistema", className="text-muted mb-3"),
                                            # Indicador visual grande con animación si está sincronizando
                                            html.Div(
                                                [
                                                    html.I(
                                                        className=f"{status_icon} text-{status_color}",
                                                        style={
                                                            "fontSize": "3rem",
                                                            "animation": "pulse 2s infinite" if is_syncing else "none",
                                                        },
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.H4(status_text, className=f"text-{status_color} mb-0"),
                                                            html.Small(last_update_text, className="text-muted"),
                                                            # Mostrar mensaje informativo si está disponible
                                                            (
                                                                html.Div(
                                                                    [
                                                                        html.Small(
                                                                            catalog_info.get("status_message", ""),
                                                                            className="text-muted fst-italic mt-1",
                                                                            style={
                                                                                "fontSize": "0.85rem",
                                                                                "lineHeight": "1.3",
                                                                            },
                                                                        )
                                                                    ],
                                                                    className="mt-2",
                                                                )
                                                                if catalog_info.get("status_message")
                                                                else None
                                                            ),
                                                        ],
                                                        className="ms-3",
                                                    ),
                                                ],
                                                className="d-flex align-items-center mb-4",
                                            ),
                                            # Botón de actualización con mejor feedback
                                            dbc.Button(
                                                html.Div(
                                                    [
                                                        html.I(
                                                            className=f"fas fa-{'spinner fa-spin' if is_syncing else 'sync-alt'} me-2"
                                                        ),
                                                        "Sincronizando..." if is_syncing else "Actualizar Todo",
                                                    ]
                                                ),
                                                id="catalog-update-button",
                                                color="info" if is_syncing else "primary",
                                                disabled=is_syncing,
                                                className="w-100 mb-2",
                                                size="lg",  # Botón más grande para mejor visibilidad
                                            ),
                                            # Botones individuales para CIMA y Nomenclátor
                                            dbc.Row(
                                                [
                                                    dbc.Col(
                                                        [
                                                            dbc.Button(
                                                                html.Div(
                                                                    [
                                                                        html.I(className="fas fa-pills me-2"),
                                                                        "Actualizar CIMA",
                                                                    ]
                                                                ),
                                                                id="update-cima-button",
                                                                color="outline-info",
                                                                disabled=is_syncing,
                                                                className="w-100",
                                                                size="sm",
                                                            )
                                                        ],
                                                        width=6,
                                                    ),
                                                    dbc.Col(
                                                        [
                                                            dbc.Button(
                                                                html.Div(
                                                                    [
                                                                        html.I(className="fas fa-file-medical me-2"),
                                                                        "Actualizar Nomenclátor",
                                                                    ]
                                                                ),
                                                                id="update-nomenclator-button",
                                                                color="outline-info",
                                                                disabled=is_syncing,
                                                                className="w-100",
                                                                size="sm",
                                                            )
                                                        ],
                                                        width=6,
                                                    ),
                                                ],
                                                className="mb-3",
                                            ),
                                            # Progress bar mejorado si está sincronizando
                                            (
                                                html.Div(
                                                    [
                                                        # Indicador de etapa actual
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-info-circle text-info me-2"),
                                                                html.Strong("Etapa actual: ", className="text-info"),
                                                                html.Span(
                                                                    catalog_info.get(
                                                                        "sync_message", "Iniciando proceso..."
                                                                    ),
                                                                    className="text-muted",
                                                                ),
                                                            ],
                                                            className="mb-2 small",
                                                        ),
                                                        # Barra de progreso con porcentaje
                                                        dbc.Progress(
                                                            value=catalog_info.get("sync_progress", 0),
                                                            label=f"{catalog_info.get('sync_progress', 0)}%",
                                                            striped=True,
                                                            animated=True,
                                                            color="info",
                                                            className="mb-2",
                                                            style={
                                                                "height": "30px",
                                                                "fontSize": "14px",
                                                                "fontWeight": "bold",
                                                            },
                                                        ),
                                                        # Estadísticas adicionales si están disponibles
                                                        html.Div(
                                                            [
                                                                html.Small(
                                                                    [
                                                                        html.I(className="fas fa-clock me-1"),
                                                                        "Tiempo estimado restante: ",
                                                                        html.Strong(
                                                                            catalog_info.get(
                                                                                "estimated_time", "calculando..."
                                                                            ),
                                                                            className="text-info",
                                                                        ),
                                                                    ],
                                                                    className="text-muted",
                                                                )
                                                            ],
                                                            className="text-center",
                                                        ),
                                                    ],
                                                    className="mb-3 p-3 bg-light rounded",
                                                )
                                                if is_syncing
                                                else None
                                            ),
                                            # Advertencia si está desactualizado
                                            (
                                                dbc.Alert(
                                                    html.Div(
                                                        [
                                                            html.I(className="fas fa-exclamation-triangle me-2"),
                                                            f"El catálogo tiene {days_since_update} días sin actualizar. ",
                                                            "Se recomienda actualizar para mantener la información al día.",
                                                        ]
                                                    ),
                                                    color="warning",
                                                    className="mb-0",
                                                )
                                                if days_since_update and days_since_update > 10
                                                else None
                                            ),
                                        ]
                                    )
                                ],
                                width=12,
                                lg=6,
                            ),
                            dbc.Col(
                                [
                                    html.Div(
                                        [
                                            html.H6("Estadísticas del Catálogo", className="text-muted mb-3"),
                                            # KPIs del catálogo
                                            html.Div(
                                                [
                                                    # Total productos
                                                    html.Div(
                                                        [
                                                            html.Div(
                                                                [
                                                                    html.I(
                                                                        className="fas fa-pills text-primary",
                                                                        style={"fontSize": "1.5rem"},
                                                                    ),
                                                                    html.Div(
                                                                        [
                                                                            html.H5(
                                                                                format_number(
                                                                                    catalog_info.get(
                                                                                        "total_products", 0
                                                                                    )
                                                                                ),
                                                                                className="mb-0 text-primary",
                                                                            ),
                                                                            html.Small(
                                                                                "Productos totales",
                                                                                className="text-muted",
                                                                            ),
                                                                        ],
                                                                        className="ms-2",
                                                                    ),
                                                                ],
                                                                className="d-flex align-items-center",
                                                            )
                                                        ],
                                                        className="p-3 bg-light rounded mb-2",
                                                    ),
                                                    # Con CIMA
                                                    html.Div(
                                                        [
                                                            html.Div(
                                                                [
                                                                    html.I(
                                                                        className="fas fa-check-double text-success",
                                                                        style={"fontSize": "1.5rem"},
                                                                    ),
                                                                    html.Div(
                                                                        [
                                                                            html.H5(
                                                                                format_number(
                                                                                    catalog_info.get("with_cima", 0)
                                                                                ),
                                                                                className="mb-0 text-success",
                                                                            ),
                                                                            html.Small(
                                                                                "Con datos CIMA", className="text-muted"
                                                                            ),
                                                                        ],
                                                                        className="ms-2",
                                                                    ),
                                                                ],
                                                                className="d-flex align-items-center",
                                                            )
                                                        ],
                                                        className="p-3 bg-light rounded mb-2",
                                                    ),
                                                    # Con Nomenclátor
                                                    html.Div(
                                                        [
                                                            html.Div(
                                                                [
                                                                    html.I(
                                                                        className="fas fa-file-medical text-info",
                                                                        style={"fontSize": "1.5rem"},
                                                                    ),
                                                                    html.Div(
                                                                        [
                                                                            html.H5(
                                                                                format_number(
                                                                                    catalog_info.get(
                                                                                        "with_nomenclator", 0
                                                                                    )
                                                                                ),
                                                                                className="mb-0 text-info",
                                                                            ),
                                                                            html.Small(
                                                                                "En nomenclátor", className="text-muted"
                                                                            ),
                                                                        ],
                                                                        className="ms-2",
                                                                    ),
                                                                ],
                                                                className="d-flex align-items-center",
                                                            )
                                                        ],
                                                        className="p-3 bg-light rounded",
                                                    ),
                                                ]
                                            ),
                                        ]
                                    )
                                ],
                                width=12,
                                lg=6,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Historial de sincronizaciones
                    (
                        html.Div(
                            [
                                html.Hr(),
                                html.H6("Historial de Sincronizaciones", className="text-muted mb-3"),
                                html.Ul(
                                    (
                                        sync_history_items
                                        if sync_history_items
                                        else [
                                            html.Li(
                                                html.Small(
                                                    "No hay sincronizaciones registradas",
                                                    className="text-muted fst-italic",
                                                )
                                            )
                                        ]
                                    ),
                                    className="list-unstyled mb-0",
                                ),
                            ]
                        )
                        if catalog_info.get("show_history", True)
                        else None
                    ),
                    # Información adicional
                    html.Div(
                        [
                            html.Hr(),
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-info-circle me-1 text-info"),
                                                    "Actualización automática: ",
                                                    html.Strong("Cada 10 días", className="text-info"),
                                                ],
                                                className="text-muted",
                                            )
                                        ],
                                        width=12,
                                        md=6,
                                    ),
                                    dbc.Col(
                                        [
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-clock me-1 text-muted"),
                                                    "Próxima actualización: ",
                                                    html.Strong(
                                                        (
                                                            (
                                                                datetime.fromisoformat(catalog_info["last_update"])
                                                                + timedelta(days=10)
                                                            ).strftime("%d/%m/%Y")
                                                            if catalog_info.get("last_update")
                                                            else "N/A"
                                                        ),
                                                        className="text-muted",
                                                    ),
                                                ],
                                                className="text-muted text-md-end",
                                            )
                                        ],
                                        width=12,
                                        md=6,
                                    ),
                                ]
                            ),
                        ],
                        className="mt-3",
                    ),
                ]
            ),
        ],
        className="shadow-sm h-100",
    )


def create_catalog_mini_card(catalog_info: Optional[Dict] = None) -> dbc.Card:
    """
    Versión compacta del panel de control para espacios reducidos.
    """
    if not catalog_info:
        catalog_info = {"total_products": 0}

    status = catalog_info.get("status", "DESCONOCIDO")
    is_syncing = catalog_info.get("is_syncing", False)

    if is_syncing:
        color = "info"
        icon = "fas fa-sync-alt fa-spin"
        text = "Actualizando..."
    elif status == "SINCRONIZADO":
        color = "success"
        icon = "fas fa-check-circle"
        text = format_number(catalog_info.get("total_products", 0))
    else:
        color = "warning"
        icon = "fas fa-exclamation-circle"
        text = "Actualizar"

    return dbc.Card(
        [
            dbc.CardBody(
                [
                    html.Div(
                        [
                            html.I(className=f"{icon} text-{color}", style={"fontSize": "2rem"}),
                            html.Div(
                                [
                                    html.H4(text, className=f"text-{color} mb-0"),
                                    html.Small("Catálogo", className="text-muted"),
                                ],
                                className="ms-3",
                            ),
                        ],
                        className="d-flex align-items-center",
                    ),
                    dbc.Button(
                        "Gestionar",
                        id="catalog-mini-manage-btn",
                        color=color,
                        outline=True,
                        size="sm",
                        className="mt-3 w-100",
                    ),
                ],
                className="text-center",
            )
        ],
        className="shadow-sm h-100",
    )
