"""
Title Component - Sistema de Diseño xFarma

Componente base para crear títulos consistentes con jerarquía automática.
Implementa design tokens de tipografía.

Uso:
    from components.base import Title

    # Título de nivel 1
    Title("Dashboard Principal", level=1)

    # Título de nivel 3 con color personalizado
    Title("Resumen de Ventas", level=3, color="primary")
"""

from dash import html
from styles.design_tokens import COLORS, SPACING, TYPOGRAPHY


def Title(children, level=2, color=None, className="", margin_bottom=None, **kwargs):
    """
    Componente base para títulos con jerarquía tipográfica.

    Args:
        children: Texto del título
        level: Nivel jerárquico del título (1-5)
            - 1: h1 - Títulos principales de página (2.5rem)
            - 2: h2 - Títulos de sección (2rem)
            - 3: h3 - Subtítulos importantes (1.75rem)
            - 4: h4 - Subtítulos secundarios (1.5rem)
            - 5: h5 - Títulos terciarios (1.25rem)
        color: Color del texto (usa nombres de tokens COLORS o None para default)
        className: Clases CSS adicionales
        margin_bottom: Margen inferior personalizado (usa SPACING por defecto)
        **kwargs: Props adicionales para el elemento HTML

    Returns:
        html.H1-H5: Elemento de título HTML configurado

    Ejemplos:
        # Título principal de página
        Title("Dashboard de Ventas", level=1)

        # Subtítulo de sección con color
        Title("Análisis Mensual", level=3, color="primary")

        # Título sin margen inferior
        Title("Filtros", level=4, margin_bottom="0")
    """
    # Mapeo de niveles a elementos HTML
    heading_components = {
        1: html.H1,
        2: html.H2,
        3: html.H3,
        4: html.H4,
        5: html.H5,
    }

    # Validar nivel
    if level not in heading_components:
        level = 2  # Default a h2 si el nivel es inválido

    # Obtener estilos de tipografía según el nivel
    typography_key = f"h{level}"
    typography = TYPOGRAPHY[typography_key]

    # Estilos base del título
    base_style = {
        "fontSize": typography["size"],
        "fontWeight": typography["weight"],
        "lineHeight": typography["line_height"],
        "marginBottom": margin_bottom if margin_bottom is not None else SPACING["m"],
        "marginTop": "0",
    }

    # Aplicar color si se especifica
    if color:
        # Si el color es un nombre de token, usar ese valor
        base_style["color"] = COLORS.get(color, color)
    else:
        # Color por defecto
        base_style["color"] = COLORS["text_primary"]

    # Combinar estilos personalizados
    final_style = {**base_style, **kwargs.pop("style", {})}

    # Crear el componente según el nivel
    HeadingComponent = heading_components[level]

    return HeadingComponent(children=children, className=f"{className}".strip(), style=final_style, **kwargs)
