"""
Base Card Component - Sistema de Diseño xFarma

Componente base para crear tarjetas consistentes en toda la aplicación.
Implementa design tokens para colores, sombras y spacing.

Uso:
    from components.base import BaseCard

    # Card básica
    BaseCard(children=[...])

    # Card destacada
    BaseCard(children=[...], variant="highlighted")

    # Card con sombra personalizada
    BaseCard(children=[...], shadow="lg")
"""

import dash_bootstrap_components as dbc
from dash import html
from styles.design_tokens import BORDER_RADIUS, COLORS, SHADOWS, SPACING


def BaseCard(
    children=None, variant="default", shadow="sm", padding=None, className="", role=None, aria_label=None, **kwargs
):
    """
    Componente base para tarjetas (cards).

    Args:
        children: Contenido de la tarjeta
        variant: Variante visual de la tarjeta
            - "default": Tarjeta estándar con fondo blanco
            - "highlighted": Tarjeta destacada con borde coloreado
        shadow: Tamaño de la sombra ("none", "sm", "md", "lg", "xl")
        padding: Padding personalizado (usa tokens SPACING si no se especifica)
        className: Clases CSS adicionales
        role: Rol ARIA para accesibilidad (ej: "region", "article", "complementary")
        aria_label: Etiqueta ARIA descriptiva para lectores de pantalla
        **kwargs: Props adicionales para dbc.Card

    Returns:
        dbc.Card: Componente de tarjeta configurado

    Ejemplos:
        # Tarjeta estándar
        BaseCard(
            children=[
                html.H4("Título"),
                html.P("Contenido")
            ]
        )

        # Tarjeta destacada con sombra grande
        BaseCard(
            children=[...],
            variant="highlighted",
            shadow="lg"
        )

        # Tarjeta con accesibilidad mejorada
        BaseCard(
            children=[...],
            role="region",
            aria_label="Resumen de ventas mensuales"
        )
    """
    # Estilos base de la tarjeta
    base_style = {
        "borderRadius": BORDER_RADIUS["md"],
        "boxShadow": SHADOWS.get(shadow, SHADOWS["sm"]),
        "padding": padding if padding else SPACING["m"],
        "backgroundColor": COLORS["bg_primary"],
        "border": f"1px solid {COLORS['border_light']}",
        "height": "100%",
    }

    # Aplicar variantes
    if variant == "highlighted":
        base_style.update(
            {
                "borderLeft": f"4px solid {COLORS['primary']}",
                "backgroundColor": COLORS["bg_tertiary"],
            }
        )

    # Combinar estilos personalizados
    final_style = {**base_style, **kwargs.pop("style", {})}

    # Crear card sin props ARIA (dbc.Card no los soporta)
    card = dbc.Card(children=children, className=f"{className}".strip(), style=final_style, **kwargs)

    # Si se especifican props ARIA, envolver en html.Div
    if role or aria_label:
        accessibility_props = {}
        if role:
            accessibility_props["role"] = role
        if aria_label:
            accessibility_props["aria-label"] = aria_label

        return html.Div(card, **accessibility_props)

    return card
