"""
Base Button Component - Sistema de Diseño xFarma

Componente base para crear botones consistentes en toda la aplicación.
Implementa design tokens para colores, tamaños y spacing.

Uso:
    from components.base import BaseButton

    # Botón primario
    BaseButton("Guardar", variant="primary")

    # Botón secundario pequeño
    BaseButton("Cancelar", variant="secondary", size="sm")

    # Botón peligroso grande
    BaseButton("Eliminar", variant="danger", size="lg")
"""

import dash_bootstrap_components as dbc
from styles.design_tokens import BORDER_RADIUS, SPACING, TRANSITIONS


def BaseButton(children, variant="primary", size="md", outline=False, disabled=False, className="", **kwargs):
    """
    Componente base para botones.

    Args:
        children: Texto o contenido del botón
        variant: Variante de color del botón
            - "primary": Botón principal (azul oscuro)
            - "secondary": Botón secundario (gris)
            - "success": Botón de éxito (verde)
            - "danger": Botón de peligro (rojo)
            - "warning": Botón de advertencia (naranja)
            - "info": Botón informativo (azul claro)
        size: Tamaño del botón ("sm", "md", "lg")
        outline: Si True, usa estilo outline en lugar de sólido
        disabled: Si True, deshabilita el botón
        className: Clases CSS adicionales
        **kwargs: Props adicionales para dbc.Button

    Returns:
        dbc.Button: Componente de botón configurado

    Ejemplos:
        # Botón primario estándar
        BaseButton("Guardar", variant="primary")

        # Botón outline secundario
        BaseButton("Cancelar", variant="secondary", outline=True)

        # Botón pequeño de peligro
        BaseButton("Eliminar", variant="danger", size="sm")
    """
    # Mapeo de variantes a colores Bootstrap
    color_map = {
        "primary": "primary",
        "secondary": "secondary",
        "success": "success",
        "danger": "danger",
        "warning": "warning",
        "info": "info",
    }

    # Mapeo de tamaños
    size_styles = {
        "sm": {
            "padding": f"{SPACING['xs']} {SPACING['s']}",
            "fontSize": "0.875rem",
        },
        "md": {
            "padding": f"{SPACING['s']} {SPACING['m']}",
            "fontSize": "1rem",
        },
        "lg": {
            "padding": f"{SPACING['m']} {SPACING['l']}",
            "fontSize": "1.125rem",
        },
    }

    # Estilos base del botón
    base_style = {
        "borderRadius": BORDER_RADIUS["md"],
        "transition": TRANSITIONS["fast"],
        "fontWeight": "400",
        **size_styles.get(size, size_styles["md"]),
    }

    # Agregar estilos de hover si no está deshabilitado
    if not disabled:
        base_style["cursor"] = "pointer"

    # Combinar estilos personalizados
    final_style = {**base_style, **kwargs.pop("style", {})}

    return dbc.Button(
        children=children,
        color=color_map.get(variant, "primary"),
        outline=outline,
        disabled=disabled,
        className=f"{className}".strip(),
        style=final_style,
        **kwargs,
    )
