"""
Componentes reutilizables para feedback UX honesto en el panel de administración.
Enfocado en administradores de farmacia con mensajes claros y orientación práctica.
"""

from typing import Any, List, Optional

import dash_bootstrap_components as dbc
from dash import html


def create_loading_button(
    button_id: str,
    default_text: str,
    loading_text: str,
    icon_class: str,
    color: str = "primary",
    outline: bool = True,
    size: str = "md",
    className: str = "",
) -> dbc.Button:
    """
    Crea un botón con estados de loading integrados.

    Args:
        button_id: ID único del botón
        default_text: Texto por defecto
        loading_text: Texto durante carga
        icon_class: Clase del icono FontAwesome
        color: Color del botón
        outline: Si usar estilo outline
        size: Tamaño (sm, md, lg)
        className: Clases CSS adicionales
    """
    return dbc.Button(
        [
            html.Span(html.I(className=f"{icon_class} me-2"), id=f"{button_id}-icon"),
            html.Span(default_text, id=f"{button_id}-text"),
        ],
        id=button_id,
        color=color,
        outline=outline,
        size=size,
        className=className,
        disabled=False,
    )


def create_metric_card_with_loading(
    title: str,
    metric_id: str,
    default_value: str = "0",
    loading_value: str = "⏳",
    error_value: str = "❌",
    tooltip_text: Optional[str] = None,
    color_class: str = "text-primary",
) -> dbc.Card:
    """
    Crea una tarjeta de métrica con estados de loading y error claros.

    Args:
        title: Título de la métrica
        metric_id: ID para actualizar el valor
        default_value: Valor por defecto
        loading_value: Valor durante carga
        error_value: Valor en caso de error
        tooltip_text: Texto explicativo en tooltip
        color_class: Clase CSS para el color
    """
    metric_content = [
        html.H3(default_value, id=metric_id, className=f"{color_class} mb-0"),
        html.Small(title, className="text-muted"),
    ]

    if tooltip_text:
        metric_content.insert(0, dbc.Tooltip(tooltip_text, target=metric_id, placement="top"))

    return dbc.Card([dbc.CardBody(metric_content)], className="text-center h-100")


def create_friendly_error_alert(
    error_type: str,
    technical_details: Optional[str] = None,
    suggested_actions: Optional[List[str]] = None,
    show_contact_info: bool = True,
) -> dbc.Alert:
    """
    Crea alertas de error amigables para administradores de farmacia.

    Args:
        error_type: Tipo de error ('connection', 'server', 'timeout', 'not_found')
        technical_details: Detalles técnicos opcionales
        suggested_actions: Lista de acciones sugeridas
        show_contact_info: Si mostrar info de contacto técnico
    """
    error_messages = {
        "connection": {
            "title": "Error de Conexión",
            "icon": "fas fa-wifi-slash",
            "message": "No se puede conectar al servidor en este momento.",
            "default_actions": [
                "Verificar conexión a internet",
                "Reintentar en unos minutos",
                "Verificar que el backend esté funcionando",
            ],
        },
        "server": {
            "title": "Error del Servidor",
            "icon": "fas fa-server",
            "message": "El servidor ha encontrado un error interno.",
            "default_actions": [
                "Esperar unos minutos antes de reintentar",
                "Si el problema persiste, contactar soporte técnico",
            ],
        },
        "timeout": {
            "title": "Tiempo de Espera Agotado",
            "icon": "fas fa-clock",
            "message": "La operación está tardando más de lo esperado.",
            "default_actions": [
                "La operación puede estar ejecutándose en segundo plano",
                "Esperar 5-10 minutos antes de reintentar",
                "Verificar el estado en el dashboard de salud",
            ],
        },
        "not_found": {
            "title": "Servicio No Disponible",
            "icon": "fas fa-search",
            "message": "El servicio solicitado no está disponible en este momento.",
            "default_actions": [
                "Verificar configuración del sistema",
                "El servicio puede estar en mantenimiento",
                "Contactar administrador técnico",
            ],
        },
        "unauthorized": {
            "title": "Acceso Denegado",
            "icon": "fas fa-lock",
            "message": "No tiene permisos para realizar esta operación.",
            "default_actions": ["Verificar que tiene rol de administrador", "Contactar administrador del sistema"],
        },
    }

    error_info = error_messages.get(error_type, error_messages["server"])
    actions = suggested_actions or error_info["default_actions"]

    alert_content = [
        html.Div([html.I(className=f"{error_info['icon']} me-2"), html.Strong(error_info["title"])], className="mb-2"),
        html.P(error_info["message"], className="mb-3"),
        html.Div(
            [html.Strong("¿Qué puede hacer?"), html.Ul([html.Li(action) for action in actions], className="mb-2")]
        ),
    ]

    if technical_details:
        alert_content.append(html.Hr())
        alert_content.append(
            html.Details(
                [
                    html.Summary("Detalles técnicos", className="text-muted small"),
                    html.Code(technical_details, className="small d-block mt-2 p-2 bg-light rounded"),
                ],
                className="mt-2",
            )
        )

    if show_contact_info:
        alert_content.append(html.Hr())
        alert_content.append(
            html.Small(
                [
                    html.I(className="fas fa-life-ring me-1"),
                    "Para soporte técnico: contacte al administrador del sistema",
                ],
                className="text-muted",
            )
        )

    return dbc.Alert(alert_content, color="danger", dismissable=True, className="border border-danger")


def create_sync_progress_indicator(operation_name: str, progress_id: str, show_progress: bool = True) -> html.Div:
    """
    Crea un indicador de progreso para operaciones de sincronización.

    Args:
        operation_name: Nombre de la operación
        progress_id: ID para actualizar el progreso
        show_progress: Si mostrar barra de progreso
    """
    components = [
        html.Div(
            [
                html.I(className="fas fa-sync fa-spin me-2 text-primary"),
                html.Strong(f"Sincronizando {operation_name}...", className="text-primary"),
            ],
            className="mb-2",
        )
    ]

    if show_progress:
        components.append(dbc.Progress(id=progress_id, value=0, striped=True, animated=True, className="mb-2"))

    components.extend(
        [
            html.Small(
                [
                    html.I(className="fas fa-info-circle me-1"),
                    "Esta operación puede tardar varios minutos. No cierre la ventana.",
                ],
                className="text-muted d-block",
            ),
            html.Small(
                [
                    html.I(className="fas fa-clock me-1"),
                    "Tiempo estimado: ",
                    html.Span("Calculando...", id=f"{progress_id}-eta"),
                ],
                className="text-muted d-block mt-1",
            ),
        ]
    )

    return html.Div(
        components,
        id=f"{progress_id}-container",
        className="border-start border-primary border-3 ps-3 py-2",
        style={"display": "none"},
    )


def create_status_badge_with_tooltip(status: str, tooltip_text: str, badge_id: str) -> html.Div:
    """
    Crea un badge de estado con tooltip explicativo.

    Args:
        status: Estado ('online', 'offline', 'warning', 'unknown')
        tooltip_text: Texto explicativo
        badge_id: ID del badge
    """
    status_config = {
        "online": {"text": "ONLINE", "color": "success", "icon": "fas fa-check-circle"},
        "offline": {"text": "OFFLINE", "color": "danger", "icon": "fas fa-times-circle"},
        "warning": {"text": "ADVERTENCIA", "color": "warning", "icon": "fas fa-exclamation-triangle"},
        "unknown": {"text": "DESCONOCIDO", "color": "secondary", "icon": "fas fa-question-circle"},
        "loading": {"text": "VERIFICANDO...", "color": "info", "icon": "fas fa-spinner fa-spin"},
    }

    config = status_config.get(status, status_config["unknown"])

    return html.Div(
        [
            dbc.Badge([html.I(className=f"{config['icon']} me-1"), config["text"]], id=badge_id, color=config["color"]),
            dbc.Tooltip(tooltip_text, target=badge_id, placement="bottom"),
        ]
    )


# create_operation_feedback_toast ELIMINADO - usar toast_manager en su lugar
# Ver frontend/components/toast_manager.py para notificaciones toast


def create_not_available_placeholder(
    feature_name: str, reason: str = "configuración adicional", coming_soon: bool = False
) -> dbc.Alert:
    """
    Crea un placeholder honesto para funcionalidades no disponibles.

    Args:
        feature_name: Nombre de la funcionalidad
        reason: Razón por la que no está disponible
        coming_soon: Si está planificado para pronto
    """
    icon = "fas fa-construction" if coming_soon else "fas fa-info-circle"
    color = "warning" if coming_soon else "info"

    message = f"La funcionalidad '{feature_name}' no está disponible."

    if coming_soon:
        reason_text = "Esta funcionalidad estará disponible próximamente."
    else:
        reason_text = f"Requiere {reason} del servidor."

    return dbc.Alert(
        [
            html.I(className=f"{icon} me-2"),
            html.Strong(message),
            html.Br(),
            html.Small(reason_text, className="text-muted"),
        ],
        color=color,
        className="text-center",
    )


# create_skeleton_loader ELIMINADO - usar skeleton_loader.create_skeleton_loader en su lugar
# Ver frontend/components/skeleton_loader.py para loaders de carga


# Funciones de estado para callbacks
def get_button_loading_state(is_loading: bool, default_text: str, loading_text: str, icon_class: str):
    """
    Retorna el estado de un botón durante loading.
    """
    if is_loading:
        return (
            [html.I(className="fas fa-spinner fa-spin me-2"), loading_text],
            True,  # disabled
            "outline-secondary",  # color
        )
    else:
        return ([html.I(className=f"{icon_class} me-2"), default_text], False, "outline-primary")  # disabled  # color


def get_metric_display_value(
    value: Any, is_loading: bool = False, has_error: bool = False, format_func: Optional[callable] = None
) -> str:
    """
    Formatea un valor de métrica según su estado.

    Args:
        value: Valor de la métrica
        is_loading: Si está cargando
        has_error: Si hay error
        format_func: Función para formatear el valor
    """
    if has_error:
        return "❌ No disponible"

    if is_loading:
        return "⏳ Cargando..."

    if value is None or value == "":
        return "➖ Sin datos"

    if format_func:
        try:
            return format_func(value)
        except:
            return str(value)

    return str(value)
