"""
Zona de peligro para operaciones administrativas críticas y destructivas.
Requiere múltiples confirmaciones y validaciones para prevenir accidentes.
"""

import logging
from datetime import datetime
from typing import Dict, Literal

logger = logging.getLogger(__name__)

import dash_bootstrap_components as dbc
from dash import ALL, Input, Output, State, ctx, dcc, html

from utils.request_coordinator import request_coordinator
from components.toast_manager import error_toast, success_toast, warning_toast

DangerousOperation = Literal[
    "vacuum_database",
    "force_catalog_rebuild",
    "clear_all_cache",
    "cleanup_zombie_uploads",
    # Herramientas de mantenimiento (Issue #452)
    "reenrich_sales",
    "clear_cache",
    "verify_connectivity",
]


class AdminDangerZone:
    """Gestor de operaciones peligrosas con múltiples niveles de protección."""

    # Configuración de operaciones peligrosas
    DANGEROUS_OPERATIONS = {
        "vacuum_database": {
            "title": "VACUUM Base de Datos",
            "description": "Reconstruye tablas eliminando espacio no utilizado y optimiza rendimiento",
            "risk_level": "high",
            "estimated_time": "5-30 minutos",
            "requires_text_confirmation": True,
            "confirmation_text": "VACUUM DATABASE",
            "impacts": [
                "Bloqueo temporal de escrituras durante proceso",
                "Alto uso de CPU y disco",
                "Posible timeout en operaciones largas",
            ],
            "prerequisites": [
                "Backup reciente disponible",
                "Ventana de mantenimiento programada",
                "Monitoreo de recursos activo",
            ],
            "color": "danger",
            "icon": "fas fa-database",
        },
        "force_catalog_rebuild": {
            "title": "Reconstruir Catálogo Completo",
            "description": "Elimina y recrea todo el catálogo de productos desde fuentes externas",
            "risk_level": "high",
            "estimated_time": "2-8 horas",
            "requires_text_confirmation": True,
            "confirmation_text": "REBUILD CATALOG",
            "impacts": [
                "Pérdida temporal de enriquecimientos",
                "Análisis bloqueados durante proceso",
                "Posible pérdida de datos no sincronizados",
            ],
            "prerequisites": [
                "APIs externas disponibles (CIMA, Nomenclátor)",
                "Espacio suficiente en disco",
                "Conexión estable a internet",
            ],
            "color": "danger",
            "icon": "fas fa-sync-alt",
        },
        "clear_all_cache": {
            "title": "Limpiar Todo el Cache",
            "description": "Elimina completamente todos los datos de cache (Redis + aplicación)",
            "risk_level": "medium",
            "estimated_time": "30 segundos - 2 minutos",
            "requires_text_confirmation": True,
            "confirmation_text": "CLEAR ALL CACHE",
            "impacts": [
                "Degradación temporal de rendimiento",
                "Recarga lenta de datos frecuentes",
                "Posible timeout en primeras consultas",
            ],
            "prerequisites": ["Cache no crítico para operaciones en curso"],
            "color": "warning",
            "icon": "fas fa-trash-alt",
        },
        "cleanup_zombie_uploads": {
            "title": "Limpiar Uploads Zombie",
            "description": "Cancela uploads congelados en procesamiento por más de 1 hora sin progreso",
            "risk_level": "low",
            "estimated_time": "10-30 segundos",
            "requires_text_confirmation": False,
            "impacts": [
                "Uploads congelados marcados como ERROR",
                "Libera recursos del sistema",
                "Permite reintentar uploads fallidos",
            ],
            "prerequisites": ["Verificar que no hay uploads legítimos en progreso"],
            "color": "warning",
            "icon": "fas fa-broom",
        },
        # === HERRAMIENTAS DE MANTENIMIENTO (Issue #452) ===
        "reenrich_sales": {
            "title": "Re-enriquecer Ventas",
            "description": "Busca ventas en manual_review que ahora tienen match en catálogo y las enriquece automáticamente",
            "risk_level": "low",
            "estimated_time": "30-120 segundos",
            "requires_text_confirmation": False,
            "impacts": [
                "Actualiza status de ventas de manual_review a enriched",
                "Puede tardar si hay muchos registros pendientes",
                "No afecta ventas ya enriquecidas",
            ],
            "prerequisites": ["Catálogo de productos actualizado"],
            "color": "info",
            "icon": "fas fa-magic",
        },
        "clear_cache": {
            "title": "Limpiar Cache",
            "description": "Elimina datos temporales en cache para mejorar rendimiento y forzar recarga de datos",
            "risk_level": "low",
            "estimated_time": "5-10 segundos",
            "requires_text_confirmation": False,
            "impacts": [
                "Degradación temporal de rendimiento",
                "Recarga lenta de datos frecuentes",
            ],
            "prerequisites": [],
            "color": "warning",
            "icon": "fas fa-broom",
        },
        "verify_connectivity": {
            "title": "Verificar Conectividad",
            "description": "Prueba conexiones a servicios externos (CIMA, base de datos, Redis, etc.)",
            "risk_level": "low",
            "estimated_time": "10-15 segundos",
            "requires_text_confirmation": False,
            "impacts": [
                "Solo lectura - no modifica datos",
                "Genera reporte de estado de conexiones",
            ],
            "prerequisites": [],
            "color": "info",
            "icon": "fas fa-network-wired",
        },
    }

    # Niveles de riesgo
    RISK_LEVELS = {
        "low": {
            "badge": "success",
            "text": "Riesgo Bajo",
            "icon": "fas fa-check-circle",
            "warning": "Operación segura con mínimo impacto",
        },
        "medium": {
            "badge": "warning",
            "text": "Riesgo Medio",
            "icon": "fas fa-exclamation-triangle",
            "warning": "Requiere precaución - puede interrumpir operaciones",
        },
        "high": {
            "badge": "danger",
            "text": "Riesgo Alto",
            "icon": "fas fa-ban",
            "warning": "PELIGROSO - puede causar pérdida de datos o interrupciones severas",
        },
    }


def create_danger_zone_panel(panel_id: str = "admin-danger-zone") -> html.Div:
    """
    Crea el panel de zona de peligro para administradores.

    Args:
        panel_id: ID único del panel
    """
    return dbc.Card(
        [
            # Header con advertencia visible
            dbc.CardHeader(
                [
                    html.Div(
                        [
                            html.H5(
                                html.Div(
                                    [
                                        html.I(className="fas fa-radiation-alt me-2 text-danger"),
                                        "ZONA DE PELIGRO - OPERACIONES CRÍTICAS",
                                    ]
                                ),
                                className="mb-1 text-danger",
                            ),
                            dbc.Badge("⚠️ SOLO ADMINISTRADORES EXPERTOS", color="danger", className="mb-2"),
                            html.P(
                                "Las operaciones en esta sección pueden causar interrupciones del servicio, "
                                "pérdida de datos o problemas de rendimiento. Use con extrema precaución.",
                                className="small text-muted mb-0",
                            ),
                        ]
                    )
                ],
                className="bg-danger bg-opacity-10 border-danger",
            ),
            dbc.CardBody(
                [
                    # Stores para gestión de estado
                    dcc.Store(id=f"{panel_id}-operation-store", data={}),
                    dcc.Store(id=f"{panel_id}-confirmation-store", data={}),
                    # Modal especializado para operaciones peligrosas
                    create_dangerous_operation_modal(panel_id),
                    # Advertencia inicial
                    dbc.Alert(
                        [
                            html.H6(
                                html.Div([html.I(className="fas fa-shield-alt me-2"), "Protecciones Implementadas"]),
                                className="alert-heading",
                            ),
                            html.Ul(
                                [
                                    html.Li("Confirmación de texto requerida para operaciones críticas"),
                                    html.Li("Validación múltiple antes de ejecución"),
                                    html.Li("Logs detallados de todas las acciones"),
                                    html.Li("Reversión automática en caso de error (cuando sea posible)"),
                                ],
                                className="mb-0",
                            ),
                        ],
                        color="info",
                        className="mb-4",
                    ),
                    # Contenido principal
                    html.Div(id=f"{panel_id}-content", children=[create_dangerous_operations_grid()]),
                ]
            ),
        ],
        className="border-danger shadow",
    )


def create_dangerous_operations_grid() -> html.Div:
    """Crea la grilla de operaciones peligrosas organizadas por nivel de riesgo."""

    # Agrupar por nivel de riesgo
    operations_by_risk = {"high": [], "medium": [], "low": []}

    for operation_id, config in AdminDangerZone.DANGEROUS_OPERATIONS.items():
        risk_level = config.get("risk_level", "medium")
        operations_by_risk[risk_level].append((operation_id, config))

    sections = []

    # Crear secciones por nivel de riesgo
    for risk_level in ["high", "medium", "low"]:
        operations = operations_by_risk[risk_level]
        if not operations:
            continue

        risk_info = AdminDangerZone.RISK_LEVELS.get(risk_level)

        # Header de sección
        section_header = html.Div(
            [
                html.H6(
                    html.Div([html.I(className=f"{risk_info['icon']} me-2"), risk_info["text"]]),
                    className=f"text-{risk_info['badge']}",
                ),
                html.Small(risk_info["warning"], className="text-muted"),
            ],
            className="mb-3",
        )

        # Cards de operaciones
        operation_cards = []
        for operation_id, config in operations:
            card = create_dangerous_operation_card(operation_id, config)
            operation_cards.append(card)

        sections.append(html.Div([section_header, dbc.Row(operation_cards, className="mb-4")]))

    return html.Div(sections)


def create_dangerous_operation_card(operation_id: str, config: Dict) -> dbc.Col:
    """Crea una tarjeta para una operación peligrosa específica."""
    risk_info = AdminDangerZone.RISK_LEVELS.get(config.get("risk_level", "medium"))

    return dbc.Col(
        [
            dbc.Card(
                [
                    dbc.CardBody(
                        [
                            # Header con icono y nivel de riesgo
                            html.Div(
                                [
                                    html.I(className=f"{config['icon']} fa-2x text-{config['color']}"),
                                    dbc.Badge(
                                        html.Div([html.I(className=f"{risk_info['icon']} me-1"), risk_info["text"]]),
                                        color=risk_info["badge"],
                                        className="ms-auto",
                                    ),
                                ],
                                className="d-flex justify-content-between align-items-start mb-3",
                            ),
                            # Título y descripción
                            html.H6(config["title"], className="card-title text-dark"),
                            html.P(config["description"], className="card-text text-muted small mb-3"),
                            # Información crítica
                            html.Div(
                                [
                                    html.Div(
                                        [
                                            html.I(className="fas fa-clock me-1 text-warning"),
                                            html.Small(f"Tiempo: {config['estimated_time']}", className="text-muted"),
                                        ],
                                        className="mb-2",
                                    ),
                                    # Prerequisitos si existen
                                    html.Div(
                                        (
                                            [
                                                html.I(className="fas fa-list-check me-1 text-info"),
                                                html.Small(
                                                    f"Prerequisitos: {len(config.get('prerequisites', []))}",
                                                    className="text-muted",
                                                ),
                                            ]
                                            if config.get("prerequisites")
                                            else []
                                        ),
                                        className="mb-3",
                                    ),
                                ]
                            ),
                            # Botón de acción con color según riesgo
                            dbc.Button(
                                html.Div([html.I(className=f"{config['icon']} me-2"), "Ejecutar Operación"]),
                                id={"type": "danger-operation-btn", "operation": operation_id},
                                color=config["color"],
                                size="sm",
                                className="w-100",
                                disabled=False,
                            ),
                        ]
                    )
                ],
                className="h-100 border-0 shadow-sm",
            )
        ],
        width=12,
        md=6,
        lg=4,
        className="mb-3",
    )


def create_dangerous_operation_modal(panel_id: str) -> dbc.Modal:
    """Crea modal especializado para confirmación de operaciones peligrosas."""
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(
                    html.Div(
                        [
                            html.I(className="fas fa-exclamation-triangle me-2 text-danger"),
                            html.Span(id=f"{panel_id}-modal-title", children="Confirmar Operación Peligrosa"),
                        ]
                    ),
                    className="d-flex align-items-center",
                ),
                className="bg-danger bg-opacity-10 border-danger",
            ),
            dbc.ModalBody(
                [
                    # Información de la operación
                    html.Div(id=f"{panel_id}-modal-operation-info"),
                    # Warning principal
                    dbc.Alert(
                        [
                            html.H6(
                                html.Div([html.I(className="fas fa-radiation-alt me-2"), "⚠️ ADVERTENCIA CRÍTICA ⚠️"]),
                                className="alert-heading text-danger",
                            ),
                            html.P(
                                "Esta operación puede causar interrupciones del servicio, pérdida de datos "
                                "o problemas graves de rendimiento. Verifique que:",
                                className="mb-2",
                            ),
                            html.Ul(
                                [
                                    html.Li("Tiene un backup reciente y válido"),
                                    html.Li("Ha comunicado la ventana de mantenimiento"),
                                    html.Li("No hay operaciones críticas en curso"),
                                    html.Li("Cuenta con tiempo suficiente para supervisar el proceso"),
                                ]
                            ),
                        ],
                        color="danger",
                        className="mb-4",
                    ),
                    # Prerequisitos dinámicos
                    html.Div(id=f"{panel_id}-modal-prerequisites"),
                    # Impactos esperados
                    html.Div(id=f"{panel_id}-modal-impacts"),
                    # Campo de confirmación de texto
                    html.Div(id=f"{panel_id}-modal-text-confirmation", style={"display": "none"}),
                ]
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        "Cancelar", id=f"{panel_id}-modal-cancel", color="secondary", outline=True, className="me-2"
                    ),
                    dbc.Button(
                        html.Div(
                            [html.I(className="fas fa-exclamation-triangle me-2"), "EJECUTAR OPERACIÓN PELIGROSA"]
                        ),
                        id=f"{panel_id}-modal-confirm",
                        color="danger",
                        className="ms-2",
                        disabled=True,
                    ),
                ]
            ),
        ],
        id=f"{panel_id}-modal",
        size="lg",
        centered=True,
        backdrop="static",
        keyboard=False,
    )


def create_operation_progress_panel(operation_id: str, progress_data: Dict) -> html.Div:
    """Crea panel de progreso para operaciones en ejecución."""
    config = AdminDangerZone.DANGEROUS_OPERATIONS.get(operation_id, {})

    progress = progress_data.get("progress", 0)
    status = progress_data.get("status", "running")
    current_step = progress_data.get("current_step", "Iniciando...")

    # Color según estado
    if status == "completed":
        color = "success"
        icon = "fas fa-check-circle"
    elif status == "error":
        color = "danger"
        icon = "fas fa-times-circle"
    else:
        color = "primary"
        icon = "fas fa-cog fa-spin"

    return dbc.Card(
        [
            dbc.CardHeader(
                html.H6(
                    html.Div([html.I(className=f"{icon} me-2"), f"Ejecutando: {config.get('title', operation_id)}"]),
                    className=f"text-{color} mb-0",
                )
            ),
            dbc.CardBody(
                [
                    # Barra de progreso
                    dbc.Progress(
                        [dbc.Progress(value=progress, color=color, bar=True, animated=status == "running")],
                        className="mb-3",
                        style={"height": "20px"},
                    ),
                    # Estado actual
                    html.Div([html.Strong("Estado actual: "), html.Span(current_step)], className="mb-2"),
                    # Información adicional
                    html.Div(
                        [
                            html.Small(f"Progreso: {progress}%", className="text-muted me-3"),
                            html.Small(
                                f"Tiempo estimado: {config.get('estimated_time', 'Variable')}", className="text-muted"
                            ),
                        ]
                    ),
                    # Logs en tiempo real si están disponibles
                    html.Div(id="operation-logs", className="mt-3"),
                ]
            ),
        ],
        className="mt-3",
    )


# Callbacks del componente
def register_danger_zone_callbacks(app):
    """Registra callbacks para la zona de peligro."""

    @app.callback(
        [
            Output("admin-danger-zone-modal", "is_open"),
            Output("admin-danger-zone-operation-store", "data"),
            Output("admin-danger-zone-content", "children"),
            Output("toast-trigger-store", "data"),
        ],
        [
            Input({"type": "danger-operation-btn", "operation": ALL}, "n_clicks"),
            Input("admin-danger-zone-modal-confirm", "n_clicks"),
            Input("admin-danger-zone-modal-cancel", "n_clicks"),
        ],
        [
            State("admin-danger-zone-modal", "is_open"),
            State("admin-danger-zone-operation-store", "data"),
            State("auth-tokens-store", "data"),  # FIX: Forzar sincronización de token JWT
        ],
        prevent_initial_call=True,
    )
    def handle_danger_modal_and_execution(
        n_clicks_list, confirm_clicks, cancel_clicks, modal_open, operation_data, auth_tokens
    ):
        """
        Callback consolidado que maneja apertura/cierre del modal y ejecución de operaciones.

        Cumple REGLA #11: ONE INPUT ONE CALLBACK
        - Múltiples Inputs: botones de operación, confirm, cancel
        - Múltiples Outputs: modal state, operation store, content, toast

        Usa ctx.triggered para identificar qué Input disparó el callback.

        CRÍTICO: Incluye State("auth-tokens-store", "data") para forzar sincronización.
        Sin este State, request_coordinator puede ejecutar requests sin JWT (error 401).
        Ver: docs/ZOMBIE_UPLOADS_FIX.md para detalles del timing issue.
        """
        from dash import no_update

        # DEBUGGING: Log de entrada del callback
        logger.info(f"[DANGER_ZONE] Callback triggered - ctx.triggered: {ctx.triggered}")
        logger.info(f"[DANGER_ZONE] n_clicks_list: {n_clicks_list}")
        logger.info(f"[DANGER_ZONE] confirm_clicks: {confirm_clicks}, cancel_clicks: {cancel_clicks}")

        # FIX: Verificar que tenemos token JWT antes de ejecutar operaciones
        # auth_tokens tiene estructura: {"tokens": "encrypted_jwt", "timestamp": "..."}
        if not auth_tokens or "tokens" not in auth_tokens:
            logger.warning("[DANGER_ZONE] No JWT token available - user not authenticated")
            return no_update, no_update, no_update, error_toast(
                "Error de autenticación. Por favor, recarga la página.", "No autorizado"
            )

        # BONUS: Validar que tokens no esté vacío
        if not auth_tokens.get("tokens"):
            logger.warning("[DANGER_ZONE] JWT tokens string is empty")
            return no_update, no_update, no_update, error_toast(
                "Token de autenticación vacío. Por favor, recarga la página.", "No autorizado"
            )

        if not ctx.triggered:
            logger.warning("[DANGER_ZONE] No trigger detected - returning no_update")
            return modal_open, operation_data or {}, no_update, {}

        trigger_id = ctx.triggered[0]["prop_id"]
        logger.info(f"[DANGER_ZONE] Detected trigger_id: {trigger_id}")

        # CASO 1: Click en botón de operación → Abrir modal
        if "danger-operation-btn" in trigger_id:
            logger.info("[DANGER_ZONE] Detected danger-operation-btn click")

            if not any(n_clicks_list):
                logger.warning(f"[DANGER_ZONE] n_clicks_list is empty or all None: {n_clicks_list}")
                return modal_open, operation_data or {}, no_update, {}

            button_id = trigger_id.split(".")[0]
            logger.info(f"[DANGER_ZONE] Parsing button_id: {button_id}")

            operation_id = eval(button_id)["operation"]
            logger.info(f"[DANGER_ZONE] Extracted operation_id: {operation_id}")

            config = AdminDangerZone.DANGEROUS_OPERATIONS.get(operation_id)
            if not config:
                logger.error(f"[DANGER_ZONE] No config found for operation: {operation_id}")
                return modal_open, operation_data or {}, no_update, {}

            new_operation_data = {
                "operation_id": operation_id,
                "config": config,
                "timestamp": datetime.now().isoformat(),
            }

            logger.info(f"[DANGER_ZONE] Opening modal for operation: {operation_id}")
            return True, new_operation_data, no_update, {}

        # CASO 2: Click en Cancel → Cerrar modal
        elif trigger_id.endswith("cancel.n_clicks"):
            # Guard: Solo mostrar toast si realmente hubo un click (no al cargar página)
            if cancel_clicks and cancel_clicks > 0:
                return False, {}, no_update, warning_toast("Operación cancelada por el usuario", "Cancelado")
            else:
                return False, {}, no_update, no_update

        # CASO 3: Click en Confirm → Ejecutar operación y cerrar modal
        elif trigger_id.endswith("confirm.n_clicks"):
            if not operation_data:
                return False, {}, no_update, {}

            operation_id = operation_data.get("operation_id")
            config = operation_data.get("config", {})

            try:
                # CRÍTICO: Construir headers explícitos desde auth_tokens para multi-worker safety
                # Issue #xxx: El singleton auth_context puede estar desincronizado en Gunicorn
                # Solución: Restaurar tokens desde State y construir headers explícitos
                from utils.auth import auth_manager

                explicit_headers = None
                if auth_tokens and "tokens" in auth_tokens:
                    # Restaurar tokens encriptados
                    if auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"]):
                        access_token = auth_manager.get_access_token()
                        if access_token:
                            explicit_headers = {"Authorization": f"Bearer {access_token}"}
                            logger.info("[DANGER_ZONE] Constructed explicit auth headers from State")
                        else:
                            logger.warning("[DANGER_ZONE] Failed to get access_token from auth_manager")
                    else:
                        logger.warning("[DANGER_ZONE] Failed to restore tokens from encrypted string")

                if not explicit_headers:
                    logger.error("[DANGER_ZONE] Could not construct auth headers - aborting operation")
                    return False, {}, no_update, error_toast(
                        "Error al construir headers de autenticación", "Error de Autenticación"
                    )

                # IMPORTANTE: Usar request_coordinator con auth_headers explícitos
                # Esto resuelve timing issues en multi-worker environments (Render)

                # Determinar endpoint según la operación
                if operation_id == "cleanup_zombie_uploads":
                    # Endpoint específico para cleanup de zombies
                    logger.info(f"[DANGER_ZONE] Calling cleanup endpoint with explicit_headers: {bool(explicit_headers)}")
                    result_data = request_coordinator.make_request(
                        "/api/v1/upload/cleanup/zombies",
                        method="POST",
                        params={"max_age_hours": 1.0},
                        cache_ttl=0,
                        bypass_cache=True,
                        timeout=30,
                        auth_headers=explicit_headers,
                        return_error_details=True
                    )

                elif operation_id == "reenrich_sales":
                    # Issue #452: Re-enriquecer ventas en manual_review
                    logger.info("[DANGER_ZONE] Calling reenrich execute-all endpoint")
                    result_data = request_coordinator.make_request(
                        "/api/v1/reenrichment/execute-all",
                        method="POST",
                        cache_ttl=0,
                        bypass_cache=True,
                        timeout=120,  # Puede tardar más
                        auth_headers=explicit_headers,
                        return_error_details=True
                    )

                elif operation_id == "clear_cache":
                    # Limpiar cache de aplicación
                    logger.info("[DANGER_ZONE] Calling clear-cache endpoint")
                    result_data = request_coordinator.make_request(
                        "/api/v1/admin-tools/maintenance/clear-cache",
                        method="POST",
                        cache_ttl=0,
                        bypass_cache=True,
                        timeout=30,
                        auth_headers=explicit_headers,
                        return_error_details=True
                    )

                elif operation_id == "verify_connectivity":
                    # Verificar conectividad a servicios externos
                    logger.info("[DANGER_ZONE] Calling verify-connectivity endpoint")
                    result_data = request_coordinator.make_request(
                        "/api/v1/admin-tools/maintenance/verify-connectivity",
                        method="POST",
                        cache_ttl=0,
                        bypass_cache=True,
                        timeout=30,
                        auth_headers=explicit_headers,
                        return_error_details=True
                    )

                else:
                    # Endpoint genérico para otras operaciones peligrosas
                    result_data = request_coordinator.make_request(
                        f"/api/v1/admin-tools/dangerous-operations/{operation_id}/execute",
                        method="POST",
                        data={"confirmation": True, "timestamp": datetime.now().isoformat(), "admin_confirmation": True},
                        cache_ttl=0,
                        bypass_cache=True,
                        timeout=60,
                        auth_headers=explicit_headers,
                        return_error_details=True
                    )

                # request_coordinator retorna Dict con data si exitoso, None si error
                if result_data:
                    # Verificar si hay error en la respuesta (return_error_details=True)
                    if result_data.get("error"):
                        # Estructura: {"error": True, "error_type": "...", "message": "...", "status_code": ...}
                        error_msg = result_data.get("message", "Error desconocido")
                        error_type = result_data.get("error_type", "unknown")
                        status_code = result_data.get("status_code", "N/A")
                        logger.error(f"[DANGER_ZONE] Operation failed: {error_type} ({status_code}) - {error_msg}")
                        toast_data = error_toast(f"Error al ejecutar {config['title']}: {error_msg}", f"Error de {error_type.title()}")
                        return False, {}, no_update, toast_data

                    # Crear panel de progreso si la operación es asíncrona
                    if result_data.get("async", False):
                        progress_panel = create_operation_progress_panel(operation_id, result_data)
                        toast_data = warning_toast(
                            f"Operación {config['title']} iniciada. Supervise el progreso.", "Operación en Progreso"
                        )
                        return False, {}, [progress_panel], toast_data
                    else:
                        # Operación síncrona completada
                        # Mostrar detalles específicos según operación
                        result_details = []

                        if operation_id == "cleanup_zombie_uploads":
                            zombies_cleaned = result_data.get("zombies_cleaned", 0)
                            zombie_details = result_data.get("zombie_details", [])

                            result_details.extend([
                                html.P(result_data.get("message", "Operación ejecutada exitosamente")),
                                html.Hr(),
                                html.Div([
                                    html.Strong("Uploads limpiados: "),
                                    html.Span(f"{zombies_cleaned}", className="badge bg-success ms-2")
                                ], className="mb-2"),
                            ])

                            if zombie_details:
                                result_details.append(
                                    html.Ul([
                                        html.Li([
                                            html.Code(detail.get("filename", ""), className="me-2"),
                                            html.Small(
                                                f"({detail.get('time_in_processing_hours', 0):.1f}h congelado)",
                                                className="text-muted"
                                            )
                                        ])
                                        for detail in zombie_details[:5]  # Mostrar max 5
                                    ], className="small")
                                )
                        else:
                            result_details.append(
                                html.P(result_data.get("message", "Operación ejecutada exitosamente"))
                            )

                        result_details.append(
                            html.Small(f"Ejecutado: {datetime.now().strftime('%H:%M:%S')}", className="text-muted")
                        )

                        result_card = dbc.Alert(
                            [
                                html.H6(
                                    html.Div([html.I(className="fas fa-check-circle me-2"), "Operación Completada"]),
                                    className="alert-heading",
                                ),
                                *result_details
                            ],
                            color="success",
                            dismissable=True,
                        )

                        toast_data = success_toast(f"{config['title']} completada exitosamente", "Operación Exitosa")
                        return False, {}, [result_card], toast_data

                else:
                    # result_data es None (error en la llamada API)
                    error_msg = "Sin respuesta del servidor o error de autenticación"
                    toast_data = error_toast(f"Error al ejecutar {config['title']}: {error_msg}", "Error de Operación")
                    return False, {}, no_update, toast_data

            except Exception as e:
                error_msg = f"Error de conexión: {str(e)}"
                toast_data = error_toast(f"Error al conectar para {config['title']}: {error_msg}", "Error de Conexión")
                return False, {}, no_update, toast_data

        return modal_open, operation_data or {}, no_update, {}

    @app.callback(
        [
            Output("admin-danger-zone-modal-title", "children"),
            Output("admin-danger-zone-modal-operation-info", "children"),
            Output("admin-danger-zone-modal-prerequisites", "children"),
            Output("admin-danger-zone-modal-impacts", "children"),
            Output("admin-danger-zone-modal-text-confirmation", "children"),
            Output("admin-danger-zone-modal-text-confirmation", "style"),
            Output("admin-danger-zone-modal-confirm", "disabled", allow_duplicate=True),
        ],
        Input("admin-danger-zone-operation-store", "data"),
        prevent_initial_call=True,
    )
    def update_danger_modal_content(operation_data):
        """Actualiza contenido del modal según la operación."""
        if not operation_data:
            return "Operación", html.Div([]), html.Div([]), html.Div([]), html.Div([]), {"display": "none"}, True

        config = operation_data.get("config", {})
        operation_id = operation_data.get("operation_id", "")

        # CRÍTICO: Habilitar botón automáticamente si NO requiere confirmación de texto
        requires_text_confirmation = config.get("requires_text_confirmation", False)
        confirm_button_disabled = requires_text_confirmation  # Deshabilitar solo si requiere texto

        logger.info(f"[DANGER_ZONE] Modal content update - operation: {operation_id}, requires_text: {requires_text_confirmation}, button_disabled: {confirm_button_disabled}")

        # Título
        title = f"⚠️ CONFIRMAR: {config.get('title', operation_id)}"

        # Información de operación
        operation_info = dbc.Card(
            [
                dbc.CardBody(
                    [
                        html.P(config.get("description", ""), className="mb-2"),
                        html.Div(
                            [
                                dbc.Badge(
                                    AdminDangerZone.RISK_LEVELS[config.get("risk_level", "medium")]["text"],
                                    color=AdminDangerZone.RISK_LEVELS[config.get("risk_level", "medium")]["badge"],
                                    className="me-2",
                                ),
                                html.Small(f"Tiempo estimado: {config.get('estimated_time', 'Variable')}"),
                            ]
                        ),
                    ]
                )
            ],
            color="light",
        )

        # Prerequisitos
        prerequisites = html.Div([])
        if config.get("prerequisites"):
            prereq_items = [html.Li(item) for item in config["prerequisites"]]
            prerequisites = html.Div(
                [html.H6("Prerequisitos Verificados:", className="text-info"), html.Ul(prereq_items)]
            )

        # Impactos
        impacts = html.Div([])
        if config.get("impacts"):
            impact_items = [html.Li(item, className="text-warning") for item in config["impacts"]]
            impacts = html.Div([html.H6("Impactos Esperados:", className="text-warning"), html.Ul(impact_items)])

        # Confirmación de texto
        text_confirmation = html.Div([])
        text_style = {"display": "none"}

        if config.get("requires_text_confirmation"):
            confirmation_text = config.get("confirmation_text", "CONFIRM")
            text_confirmation = html.Div(
                [
                    html.Hr(),
                    html.H6("Confirmación Requerida:", className="text-danger"),
                    html.P(
                        [
                            "Para proceder, escriba exactamente: ",
                            html.Code(confirmation_text, className="text-danger fw-bold"),
                        ]
                    ),
                    dbc.Input(
                        id="danger-text-confirmation-input",
                        placeholder=f"Escriba: {confirmation_text}",
                        className="mb-2",
                    ),
                ]
            )
            text_style = {"display": "block"}

        return title, operation_info, prerequisites, impacts, text_confirmation, text_style, confirm_button_disabled

    @app.callback(
        Output("admin-danger-zone-modal-confirm", "disabled"),
        Input("danger-text-confirmation-input", "value"),
        State("admin-danger-zone-operation-store", "data"),
        prevent_initial_call=True,
    )
    def validate_text_confirmation(input_value, operation_data):
        """Valida la confirmación de texto para habilitar ejecución."""
        if not operation_data:
            return True

        config = operation_data.get("config", {})

        if not config.get("requires_text_confirmation"):
            return False  # No requiere confirmación, habilitar botón

        required_text = config.get("confirmation_text", "")
        if input_value and input_value.strip() == required_text:
            return False  # Texto correcto, habilitar botón

        return True  # Deshabilitar botón

    # Nota: El callback execute_dangerous_operation fue consolidado
    # en handle_danger_modal_and_execution para cumplir REGLA #11


# Funciones de utilidad
def get_danger_zone_component(component_id: str = "main-danger-zone") -> html.Div:
    """
    Función principal para obtener el componente de zona de peligro.

    Args:
        component_id: ID único del componente
    """
    return create_danger_zone_panel(component_id)


def create_danger_zone_access_control() -> html.Div:
    """Crea control de acceso para la zona de peligro."""
    return dbc.Alert(
        [
            html.H6(html.Div([html.I(className="fas fa-lock me-2"), "Acceso Restringido"]), className="alert-heading"),
            html.P(
                "La zona de peligro requiere permisos de administrador. "
                "Contacte al administrador del sistema si necesita acceso.",
                className="mb-0",
            ),
        ],
        color="warning",
    )


def verify_admin_permissions() -> bool:
    """
    Verifica si el usuario tiene permisos de administrador.
    En producción, debería validar contra el sistema de autenticación.
    """
    # TODO: Implementar verificación real de permisos
    # Por ahora, siempre retorna True para desarrollo
    return True
