"""
Callbacks del treemap de NECESIDAD (Issue #461, #492, #505)

Maneja clicks en treemap para filtrar productos por categoría.
Incluye lógica 80/20 para categorías minoritarias (W14a).
Toggle entre treemap y barras horizontales (Issue #492).
Issue #505: L2 drill-down para dermocosmetica, suplementos, higiene_bucal.
NOTA: El renderizado inicial del treemap se hace en data_loading.py (REGLA #11)
"""

import logging

import dash_bootstrap_components as dbc
from dash import Input, Output, State, callback, ctx, html, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.categories import has_l2_subcategories, get_principal_info
from components.ventalibre.necesidad_treemap import create_necesidad_chart
from utils.helpers import format_currency, format_percentage

logger = logging.getLogger(__name__)


def register_treemap_callbacks(app):
    """Registrar callbacks del treemap (solo manejo de clicks)."""

    # NOTA: render_treemap movido a data_loading.py para cumplir REGLA #11
    # (ventalibre-data-store solo puede ser Input de UN callback)

    @app.callback(
        [
            Output("ventalibre-selected-category-store", "data"),
            Output("ventalibre-category-badge", "children"),
            Output("ventalibre-clear-category", "style"),
            # Issue #505: L2 modal outputs
            Output("ventalibre-l2-modal", "is_open"),
            Output("ventalibre-l2-modal-title", "children"),
            Output("ventalibre-selected-l1-for-l2", "data"),
        ],
        [
            Input("ventalibre-treemap", "clickData"),
            Input("ventalibre-clear-category", "n_clicks"),
            Input("ventalibre-l2-modal-close", "n_clicks"),
        ],
        State("ventalibre-data-store", "data"),
        State("ventalibre-l2-modal", "is_open"),
        prevent_initial_call=True,
    )
    def handle_treemap_click(click_data, clear_clicks, close_modal_clicks, treemap_data, modal_is_open):
        """
        Manejar click en treemap para filtrar productos por categoría.
        Issue #505: Si la categoría tiene L2, abre modal de drill-down en lugar de filtrar.
        También maneja el botón de limpiar filtro y cerrar modal L2.
        """
        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else ""

        # Si se clickeó el botón de cerrar modal L2
        if trigger_id == "ventalibre-l2-modal-close":
            logger.info("[VENTALIBRE] L2 modal closed")
            return no_update, no_update, no_update, False, no_update, None

        # Si se clickeó el botón de limpiar, resetear a "Todas"
        if trigger_id == "ventalibre-clear-category":
            logger.info("[VENTALIBRE] Category filter cleared")
            return None, "Todas", {"display": "none"}, no_update, no_update, no_update

        # Si es click en treemap
        if not click_data:
            raise PreventUpdate

        # Extraer categoría clickeada
        try:
            # El customdata tiene [category_id, count, percentage]
            point = click_data.get("points", [{}])[0]
            category_id = point.get("customdata", [None])[0]

            if not category_id:
                raise PreventUpdate

            # Issue #505: Check if category has L2 subcategories
            if has_l2_subcategories(category_id):
                # Open L2 modal instead of filtering
                l1_info = get_principal_info(category_id)
                modal_title = f"Subcategorías L2: {l1_info.get('name', category_id)}"
                logger.info(f"[VENTALIBRE] Opening L2 modal for: {category_id}")
                # Don't update category filter, open modal
                return no_update, no_update, no_update, True, modal_title, category_id

            # Normal L1 filter (no L2 available)
            category_label = category_id.replace("_", " ").title()
            logger.info(f"[VENTALIBRE] Category selected: {category_id}")
            # Mostrar botón de limpiar
            return category_id, category_label, {"display": "inline-block"}, no_update, no_update, no_update

        except (IndexError, TypeError, KeyError) as e:
            logger.warning(f"[VENTALIBRE] Failed to parse click data: {e}")
            raise PreventUpdate

    # ====================================================================
    # W14a: Callbacks para sección "Otras Categorías" (minoritarias)
    # ====================================================================

    @app.callback(
        [
            Output("ventalibre-other-section", "style"),
            Output("ventalibre-other-count", "children"),
            Output("ventalibre-other-count", "style"),
            Output("ventalibre-other-categories-table", "children"),
        ],
        Input("ventalibre-data-store", "modified_timestamp"),
        State("ventalibre-data-store", "data"),
        prevent_initial_call=True,
    )
    def update_other_categories_section(ts, treemap_data):
        """
        Actualizar sección de categorías minoritarias cuando cambian los datos.
        Muestra/oculta la sección según haya otras categorías.
        """
        if not treemap_data:
            raise PreventUpdate

        other_nodes = treemap_data.get("other_nodes", [])
        count = len(other_nodes)

        if count == 0:
            # Ocultar sección si no hay otras categorías
            return (
                {"display": "none"},  # Ocultar sección
                "0",                   # Badge text
                {"display": "none"},   # Ocultar badge
                None,                  # Sin tabla
            )

        # Hay categorías minoritarias - mostrar sección
        table = _create_other_categories_table(other_nodes)

        return (
            {"display": "block"},                    # Mostrar sección
            f"+{count}",                             # Badge con conteo
            {"display": "inline-block"},             # Mostrar badge
            table,                                   # Tabla renderizada
        )

    @app.callback(
        [
            Output("ventalibre-other-collapse", "is_open"),
            Output("ventalibre-toggle-icon", "className"),
        ],
        Input("ventalibre-toggle-other", "n_clicks"),
        State("ventalibre-other-collapse", "is_open"),
        prevent_initial_call=True,
    )
    def toggle_other_categories(n_clicks, is_open):
        """Toggle collapse de categorías minoritarias."""
        if n_clicks is None:
            raise PreventUpdate

        new_is_open = not is_open
        # Rotar icono según estado
        icon_class = "fas fa-chevron-up me-2" if new_is_open else "fas fa-chevron-down me-2"

        return new_is_open, icon_class

    # ====================================================================
    # Issue #492: Toggle Treemap ↔ Barras
    # ====================================================================

    @app.callback(
        [
            Output("ventalibre-chart-type-store", "data"),
            Output("ventalibre-chart-treemap-btn", "outline"),
            Output("ventalibre-chart-bars-btn", "outline"),
        ],
        [
            Input("ventalibre-chart-treemap-btn", "n_clicks"),
            Input("ventalibre-chart-bars-btn", "n_clicks"),
        ],
        State("ventalibre-chart-type-store", "data"),
        prevent_initial_call=True,
    )
    def toggle_chart_type(treemap_clicks, bars_clicks, current_type):
        """
        Manejar clicks en toggle buttons para cambiar tipo de gráfico.
        Actualiza el store y los estilos de los botones.
        """
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        if trigger_id == "ventalibre-chart-treemap-btn":
            new_type = "treemap"
            treemap_outline = False  # Activo (sólido)
            bars_outline = True      # Inactivo (outline)
        elif trigger_id == "ventalibre-chart-bars-btn":
            new_type = "bars"
            treemap_outline = True   # Inactivo (outline)
            bars_outline = False     # Activo (sólido)
        else:
            raise PreventUpdate

        logger.info(f"[VENTALIBRE] Chart type changed to: {new_type}")

        return new_type, treemap_outline, bars_outline

    @app.callback(
        Output("ventalibre-treemap-container", "children", allow_duplicate=True),
        Input("ventalibre-chart-type-store", "data"),
        State("ventalibre-data-store", "data"),
        prevent_initial_call=True,
    )
    def rerender_chart_on_type_change(chart_type, treemap_data):
        """
        Re-renderizar el gráfico cuando cambia el tipo (treemap/barras).
        Usa los datos existentes del store sin hacer nuevas llamadas API.
        """
        if not treemap_data:
            raise PreventUpdate

        chart_type = chart_type or "treemap"

        logger.info(f"[VENTALIBRE] Re-rendering chart as: {chart_type}")

        return create_necesidad_chart(treemap_data, chart_type=chart_type, height=400)


def _create_other_categories_table(other_nodes: list) -> dbc.Table:
    """
    Crear tabla de categorías minoritarias.

    Args:
        other_nodes: Lista de nodos con categorías < 2%

    Returns:
        dbc.Table con las categorías ordenadas por ventas
    """
    if not other_nodes:
        return html.P("No hay categorías minoritarias", className="text-muted small")

    # Ordenar por ventas descendente
    sorted_nodes = sorted(other_nodes, key=lambda x: x.get("sales", 0), reverse=True)

    # Crear filas de tabla
    rows = []
    for node in sorted_nodes:
        category = node.get("category", "")
        label = category.replace("_", " ").title()
        sales = node.get("sales", 0)
        count = node.get("count", 0)
        percentage = node.get("percentage", 0)

        rows.append(
            html.Tr([
                html.Td(label, className="small"),
                html.Td(format_currency(sales), className="small text-end"),
                html.Td(str(count), className="small text-end"),
                html.Td(f"{percentage:.1f}%", className="small text-end"),
            ])
        )

    return dbc.Table([
        html.Thead([
            html.Tr([
                html.Th("Categoría", className="small"),
                html.Th("Ventas", className="small text-end"),
                html.Th("Productos", className="small text-end"),
                html.Th("%", className="small text-end"),
            ])
        ]),
        html.Tbody(rows),
    ], striped=True, hover=True, size="sm", className="mb-0")
