# frontend/callbacks/ventalibre/roi_tracker.py
"""
Callbacks para ROI Tracker Widget (Issue #514)

Gestiona la carga de datos del widget ROI en el dashboard de venta libre.

Siguiendo REGLA #7.6: Multi-Worker Token Restoration para Render.
"""

import logging

from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.roi_tracker_card import create_roi_tracker_card
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_roi_tracker_callbacks(app):
    """Registra callbacks del widget ROI Tracker."""

    @app.callback(
        Output("ventalibre-roi-tracker-container", "children"),
        Input("ventalibre-tabs", "active_tab"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_roi_tracker_widget(active_tab, auth_state, auth_tokens):
        """
        Carga el widget ROI cuando se activa el tab de Analisis.

        Args:
            active_tab: Tab activo en ventalibre
            auth_state: Estado de autenticacion
            auth_tokens: Tokens encriptados para multi-worker

        Returns:
            Widget ROI Tracker card
        """
        # Solo cargar en tab de analisis (tab-analisis)
        if active_tab != "tab-analisis":
            raise PreventUpdate

        # REGLA #7.6: Verificar auth proactiva
        if not is_user_authenticated(auth_state):
            logger.debug("[ROI_TRACKER] User not authenticated - skipping load")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[ROI_TRACKER] No auth headers available - skipping API calls")
            raise PreventUpdate

        # Mostrar loading mientras cargamos
        try:
            # Llamar al endpoint del widget
            response = request_coordinator.make_request(
                "GET",
                f"{BACKEND_URL}/api/v1/roi-tracker/widget",
                auth_headers=auth_headers,
            )

            if response and not response.get("error"):
                return create_roi_tracker_card(widget_data=response)
            else:
                error_msg = response.get("error", "Error al cargar datos ROI")
                logger.warning(f"[ROI_TRACKER] API error: {error_msg}")
                return create_roi_tracker_card(error=error_msg)

        except Exception as e:
            logger.error(f"[ROI_TRACKER] Exception loading widget: {e}")
            return create_roi_tracker_card(error="Error de conexion")

    @app.callback(
        Output("roi-tracker-actions-modal", "is_open"),
        [
            Input("roi-tracker-view-actions-btn", "n_clicks"),
            Input("roi-tracker-modal-close-btn", "n_clicks"),
        ],
        State("roi-tracker-actions-modal", "is_open"),
        prevent_initial_call=True,
    )
    def toggle_actions_modal(open_clicks, close_clicks, is_open):
        """Toggle del modal de acciones pendientes."""
        trigger_id = ctx.triggered_id
        if trigger_id == "roi-tracker-view-actions-btn" and open_clicks:
            return True
        elif trigger_id == "roi-tracker-modal-close-btn" and close_clicks:
            return False
        return is_open
