"""
Callbacks para descarga de reportes (Issue #511)

- Informe PDF para Dirección: Resumen ejecutivo con KPIs y alertas
- Excel Pick-Lists para Suelo: Listas operativas de inventario

NOTA: Usamos requests directamente (no request_coordinator) porque los
endpoints devuelven contenido binario (PDF/Excel) y request_coordinator
siempre parsea JSON.

REGLA #7.6: Usamos get_auth_headers_from_tokens() para multi-worker Render.
"""

import requests
import structlog
from dash import Input, Output, State, no_update
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL

logger = structlog.get_logger(__name__)


def register_reports_callbacks(app):
    """
    Registrar callbacks para descarga de reportes.

    Args:
        app: Instancia de la aplicación Dash
    """

    @app.callback(
        Output("ventalibre-direccion-pdf-download", "data"),
        Output("ventalibre-reports-toast", "children", allow_duplicate=True),
        Input("ventalibre-download-pdf-btn", "n_clicks"),
        [
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token
        ],
        prevent_initial_call=True,
    )
    def download_direccion_pdf(n_clicks, start_date, end_date, auth_state, auth_tokens):
        """
        Descargar PDF One-Pager para Dirección.

        Returns:
            dcc.send_bytes con el PDF generado
        """
        from dash import dcc
        from components.toast_manager import error_toast

        if not n_clicks:
            raise PreventUpdate

        # REGLA #7.6: Verificar autenticación con helper
        if not is_user_authenticated(auth_state):
            logger.warning("reports.pdf.unauthorized_attempt")
            return no_update, error_toast("Sesión expirada. Por favor, inicie sesión nuevamente.")

        pharmacy_id = auth_state.get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("reports.pdf.no_pharmacy_id")
            return no_update, error_toast("No se encontró la farmacia asociada.")

        if not start_date or not end_date:
            logger.warning("reports.pdf.no_dates")
            return no_update, error_toast("Seleccione un rango de fechas válido.")

        # REGLA #7.6: Obtener auth_headers EXPLÍCITAMENTE (bypass singletons multi-worker)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.error("reports.pdf.no_auth_headers")
            return no_update, error_toast("Error de autenticación. Intente recargar la página.")

        try:
            logger.info(
                "reports.pdf.download_request",
                pharmacy_id=pharmacy_id,
                start_date=start_date,
                end_date=end_date,
            )

            # Request directo (no request_coordinator) para contenido binario
            # REGLA #7.6: Usar auth_headers explícitos, no singleton
            response = requests.get(
                f"{BACKEND_URL}/api/v1/reports/{pharmacy_id}/direccion",
                params={
                    "start_date": start_date,
                    "end_date": end_date,
                },
                headers=auth_headers,
                timeout=60,
            )

            if response.status_code == 403:
                logger.warning("reports.pdf.pro_required")
                return no_update, error_toast("Esta funcionalidad requiere suscripción PRO.")

            if response.status_code != 200:
                logger.error(
                    "reports.pdf.request_failed",
                    status_code=response.status_code,
                )
                return no_update, error_toast("Error generando el informe. Intente de nuevo.")

            # Generar nombre de archivo
            period = start_date[:7].replace("-", "")
            filename = f"Informe_Direccion_{period}.pdf"

            logger.info(
                "reports.pdf.download_success",
                filename=filename,
                size=len(response.content),
            )

            return dcc.send_bytes(response.content, filename), no_update

        except requests.Timeout:
            logger.error("reports.pdf.timeout")
            return no_update, error_toast("Tiempo de espera agotado. El servidor está ocupado.")
        except Exception as e:
            logger.error("reports.pdf.download_error", error=str(e))
            return no_update, error_toast("Error descargando el informe.")

    @app.callback(
        Output("ventalibre-suelo-excel-download", "data"),
        Output("ventalibre-reports-toast", "children", allow_duplicate=True),
        Input("ventalibre-download-excel-btn", "n_clicks"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token
        ],
        prevent_initial_call=True,
    )
    def download_suelo_excel(n_clicks, auth_state, auth_tokens):
        """
        Descargar Excel Pick-Lists para Suelo.

        Returns:
            dcc.send_bytes con el Excel generado
        """
        from dash import dcc
        from datetime import datetime, timezone
        from components.toast_manager import error_toast

        if not n_clicks:
            raise PreventUpdate

        # REGLA #7.6: Verificar autenticación con helper
        if not is_user_authenticated(auth_state):
            logger.warning("reports.excel.unauthorized_attempt")
            return no_update, error_toast("Sesión expirada. Por favor, inicie sesión nuevamente.")

        pharmacy_id = auth_state.get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("reports.excel.no_pharmacy_id")
            return no_update, error_toast("No se encontró la farmacia asociada.")

        # REGLA #7.6: Obtener auth_headers EXPLÍCITAMENTE (bypass singletons multi-worker)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.error("reports.excel.no_auth_headers")
            return no_update, error_toast("Error de autenticación. Intente recargar la página.")

        try:
            logger.info(
                "reports.excel.download_request",
                pharmacy_id=pharmacy_id,
            )

            # Request directo (no request_coordinator) para contenido binario
            # REGLA #7.6: Usar auth_headers explícitos, no singleton
            response = requests.get(
                f"{BACKEND_URL}/api/v1/reports/{pharmacy_id}/suelo",
                params={
                    "include_liquidation": True,
                    "include_restock": True,
                    "include_promotion": True,
                },
                headers=auth_headers,
                timeout=60,
            )

            if response.status_code == 403:
                logger.warning("reports.excel.pro_required")
                return no_update, error_toast("Esta funcionalidad requiere suscripción PRO.")

            if response.status_code != 200:
                logger.error(
                    "reports.excel.request_failed",
                    status_code=response.status_code,
                )
                return no_update, error_toast("Error generando el informe. Intente de nuevo.")

            # Generar nombre de archivo (UTC para consistencia)
            date_str = datetime.now(timezone.utc).strftime("%Y%m%d")
            filename = f"PickLists_{date_str}.xlsx"

            logger.info(
                "reports.excel.download_success",
                filename=filename,
                size=len(response.content),
            )

            return dcc.send_bytes(response.content, filename), no_update

        except requests.Timeout:
            logger.error("reports.excel.timeout")
            return no_update, error_toast("Tiempo de espera agotado. El servidor está ocupado.")
        except Exception as e:
            logger.error("reports.excel.download_error", error=str(e))
            return no_update, error_toast("Error descargando el informe.")

    @app.callback(
        Output("ventalibre-action-plan-download", "data"),
        Output("ventalibre-reports-toast", "children", allow_duplicate=True),
        Input("ventalibre-download-action-plan-btn", "n_clicks"),
        [
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token
        ],
        prevent_initial_call=True,
    )
    def download_action_plan(n_clicks, start_date, end_date, auth_state, auth_tokens):
        """
        Descargar Plan de Acción en Word (Issue #513).

        Genera documento Word con:
        - Portada con oportunidad económica total
        - Resumen ejecutivo con KPIs
        - Acciones agrupadas por prioridad
        - Checklist de seguimiento

        Returns:
            dcc.send_bytes con el documento Word generado
        """
        from dash import dcc
        from components.toast_manager import error_toast

        if not n_clicks:
            raise PreventUpdate

        # REGLA #7.6: Verificar autenticación con helper
        if not is_user_authenticated(auth_state):
            logger.warning("reports.action_plan.unauthorized_attempt")
            return no_update, error_toast("Sesión expirada. Por favor, inicie sesión nuevamente.")

        pharmacy_id = auth_state.get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("reports.action_plan.no_pharmacy_id")
            return no_update, error_toast("No se encontró la farmacia asociada.")

        if not start_date or not end_date:
            logger.warning("reports.action_plan.no_dates")
            return no_update, error_toast("Seleccione un rango de fechas válido.")

        # REGLA #7.6: Obtener auth_headers EXPLÍCITAMENTE (bypass singletons multi-worker)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.error("reports.action_plan.no_auth_headers")
            return no_update, error_toast("Error de autenticación. Intente recargar la página.")

        try:
            logger.info(
                "reports.action_plan.download_request",
                pharmacy_id=pharmacy_id,
                start_date=start_date,
                end_date=end_date,
            )

            # Request directo (no request_coordinator) para contenido binario
            # REGLA #7.6: Usar auth_headers explícitos, no singleton
            response = requests.get(
                f"{BACKEND_URL}/api/v1/reports/{pharmacy_id}/action-plan",
                params={
                    "start_date": start_date,
                    "end_date": end_date,
                    "format": "docx",
                    "max_actions": 20,
                },
                headers=auth_headers,
                timeout=90,  # Mayor timeout por generación de Word
            )

            if response.status_code == 403:
                logger.warning("reports.action_plan.pro_required")
                return no_update, error_toast("Esta funcionalidad requiere suscripción PRO.")

            if response.status_code != 200:
                logger.error(
                    "reports.action_plan.request_failed",
                    status_code=response.status_code,
                )
                return no_update, error_toast("Error generando el plan de acción. Intente de nuevo.")

            # Generar nombre de archivo
            period = start_date[:7].replace("-", "")
            filename = f"Plan_Accion_{period}.docx"

            logger.info(
                "reports.action_plan.download_success",
                filename=filename,
                size=len(response.content),
            )

            return dcc.send_bytes(response.content, filename), no_update

        except requests.Timeout:
            logger.error("reports.action_plan.timeout")
            return no_update, error_toast("Tiempo de espera agotado. El servidor está ocupado.")
        except Exception as e:
            logger.error("reports.action_plan.download_error", error=str(e))
            return no_update, error_toast("Error descargando el plan de acción.")
