"""
Callbacks de tabla de productos venta libre (Issue #461)

Carga de productos, paginación y búsqueda.
Sigue REGLA #7.6: Restaurar tokens antes de API calls.
"""

import logging

from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.productos_table import create_productos_table
from utils.auth import auth_manager
from utils.auth_helpers import is_user_authenticated, get_auth_headers_from_tokens
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Constantes de paginación
PAGE_SIZE = 20


def register_products_callbacks(app):
    """Registrar callbacks de tabla de productos."""

    @app.callback(
        [
            Output("ventalibre-products-container", "children"),
            Output("ventalibre-products-store", "data"),
            Output("ventalibre-page-info", "children"),
        ],
        [
            Input("ventalibre-selected-category-store", "data"),
            Input("ventalibre-search", "value"),
            Input("ventalibre-prev-page", "n_clicks"),
            Input("ventalibre-next-page", "n_clicks"),
            Input("ventalibre-data-store", "modified_timestamp"),  # FIX #536: usar data-store.modified_timestamp
        ],
        [
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("ventalibre-employee-filter", "value"),
            State("ventalibre-products-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("url", "pathname"),
        ],
        prevent_initial_call=True,
    )
    def load_products(
        selected_category,
        search_text,
        prev_clicks,
        next_clicks,
        data_timestamp,  # FIX #536: Trigger cuando data-store se actualiza
        start_date,
        end_date,
        employee_ids,
        current_products_data,
        auth_state,
        auth_tokens,
        pathname,
    ):
        """
        Cargar productos con filtros, búsqueda y paginación.

        REGLA #11: Un Input -> Un Callback (múltiples Inputs OK, múltiples States OK)
        REGLA #7.6: Restaurar tokens antes de API calls
        """
        # Guard: Solo en página correcta
        if pathname != "/ventalibre":
            raise PreventUpdate

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Obtener auth headers desde tokens store para multi-worker
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[VENTALIBRE] No auth headers available - skipping products load")
            raise PreventUpdate

        # Determinar trigger y offset
        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0] if ctx.triggered else ""

        # Calcular offset
        current_page = 1
        if current_products_data:
            current_page = current_products_data.get("page", 1)

        if trigger_id == "ventalibre-prev-page":
            current_page = max(1, current_page - 1)
        elif trigger_id == "ventalibre-next-page":
            current_page += 1
        elif trigger_id in ["ventalibre-selected-category-store", "ventalibre-search", "ventalibre-data-store"]:
            # Reset a página 1 si cambian filtros o se recargan datos
            current_page = 1

        offset = (current_page - 1) * PAGE_SIZE

        logger.debug(f"[VENTALIBRE] Loading products - page {current_page}, offset {offset}")

        try:
            # Preparar parámetros
            params = {
                "limit": PAGE_SIZE,
                "offset": offset,
            }
            if selected_category:
                params["necesidad"] = selected_category
            if search_text:
                params["search"] = search_text
            if start_date:
                params["date_from"] = start_date
            if end_date:
                params["date_to"] = end_date
            if employee_ids:
                params["employee_ids"] = employee_ids

            # API call
            response = request_coordinator.make_request(
                method="GET",
                endpoint="/api/v1/ventalibre/products",
                params=params,
                auth_headers=auth_headers,  # REGLA #7.6: Pasar headers explícitos
            )

            if not response:
                response = {"products": [], "total": 0, "page": 1, "pages": 0}

            products = response.get("products", [])
            total = response.get("total", 0)
            pages = response.get("pages", 0)

            # Crear tabla
            table_component = create_productos_table(products)

            # Info de página
            if total > 0:
                page_info = f"Página {current_page} de {pages} ({total} productos)"
            else:
                page_info = "No hay productos"

            # Guardar datos para paginación
            store_data = {
                "page": current_page,
                "pages": pages,
                "total": total,
            }

            logger.info(f"[VENTALIBRE] Loaded {len(products)} products (page {current_page}/{pages})")

            return table_component, store_data, page_info

        except Exception as e:
            logger.error(f"[VENTALIBRE] Error loading products: {e}")
            return (
                create_productos_table([]),
                {"page": 1, "pages": 0, "total": 0},
                "Error al cargar productos",
            )

    @app.callback(
        [
            Output("ventalibre-prev-page", "disabled"),
            Output("ventalibre-next-page", "disabled"),
        ],
        Input("ventalibre-products-store", "data"),
        prevent_initial_call=True,
    )
    def update_pagination_buttons(products_data):
        """
        Actualizar estado de botones de paginación.
        REGLA #11: Combinados en un solo callback (un Input -> un callback).
        """
        if not products_data:
            return True, True

        current_page = products_data.get("page", 1)
        total_pages = products_data.get("pages", 1)

        prev_disabled = current_page <= 1
        next_disabled = current_page >= total_pages

        return prev_disabled, next_disabled
