# frontend/callbacks/ventalibre/product_analysis.py
"""
Callbacks para Tab "Producto y Surtido" (Issue #494).

Maneja:
- Búsqueda de productos con autocompletado
- Carga de ficha, KPIs, alternativas, complementarios
- Scatter competitivo y Decision Box
"""

import logging

from dash import Input, Output, State, no_update
from dash.exceptions import PreventUpdate

from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_product_analysis_callbacks(app):
    """Registra todos los callbacks de Tab Producto y Surtido."""

    # =========================================================================
    # CALLBACK 1: Búsqueda de productos (autocompletado)
    # =========================================================================
    @app.callback(
        Output("ventalibre-product-search", "options"),
        Input("ventalibre-product-search", "search_value"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        prevent_initial_call=True,
    )
    def search_products(search_value, auth_state, auth_tokens):
        """Busca productos para autocompletado."""
        if not search_value or len(search_value) < 3:
            return []

        # Verificar auth
        if not is_user_authenticated(auth_state):
            return []

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[PRODUCT_ANALYSIS] No auth headers available - skipping API calls")
            return []

        try:
            response = request_coordinator.make_request(
                "/api/v1/ventalibre/search",
                method="GET",
                params={"q": search_value, "limit": 20},
                auth_headers=auth_headers,
            )

            if response and "results" in response:
                from components.ventalibre.product_search import format_search_option
                return [format_search_option(p) for p in response["results"]]

            return []

        except Exception as e:
            logger.warning(
                "[product_analysis] search error",
                extra={"search_value": search_value, "error": str(e), "error_type": type(e).__name__}
            )
            return []

    # =========================================================================
    # CALLBACK 2: Selección de producto -> cargar datos
    # =========================================================================
    @app.callback(
        Output("ventalibre-selected-product-store", "data"),
        Input("ventalibre-product-search", "value"),
        prevent_initial_call=True,
    )
    def on_product_selected(product_code):
        """Guarda el producto seleccionado."""
        if not product_code:
            return None
        return {"product_code": product_code}

    # =========================================================================
    # CALLBACK 3: Cargar todos los datos del producto (REGLA #11 compliance)
    # Un solo Input -> Un callback con múltiples Outputs
    # Optimizado: 1 API call batch en lugar de 5 individuales (~2-5s → ~0.5-1s)
    # =========================================================================
    @app.callback(
        Output("ventalibre-product-ficha-store", "data"),
        Output("ventalibre-alternatives-store", "data"),
        Output("ventalibre-complementary-store", "data"),
        Output("ventalibre-competitive-store", "data"),
        Output("ventalibre-recommendation-store", "data"),
        Input("ventalibre-selected-product-store", "data"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        prevent_initial_call=True,
    )
    def load_all_product_data(selected_product, auth_state, auth_tokens):
        """
        Carga todos los datos del producto usando endpoint batch.

        Usa /full-analysis para obtener todos los datos en una sola llamada,
        reduciendo latencia de ~2-5s a ~0.5-1s.

        Outputs:
        - Ficha con KPIs
        - Alternativas (mismo NECESIDAD)
        - Complementarios (MBA)
        - Análisis competitivo (Scatter)
        - Recomendación algorítmica
        """
        # Valores por defecto
        empty_result = (None, None, None, None, None)

        if not selected_product or not selected_product.get("product_code"):
            return empty_result

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[PRODUCT_ANALYSIS] No auth headers available - skipping API calls")
            return empty_result

        product_code = selected_product["product_code"]

        try:
            # Una sola llamada batch en lugar de 5 individuales
            response = request_coordinator.make_request(
                f"/api/v1/ventalibre/products/{product_code}/full-analysis",
                method="GET",
                params={
                    "alternatives_limit": 10,
                    "complementary_days": 90,
                    "complementary_limit": 10,
                },
                auth_headers=auth_headers,
            )

            if response:
                return (
                    response.get("ficha"),
                    response.get("alternatives"),
                    response.get("complementary"),
                    response.get("competitive"),
                    response.get("recommendation"),
                )

            return empty_result

        except Exception as e:
            logger.warning(
                "[product_analysis] full-analysis error",
                extra={"product_code": product_code, "error": str(e), "error_type": type(e).__name__}
            )
            return empty_result

    # =========================================================================
    # CALLBACK 8: Renderizar Ficha + KPIs (REGLA #11: un Input, un callback)
    # =========================================================================
    @app.callback(
        Output("ventalibre-product-ficha-container", "children"),
        Output("ventalibre-product-kpis-container", "children"),
        Input("ventalibre-product-ficha-store", "data"),
    )
    def render_product_ficha_and_kpis(ficha_data):
        """Renderiza la ficha del producto y los KPIs."""
        from components.ventalibre import (
            create_product_ficha,
            create_product_ficha_placeholder,
            create_product_kpis,
            create_kpis_placeholder,
        )

        # Ficha
        if not ficha_data:
            ficha_content = create_product_ficha_placeholder()
        else:
            ficha_content = create_product_ficha(ficha_data)

        # KPIs
        if not ficha_data or "kpis" not in ficha_data:
            kpis_content = create_kpis_placeholder()
        else:
            kpis_content = create_product_kpis(ficha_data["kpis"])

        return ficha_content, kpis_content

    # =========================================================================
    # CALLBACK 10: Renderizar Alternativas
    # =========================================================================
    @app.callback(
        Output("ventalibre-alternatives-container", "children"),
        Input("ventalibre-alternatives-store", "data"),
    )
    def render_alternatives(alternatives_data):
        """Renderiza tabla de alternativas."""
        from components.ventalibre import (
            create_alternatives_table,
            create_alternatives_placeholder,
        )

        if not alternatives_data:
            return create_alternatives_placeholder()

        return create_alternatives_table(
            alternatives=alternatives_data.get("alternatives", []),
            category=alternatives_data.get("category"),
            base_margin=alternatives_data.get("base_product_margin"),
        )

    # =========================================================================
    # CALLBACK 11: Renderizar Complementarios
    # =========================================================================
    @app.callback(
        Output("ventalibre-complementary-container", "children"),
        Input("ventalibre-complementary-store", "data"),
    )
    def render_complementary(complementary_data):
        """Renderiza tabla de complementarios (MBA)."""
        from components.ventalibre import (
            create_complementary_table,
            create_complementary_placeholder,
        )

        if not complementary_data:
            return create_complementary_placeholder()

        return create_complementary_table(
            complementary=complementary_data.get("complementary", []),
            base_product_name=complementary_data.get("base_product_name"),
            basket_count=complementary_data.get("base_product_basket_count"),
            analysis_period_days=complementary_data.get("analysis_period_days", 90),
        )

    # =========================================================================
    # CALLBACK 12: Renderizar Scatter Competitivo
    # =========================================================================
    @app.callback(
        Output("ventalibre-competitive-container", "children"),
        Input("ventalibre-competitive-store", "data"),
    )
    def render_competitive_scatter(competitive_data):
        """Renderiza scatter plot competitivo."""
        from components.ventalibre import (
            create_competitive_scatter,
            create_scatter_placeholder,
        )

        if not competitive_data:
            return create_scatter_placeholder()

        return create_competitive_scatter(
            data_points=competitive_data.get("data_points", []),
            category=competitive_data.get("category"),
            avg_sales=competitive_data.get("category_avg_sales"),
            avg_margin=competitive_data.get("category_avg_margin"),
        )

    # =========================================================================
    # CALLBACK 13: Renderizar Decision Box
    # =========================================================================
    @app.callback(
        Output("ventalibre-decision-box-container", "children"),
        Input("ventalibre-recommendation-store", "data"),
    )
    def render_decision_box(recommendation_data):
        """Renderiza el Decision Box con semáforo."""
        from components.ventalibre import (
            create_decision_box,
            create_decision_placeholder,
        )

        if not recommendation_data:
            return create_decision_placeholder()

        return create_decision_box(
            recommendation=recommendation_data.get("recommendation"),
            factors=recommendation_data.get("factors"),
            product_name=recommendation_data.get("product_name"),
        )

    # =========================================================================
    # CALLBACK 14: Toggle detalle recomendación
    # =========================================================================
    @app.callback(
        Output("ventalibre-recommendation-collapse", "is_open"),
        Input("ventalibre-recommendation-toggle", "n_clicks"),
        State("ventalibre-recommendation-collapse", "is_open"),
        prevent_initial_call=True,
    )
    def toggle_recommendation_detail(n_clicks, is_open):
        """Alterna visibilidad del detalle de recomendación."""
        if n_clicks:
            return not is_open
        return is_open

    logger.info("[product_analysis] 9 callbacks registered (Issue #494)")
