"""
L2 Drill-Down Callbacks (Issue #505)

Callbacks para el modal de subcategorías L2:
1. Cargar datos L2 desde API cuando se selecciona L1
2. Renderizar treemap L2
3. Click en L2 → filtrar productos + cerrar modal

NOTA: La apertura/cierre del modal L2 está integrada en treemap.py (REGLA #11)
para evitar duplicar el Input ventalibre-treemap.
"""

import logging
from typing import Optional

from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.categories import (
    L1_WITH_L2,
    get_l2_info,
    get_principal_info,
    has_l2_subcategories,
)
from components.ventalibre.l2_drilldown import (
    create_l2_archetype_legend,
    create_l2_coverage_badge,
    create_l2_treemap,
)
from components.ventalibre.l2_value_quadrant import create_l2_value_quadrant
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL
from utils.pharmacy_context import get_current_pharmacy_id
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_l2_drilldown_callbacks(app):
    """Register L2 drill-down callbacks."""

    # NOTA: handle_l1_click_open_l2_modal está integrado en treemap.py
    # para cumplir REGLA #11 (evitar Input duplicado ventalibre-treemap)

    @app.callback(
        [
            Output("ventalibre-l2-treemap-container", "children"),
            Output("ventalibre-l2-coverage-badge", "children"),
            Output("ventalibre-l2-archetype-legend", "children"),
            Output("ventalibre-l2-quadrant-container", "children"),
        ],
        [
            Input("ventalibre-selected-l1-for-l2", "data"),
            # Issue #537 FIX: Añadir fechas como Inputs para recargar cuando cambian
            Input("ventalibre-date-range", "start_date"),
            Input("ventalibre-date-range", "end_date"),
        ],
        [
            State("ventalibre-employee-filter", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_l2_data(l1_category, start_date, end_date, employee_id, auth_state, auth_tokens):
        """
        Load L2 sales data and value quadrant from API when L1 is selected.
        """
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if not l1_category:
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[L2_DRILLDOWN] No auth headers available - skipping API calls")
            raise PreventUpdate

        # Default empty quadrant
        empty_quadrant = create_l2_value_quadrant(None, l1_category)

        try:
            # REGLA #7.6: Use get_current_pharmacy_id() for consistent auth handling
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError as e:
                logger.warning(f"l2_drilldown.no_pharmacy_id: {e}")
                raise PreventUpdate

            # Call API for L2 sales data
            params = {
                "date_from": start_date,
                "date_to": end_date,
            }
            if employee_id:
                params["employee_ids"] = [employee_id]

            response = request_coordinator.make_request(
                "GET",
                f"{BACKEND_URL}/api/v1/ventalibre/sales-by-l2/{l1_category}",
                params=params,
                auth_headers=auth_headers,
            )

            if not response or "error" in response:
                logger.error(f"Error loading L2 data: {response}")
                return (
                    create_l2_treemap({}, l1_category),
                    create_l2_coverage_badge({"total": 0, "classified": 0, "percentage": 0}, l1_category),
                    create_l2_archetype_legend(l1_category),
                    empty_quadrant,
                )

            # Create treemap component
            treemap = create_l2_treemap(response, l1_category)

            # Get coverage from response or calculate
            coverage = response.get("coverage", {})
            if not coverage:
                nodes = response.get("nodes", [])
                total_sales = response.get("total_sales", 0)
                classified_sales = sum(n.get("sales", 0) for n in nodes)
                coverage = {
                    "total": int(total_sales),
                    "classified": int(classified_sales),
                    "percentage": (classified_sales / total_sales * 100) if total_sales > 0 else 0,
                }

            coverage_badge = create_l2_coverage_badge(coverage, l1_category)
            legend = create_l2_archetype_legend(l1_category)

            # Fetch value quadrant data (separate API call)
            try:
                quadrant_response = request_coordinator.make_request(
                    "GET",
                    f"{BACKEND_URL}/api/v1/ventalibre/value-quadrant-l2/{l1_category}",
                    params=params,
                    auth_headers=auth_headers,
                )

                if quadrant_response and "error" not in quadrant_response:
                    quadrant = create_l2_value_quadrant(quadrant_response, l1_category)
                else:
                    quadrant = empty_quadrant
            except Exception as qe:
                logger.warning(f"Error loading quadrant data: {qe}")
                quadrant = empty_quadrant

            return treemap, coverage_badge, legend, quadrant

        except Exception as e:
            logger.error(
                "l2_drilldown.load_error",
                extra={"l1_category": l1_category, "error": str(e)}
            )
            return (
                create_l2_treemap({}, l1_category),
                create_l2_coverage_badge({"total": 0, "classified": 0, "percentage": 0}, l1_category),
                create_l2_archetype_legend(l1_category),
                empty_quadrant,
            )

    @app.callback(
        [
            Output("ventalibre-selected-category-store", "data", allow_duplicate=True),
            Output("ventalibre-l2-modal", "is_open", allow_duplicate=True),
        ],
        Input("ventalibre-l2-treemap", "clickData"),
        [
            State("ventalibre-selected-l1-for-l2", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_l2_click(click_data, l1_category, auth_state):
        """
        Handle click on L2 treemap.

        Combined callback (REGLA #11):
        - Sets selected category to L2 for product filtering
        - Closes L2 modal
        """
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if not click_data or not l1_category:
            raise PreventUpdate

        try:
            point = click_data.get("points", [{}])[0]
            custom_data = point.get("customdata", [])

            if not custom_data:
                raise PreventUpdate

            # Get l2_category_id from customdata (index 0)
            l2_category = custom_data[0] if len(custom_data) > 0 else None

            if not l2_category:
                raise PreventUpdate

            logger.info(
                "l2_drilldown.category_selected",
                extra={"l2_category": l2_category, "l1_category": l1_category}
            )

            # Return L2 category to filter products + close modal
            # Format: {"level": 2, "l1": "dermocosmetica", "l2": "solar_facial"}
            category_data = {
                "level": 2,
                "l1": l1_category,
                "l2": l2_category,
            }

            return category_data, False  # Set category + close modal

        except (KeyError, IndexError) as e:
            logger.warning(f"Error parsing L2 treemap click data: {e}")
            raise PreventUpdate
