"""
Callbacks del modal de feedback (Issue #461)

Permite reportar errores de clasificación a través del FeedbackServiceV2.
"""

import logging

from dash import Input, Output, State, ALL, callback, ctx, no_update
from dash.exceptions import PreventUpdate
import dash_bootstrap_components as dbc
from dash import html

from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator
from components.toast_manager import success_toast, error_toast

logger = logging.getLogger(__name__)


def register_feedback_callbacks(app):
    """Registrar callbacks del modal de feedback."""

    @app.callback(
        [
            Output("ventalibre-feedback-modal", "is_open"),
            Output("ventalibre-feedback-content", "children"),
            Output("ventalibre-feedback-product-store", "data"),
            Output("ventalibre-feedback-category", "options"),
            Output("ventalibre-feedback-category", "value"),
            Output("ventalibre-feedback-comment", "value"),
        ],
        [
            Input({"type": "ventalibre-feedback-btn", "id": ALL}, "n_clicks"),
            Input("ventalibre-feedback-cancel", "n_clicks"),
        ],
        [
            State("ventalibre-products-store", "data"),
            State("ventalibre-data-store", "data"),
            State("auth-tokens-store", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_feedback_modal(
        feedback_clicks,
        cancel_clicks,
        products_data,
        treemap_data,
        auth_tokens,
        auth_state,
    ):
        """
        Abrir/cerrar modal de feedback y cargar datos del producto.

        Usa pattern matching callbacks para capturar clicks de cualquier botón de feedback.
        """
        if not ctx.triggered:
            raise PreventUpdate

        trigger = ctx.triggered[0]
        trigger_id = trigger["prop_id"].split(".")[0]

        # Si es cancel, cerrar modal
        if "ventalibre-feedback-cancel" in trigger_id:
            return False, "", None, [], None, ""

        # Si es click en botón feedback
        if "ventalibre-feedback-btn" in trigger_id:
            # Verificar que hubo click real
            if not any(c for c in (feedback_clicks or []) if c):
                raise PreventUpdate

            # Obtener ID del producto clickeado
            try:
                import json
                triggered_dict = json.loads(trigger_id)
                product_id = triggered_dict.get("id")
            except Exception:
                raise PreventUpdate

            if not product_id:
                raise PreventUpdate

            logger.info(f"[VENTALIBRE] Opening feedback modal for product: {product_id}")

            # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
            auth_headers = get_auth_headers_from_tokens(auth_tokens)

            # Obtener categorías específicas (NecesidadEspecifica) desde API
            categories = []
            try:
                # request_coordinator devuelve JSON parseado directamente (list/dict), no Response
                api_categories = request_coordinator.make_request(
                    method="GET",
                    endpoint="/api/v1/feedback/categories",
                    auth_headers=auth_headers,
                )
                if api_categories and isinstance(api_categories, list):
                    for cat in api_categories:
                        categories.append({
                            "label": cat.get("name", cat.get("slug", "")),
                            "value": cat.get("slug", ""),
                        })
                    logger.info(f"[VENTALIBRE] Loaded {len(categories)} categories from API")
            except Exception as e:
                logger.warning(f"[VENTALIBRE] Failed to load categories: {e}")

            # Fallback: usar treemap_data si API falló
            if not categories and treemap_data and "nodes" in treemap_data:
                for node in treemap_data["nodes"]:
                    cat_id = node["category"]
                    cat_label = cat_id.replace("_", " ").title()
                    categories.append({"label": cat_label, "value": cat_id})

            # Contenido del modal
            content = html.Div([
                html.P([
                    "¿La clasificación de este producto es incorrecta? ",
                    html.Strong("Ayúdanos a mejorar"),
                    " indicando la categoría correcta.",
                ]),
                html.Small(
                    f"Producto ID: {product_id}",
                    className="text-muted",
                ),
            ])

            return True, content, {"product_id": product_id}, categories, None, ""

        raise PreventUpdate

    @app.callback(
        [
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("ventalibre-feedback-modal", "is_open", allow_duplicate=True),
        ],
        Input("ventalibre-feedback-submit", "n_clicks"),
        [
            State("ventalibre-feedback-product-store", "data"),
            State("ventalibre-feedback-category", "value"),
            State("ventalibre-feedback-comment", "value"),
            State("auth-tokens-store", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def submit_feedback_and_close_modal(
        n_clicks,
        product_data,
        suggested_category,
        comment,
        auth_tokens,
        auth_state,
    ):
        """
        Enviar feedback y cerrar modal.

        REGLA #11: Combinados submit + close en un solo callback.
        Usa FeedbackServiceV2 (Issue #457) para registrar la corrección.
        """
        if not n_clicks or not product_data:
            raise PreventUpdate

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            return error_toast("Sesión expirada. Por favor, vuelve a iniciar sesión."), no_update

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[VENTALIBRE] No auth headers available for feedback submit")
            return error_toast("Error de autenticación. Por favor, vuelve a iniciar sesión."), no_update

        product_id = product_data.get("product_id")
        if not product_id:
            return error_toast("Error: ID de producto no encontrado"), no_update

        if not suggested_category:
            return error_toast("Por favor, selecciona una categoría"), no_update

        logger.info(f"[VENTALIBRE] Submitting feedback for {product_id} -> {suggested_category}")

        try:
            # Llamar al endpoint de corrección ventalibre (Issue #461)
            # request_coordinator devuelve JSON parseado directamente (dict), no Response
            result = request_coordinator.make_request(
                method="POST",
                endpoint="/api/v1/ventalibre/correct",
                data={
                    "product_id": product_id,
                    "corrected_category": suggested_category,
                    "reviewer_notes": comment or None,
                },
                return_error_details=True,  # Para obtener detalles del error
                auth_headers=auth_headers,
            )

            # Verificar si hubo error
            if result and isinstance(result, dict):
                if result.get("error"):
                    # Error estructurado del request_coordinator
                    error_msg = result.get("message", "Error desconocido")
                    status = result.get("status_code", 500)
                    if status == 400:
                        return error_toast(f"Error: {error_msg}"), no_update
                    else:
                        return error_toast("Error al enviar la corrección. Inténtalo de nuevo."), no_update
                elif result.get("success"):
                    # Éxito
                    logger.info(f"[VENTALIBRE] Correction submitted successfully for {product_id}")
                    return success_toast("Categoría corregida. Gracias por tu aporte."), False
                else:
                    # Respuesta inesperada pero puede ser éxito si tiene product_id
                    if result.get("product_id"):
                        return success_toast("Categoría corregida. Gracias por tu aporte."), False
                    return error_toast("Respuesta inesperada del servidor."), no_update
            else:
                return error_toast("Error al enviar la corrección. Inténtalo de nuevo."), no_update

        except Exception as e:
            logger.error(f"[VENTALIBRE] Error submitting feedback: {e}")
            return error_toast(f"Error al enviar reporte: {str(e)}"), no_update
