"""
Executive Summary Callbacks (Issue #509)

Callbacks para:
1. Cargar resumen ejecutivo desde NarrativeService API
2. Manejar feedback (thumbs up/thumbs down)
3. Navegar a detalle desde boton de accion

Sigue REGLA #7.6: Restaurar tokens desde auth-tokens-store antes de API calls.
"""

import hashlib
import json
import logging
from typing import Optional

from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.executive_summary_card import create_executive_summary_card
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL
from utils.pharmacy_context import get_current_pharmacy_id
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_executive_summary_callbacks(app):
    """Registrar callbacks de Executive Summary."""

    @app.callback(
        Output("ventalibre-executive-summary-container", "children"),
        Input("ventalibre-kpis-store", "data"),
        [
            State("ventalibre-date-range", "start_date"),  # FIX #536: Cambiar a State (REGLA #11)
            State("ventalibre-date-range", "end_date"),    # El callback se dispara por kpis-store
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_executive_summary(kpis_data, start_date, end_date, auth_state, auth_tokens):
        """
        Carga el resumen ejecutivo cuando hay KPIs disponibles.

        Se dispara cuando:
        - Se cargan los KPIs del dashboard (que incluye cambios de fecha)

        FIX Issue #536: ventalibre-date-range movido a State para cumplir REGLA #11.
        El callback data_loading.py es el único que escucha date-range como Input.
        """
        # Verificar autenticacion
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if not start_date or not end_date:
            return create_executive_summary_card(loading=False, error=None)

        # REGLA #7.6: Obtener auth headers EXPLÍCITOS para multi-worker
        # (get_auth_headers_from_tokens ya restaura tokens internamente)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("executive_summary.no_auth_headers")
            return create_executive_summary_card(error="Sesión expirada")

        try:
            # Usar get_current_pharmacy_id() para consistencia con otros callbacks
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError as e:
                logger.warning(f"executive_summary.no_pharmacy_id: {e}")
                return create_executive_summary_card(error="No hay farmacia configurada")

            # Mostrar loading mientras cargamos
            # (El callback se re-ejecutara cuando llegue la respuesta)

            # Llamar API NarrativeService
            response = request_coordinator.make_request(
                "GET",
                f"{BACKEND_URL}/api/v1/narrative/{pharmacy_id}/summary",
                params={
                    "date_from": start_date,
                    "date_to": end_date,
                    "language": "es",
                },
                auth_headers=auth_headers,  # REGLA #7.6: Pasar headers explícitos
            )

            if not response:
                logger.error("executive_summary.empty_response")
                return create_executive_summary_card(error="Sin respuesta del servidor")

            if "error" in response:
                logger.error(f"executive_summary.api_error: {response.get('error')}")
                return create_executive_summary_card(
                    error=response.get("detail", "Error desconocido")
                )

            logger.info(
                "executive_summary.loaded",
                extra={
                    "pharmacy_id": pharmacy_id,
                    "grounding_valid": response.get("grounding_valid"),
                    "insights_used": response.get("insights_used"),
                },
            )

            return create_executive_summary_card(summary_data=response)

        except Exception as e:
            logger.error(
                "executive_summary.load_error",
                extra={"error": str(e)},
                exc_info=True,
            )
            return create_executive_summary_card(error="Error al generar resumen")

    @app.callback(
        [
            Output("narrative-feedback-toast", "is_open"),
            Output("narrative-feedback-toast", "children"),
        ],
        [
            Input("narrative-feedback-up", "n_clicks"),
            Input("narrative-feedback-down", "n_clicks"),
        ],
        [
            State("ventalibre-executive-summary-container", "children"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_narrative_feedback(
        n_up, n_down, summary_container, auth_state, auth_tokens
    ):
        """
        Maneja feedback del usuario sobre la narrativa.

        Envia al backend el feedback (👍/👎) para mejorar el modelo.
        """
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if not ctx.triggered_id:
            raise PreventUpdate

        is_helpful = ctx.triggered_id == "narrative-feedback-up"

        # REGLA #7.6: Obtener auth headers EXPLÍCITOS para multi-worker
        # (get_auth_headers_from_tokens ya restaura tokens internamente)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            return True, "Sesión expirada"

        try:
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError:
                raise PreventUpdate

            # Generar hash del contenido (simplificado)
            narrative_hash = hashlib.sha256(
                json.dumps(summary_container, sort_keys=True, default=str).encode()
            ).hexdigest()[:32]

            # Enviar feedback
            response = request_coordinator.make_request(
                "POST",
                f"{BACKEND_URL}/api/v1/narrative/{pharmacy_id}/feedback",
                data={
                    "narrative_hash": narrative_hash,
                    "is_helpful": is_helpful,
                },
                auth_headers=auth_headers,  # REGLA #7.6: Pasar headers explícitos
            )

            if response and "error" not in response:
                emoji = "👍" if is_helpful else "👎"
                logger.info(
                    "executive_summary.feedback_submitted",
                    extra={"is_helpful": is_helpful, "pharmacy_id": pharmacy_id},
                )
                return True, f"Gracias por tu feedback {emoji}"
            else:
                return True, "Error al enviar feedback"

        except Exception as e:
            logger.error(
                "executive_summary.feedback_error",
                extra={"error": str(e)},
                exc_info=True,
            )
            return True, "Error al enviar feedback"

    @app.callback(
        [
            Output("ventalibre-selected-category-store", "data", allow_duplicate=True),
            Output("ventalibre-l2-modal", "is_open", allow_duplicate=True),
            Output("ventalibre-selected-l1-for-l2", "data", allow_duplicate=True),
        ],
        Input("narrative-action-button", "n_clicks"),
        [
            State("narrative-action-target", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_action_navigation(n_clicks, action_target, auth_state):
        """
        Maneja navegacion desde el boton de accion.

        Segun el tipo de target:
        - category: Actualiza ventalibre-selected-category-store
        - l2: Abre modal L2 con la categoria
        - product: (futuro) Navega a ficha de producto
        """
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if not n_clicks or not action_target:
            raise PreventUpdate

        target_type = action_target.get("type")
        target_id = action_target.get("id")

        if not target_type or not target_id:
            raise PreventUpdate

        logger.info(
            "executive_summary.action_navigation",
            extra={"target_type": target_type, "target_id": target_id},
        )

        if target_type == "category":
            # Navegar a categoria L1 (actualizar store)
            category_data = {"level": 1, "l1": target_id, "l2": None}
            return category_data, no_update, no_update

        elif target_type == "l2":
            # Abrir modal L2 con la categoria
            return no_update, True, target_id

        else:
            # Tipo no soportado (product, etc.)
            raise PreventUpdate

    @app.callback(
        Output("ventalibre-executive-summary-container", "children", allow_duplicate=True),
        Input("narrative-retry-button", "n_clicks"),
        [
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def retry_load_summary(n_clicks, start_date, end_date, auth_state, auth_tokens):
        """
        Reintentar carga del resumen ejecutivo.

        Hace la llamada real a la API (no solo muestra loading).
        """
        if not n_clicks:
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        if not start_date or not end_date:
            return create_executive_summary_card(error="Seleccione un rango de fechas")

        # REGLA #7.6: Obtener auth headers EXPLÍCITOS para multi-worker
        # (get_auth_headers_from_tokens ya restaura tokens internamente)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("executive_summary.retry_no_auth_headers")
            return create_executive_summary_card(error="Sesión expirada")

        try:
            try:
                pharmacy_id = get_current_pharmacy_id()
            except ValueError as e:
                logger.warning(f"refresh_summary.no_pharmacy_id: {e}")
                return create_executive_summary_card(error="No hay farmacia configurada")

            # Llamar API NarrativeService
            response = request_coordinator.make_request(
                "GET",
                f"{BACKEND_URL}/api/v1/narrative/{pharmacy_id}/summary",
                params={
                    "date_from": start_date,
                    "date_to": end_date,
                    "language": "es",
                },
                auth_headers=auth_headers,  # REGLA #7.6: Pasar headers explícitos
            )

            if not response:
                return create_executive_summary_card(error="Sin respuesta del servidor")

            if "error" in response:
                return create_executive_summary_card(
                    error=response.get("detail", "Error al reintentar")
                )

            logger.info(
                "executive_summary.retry_success",
                extra={"pharmacy_id": pharmacy_id},
            )

            return create_executive_summary_card(summary_data=response)

        except Exception as e:
            logger.error(
                "executive_summary.retry_error",
                extra={"error": str(e)},
                exc_info=True,
            )
            return create_executive_summary_card(error="Error al reintentar carga")
