"""
Callbacks para sección Evolución Temporal (Issue #491)

Gráfico de evolución temporal de ventas + Tabla YoY + Top Contributors.
Ubicación: Tab 1 (Análisis), debajo del treemap.

Sigue REGLA #7.6: Restaurar tokens desde auth-tokens-store antes de API calls.
Sigue REGLA #11: Un Input → Un Callback (consolidado).
Sigue Observación #2: Period selector tiene prioridad sobre start_date global.
Sigue Observación #3: Cada API call en try/except independiente.
"""

import logging
from datetime import datetime
from typing import Optional

import dash_bootstrap_components as dbc
from dash import Input, Output, State, html
from dash.exceptions import PreventUpdate
from dateutil.relativedelta import relativedelta

from components.ventalibre.evolution_chart import create_evolution_chart
from components.ventalibre.top_contributors import create_top_contributors_table
from components.ventalibre.yoy_table import create_yoy_table
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.pharmacy_context import get_current_pharmacy_id
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def _create_error_placeholder(message: str) -> html.Div:
    """
    Crea placeholder de error para cuando una API falla.

    Args:
        message: Mensaje de error a mostrar

    Returns:
        html.Div con alerta de error
    """
    return html.Div([
        dbc.Alert([
            html.I(className="fas fa-exclamation-triangle me-2"),
            message,
        ], color="warning", className="mb-0"),
    ])


def _create_empty_placeholder(message: str) -> html.Div:
    """
    Crea placeholder para cuando no hay datos.

    Args:
        message: Mensaje informativo

    Returns:
        html.Div con mensaje
    """
    return html.Div([
        html.Div([
            html.I(className="fas fa-info-circle fa-2x text-muted mb-2"),
            html.P(message, className="text-muted mb-0"),
        ], className="text-center py-4"),
    ])


def register_evolution_callbacks(app):
    """
    Registrar callbacks para la sección de Evolución Temporal.

    Args:
        app: Instancia de la aplicación Dash
    """
    _register_evolution_data_callback(app)


def _register_evolution_data_callback(app):
    """
    Callback CONSOLIDADO para cargar datos de evolución temporal.

    REGLA #11: Un Input → Un Callback.
    Este callback hace 3 llamadas API independientes y actualiza los 3 componentes.
    Cada API call tiene su propio try/except (Observación #3).
    """

    @app.callback(
        [
            Output("ventalibre-evolution-container", "children"),
            Output("ventalibre-yoy-container", "children"),
            Output("ventalibre-contributors-container", "children"),
        ],
        [
            Input("ventalibre-evolution-period", "value"),
            Input("ventalibre-contributors-direction", "value"),
            Input("ventalibre-date-range", "end_date"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_evolution_data(
        period_months: int,
        direction: str,
        end_date: Optional[str],
        auth_state: dict,
        auth_tokens: dict,
    ):
        """
        Cargar datos de evolución temporal desde 3 endpoints.

        Args:
            period_months: Período en meses (6, 12, 24)
            direction: Filtro de dirección para contributors (all, up, down)
            end_date: Fecha fin del rango global
            auth_state: Estado de autenticación
            auth_tokens: Tokens encriptados

        Returns:
            Tuple con los 3 componentes renderizados
        """
        # 1. Auth Guard
        if not is_user_authenticated(auth_state):
            logger.debug("[EVOLUTION] User not authenticated - skipping")
            raise PreventUpdate

        # 2. REGLA #7.6: Obtener auth headers explícitos
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[EVOLUTION] No auth headers available")
            return (
                _create_error_placeholder("Sesión expirada. Por favor, recarga la página."),
                _create_error_placeholder("Sesión expirada."),
                _create_error_placeholder("Sesión expirada."),
            )

        # 3. Defensive check: User must have associated pharmacy to view evolution data
        # (Even though API uses JWT, we prevent unnecessary API calls for edge cases)
        pharmacy_id = get_current_pharmacy_id()
        if not pharmacy_id:
            logger.warning("[EVOLUTION] No pharmacy_id found")
            return (
                _create_empty_placeholder("No se encontró farmacia asociada."),
                _create_empty_placeholder("No se encontró farmacia."),
                _create_empty_placeholder("No se encontró farmacia."),
            )

        # 4. Calcular fechas (Observación #2: period tiene prioridad)
        if end_date:
            try:
                final_end = datetime.fromisoformat(end_date.replace("Z", "+00:00"))
            except ValueError:
                final_end = datetime.now()
        else:
            final_end = datetime.now()

        final_start = final_end - relativedelta(months=period_months or 12)

        # Formatear fechas para API
        date_from = final_start.strftime("%Y-%m-%d")
        date_to = final_end.strftime("%Y-%m-%d")

        logger.info(
            f"[EVOLUTION] Loading data: period={period_months}m, "
            f"range={date_from} to {date_to}, direction={direction}"
        )

        # 5. Preparar parámetros comunes (sin pharmacy_id - viene del token)
        base_params = {
            "date_from": date_from,
            "date_to": date_to,
        }

        # ========================================================
        # 6. Fetch Evolution Chart Data (independiente)
        # ========================================================
        try:
            evo_resp = request_coordinator.make_request(
                "/api/v1/ventalibre/time-series",
                method="GET",
                params=base_params,
                auth_headers=auth_headers,
            )

            # Validar respuesta Y contenido no vacío
            time_series = evo_resp.get("time_series", []) if evo_resp else []
            category_summary = evo_resp.get("category_summary", []) if evo_resp else []

            if time_series and category_summary:
                evo_component = create_evolution_chart(
                    time_series_data=time_series,
                    category_summary=category_summary,
                    chart_type="line",
                    top_n=6,
                )
            else:
                evo_component = _create_empty_placeholder(
                    "No hay datos de evolución temporal disponibles."
                )

        except Exception as e:
            logger.error(f"[EVOLUTION] time-series API failed: {e}")
            evo_component = _create_error_placeholder(
                "Error cargando evolución temporal"
            )

        # ========================================================
        # 7. Fetch YoY Comparison Data (independiente)
        # ========================================================
        try:
            yoy_resp = request_coordinator.make_request(
                "/api/v1/ventalibre/yoy-comparison",
                method="GET",
                params=base_params,
                auth_headers=auth_headers,
            )

            # Validar respuesta Y contenido no vacío
            categories = yoy_resp.get("categories", []) if yoy_resp else []

            if categories:
                yoy_component = create_yoy_table(yoy_resp)
            else:
                yoy_component = _create_empty_placeholder(
                    "Requiere al menos 12 meses de datos para comparación YoY."
                )

        except Exception as e:
            logger.error(f"[EVOLUTION] yoy-comparison API failed: {e}")
            yoy_component = _create_error_placeholder(
                "Error cargando comparación anual"
            )

        # ========================================================
        # 8. Fetch Top Contributors Data (independiente)
        # ========================================================
        try:
            contrib_params = {**base_params, "direction": direction or "all"}
            contrib_resp = request_coordinator.make_request(
                "/api/v1/ventalibre/top-contributors",
                method="GET",
                params=contrib_params,
                auth_headers=auth_headers,
            )

            # Validar respuesta Y contenido no vacío
            contributors = contrib_resp.get("contributors", []) if contrib_resp else []

            if contributors:
                contrib_component = create_top_contributors_table(
                    contributors_data=contrib_resp,
                    direction_filter=direction or "all",
                )
            else:
                contrib_component = _create_empty_placeholder(
                    "No hay datos de contribuidores disponibles."
                )

        except Exception as e:
            logger.error(f"[EVOLUTION] top-contributors API failed: {e}")
            contrib_component = _create_error_placeholder(
                "Error cargando contribuidores"
            )

        return evo_component, yoy_component, contrib_component
