"""
Callbacks de carga de datos para Venta Libre (Issue #461)

Carga inicial de KPIs, datos treemap y categorías.
Sigue REGLA #7.6: Restaurar tokens desde auth-tokens-store antes de API calls.
"""

import logging
import requests
from datetime import datetime, timedelta

from dash import Input, Output, State, callback, ctx, no_update
import plotly.graph_objects as go
from dash.exceptions import PreventUpdate

from utils.auth import auth_manager
from utils.auth_helpers import is_user_authenticated, get_auth_headers_from_tokens
from utils.request_coordinator import request_coordinator
from utils.helpers import format_currency, format_number, format_percentage
from utils.api_client import api_client
from utils.pharmacy_context import get_current_pharmacy_id
from components.ventalibre.necesidad_treemap import create_necesidad_treemap
from components.ventalibre.categories import (
    ESPECIFICA_TO_PRINCIPAL,
    PRINCIPAL_CATEGORIES,
    get_principal_category,
)

logger = logging.getLogger(__name__)


def _aggregate_by_principal(nodes: list) -> list:
    """
    Agrupar nodos de NecesidadEspecifica por NecesidadPrincipal.

    Transforma ~180 categorías específicas en 18 categorías principales
    para una visualización más manejable.

    Args:
        nodes: Lista de nodos con category (NecesidadEspecifica)

    Returns:
        Lista de nodos agrupados por NecesidadPrincipal
    """
    if not nodes:
        return []

    # Agrupar por principal
    principal_totals = {}

    for node in nodes:
        especifica = node.get("category", "otros")
        principal = get_principal_category(especifica)

        if principal not in principal_totals:
            principal_totals[principal] = {
                "sales": 0,
                "count": 0,
                "pending_corrections": 0,
                "subcategories": 0,
            }

        principal_totals[principal]["sales"] += node.get("sales", 0)
        principal_totals[principal]["count"] += node.get("count", 0)
        principal_totals[principal]["pending_corrections"] += node.get("pending_corrections", 0)
        principal_totals[principal]["subcategories"] += 1

    # Calcular total para porcentajes
    total_sales = sum(p["sales"] for p in principal_totals.values())

    # Convertir a lista de nodos
    result = []
    for principal, data in principal_totals.items():
        info = PRINCIPAL_CATEGORIES.get(principal, {"name": principal.title(), "color": "#95a5a6"})
        percentage = (data["sales"] / total_sales * 100) if total_sales > 0 else 0

        result.append({
            "category": principal,
            "display_name": info["name"],
            "sales": data["sales"],
            "count": data["count"],
            "percentage": percentage,
            "pending_corrections": data["pending_corrections"],
            "subcategories": data["subcategories"],
        })

    # Ordenar por ventas descendente
    return sorted(result, key=lambda x: x["sales"], reverse=True)


# Estilo para controles deshabilitados (PRO gating)
DISABLED_CONTROL_STYLE = {
    "opacity": "0.5",
    "cursor": "not-allowed",
    "backgroundColor": "#f8f9fa",
}


def register_data_loading_callbacks(app):
    """Registrar callbacks de carga de datos."""

    # ============================================================================
    # 0.5 CALLBACK CARGAR EMPLEADOS - Issue #402: Employee filtering integration
    # ============================================================================
    @app.callback(
        [
            Output("ventalibre-employee-filter", "options"),
            Output("ventalibre-employee-filter", "disabled"),
            Output("ventalibre-employee-filter", "style"),
            Output("ventalibre-employee-filter", "placeholder"),
        ],
        [Input("url", "pathname"), Input("auth-ready", "data")],
        [State("auth-state", "data")],
        prevent_initial_call=False,
    )
    def load_employee_options(pathname, auth_ready, auth_state):
        """
        Cargar opciones del dropdown de empleados para el filtro.

        Issue #402: Employee filtering integration - Las opciones se cargan
        dinámicamente desde /api/v1/employees/{pharmacy_id}.
        Issue #435: FREE tier UX - Deshabilitar dropdown con placeholder PRO.

        Incluye valor especial "__sin_empleado__" para ventas sin empleado asignado.

        PRO Gating: El endpoint requiere plan PRO+. Si el usuario es FREE,
        el dropdown queda deshabilitado con placeholder indicativo.
        """
        # Guard: Solo ejecutar en /ventalibre
        if pathname != "/ventalibre":
            raise PreventUpdate

        # Guard: Esperar auth-ready
        if not auth_ready:
            logger.debug("[VENTALIBRE] load_employee_options: Waiting for auth-ready")
            raise PreventUpdate

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[VENTALIBRE] load_employee_options: User not authenticated")
            raise PreventUpdate

        # Guard: Verificar tier ANTES de llamar API (evita 403 innecesarios)
        user_data = auth_state.get("user", {})
        subscription_plan = user_data.get("subscription_plan", "").lower()

        # Issue #435: FREE tier - Deshabilitar dropdown con placeholder PRO
        if subscription_plan == "free":
            logger.debug("[VENTALIBRE] load_employee_options: FREE tier - disabling employee filter")
            return [], True, DISABLED_CONTROL_STYLE, "👑 Disponible en plan PRO"

        # Usuario PRO/MAX: Cargar empleados desde API
        try:
            pharmacy_id = get_current_pharmacy_id()
            logger.info(f"[VENTALIBRE] load_employee_options: Loading employees for pharmacy {pharmacy_id}")

            response = api_client.get(f"/api/v1/employees/{pharmacy_id}")

            if not response:
                logger.warning("[VENTALIBRE] load_employee_options: Empty response from employees API")
                return [], False, {}, "Todos los empleados"

            employees = response.get("employees", [])

            # Construir opciones para el dropdown
            options = []

            # Opción especial para ventas sin empleado
            options.append({"label": "🚫 Sin empleado asignado", "value": "__sin_empleado__"})

            # Agregar cada empleado con su contador de ventas
            for emp in employees:
                name = emp.get("name", "Desconocido")
                sales_count = emp.get("sales_count", 0)
                options.append({"label": f"{name} ({format_number(sales_count)} ventas)", "value": name})

            logger.info(f"[VENTALIBRE] load_employee_options: Loaded {len(options)} employee options")
            return options, False, {}, "Todos los empleados"

        except requests.exceptions.HTTPError as e:
            # 403 = Usuario FREE sin acceso a esta función PRO (fallback si guard falla)
            if e.response is not None and e.response.status_code == 403:
                logger.info("[VENTALIBRE] load_employee_options: User is FREE tier (403) - employee filter disabled")
                return [], True, DISABLED_CONTROL_STYLE, "👑 Disponible en plan PRO"
            logger.error(f"[VENTALIBRE] load_employee_options: HTTP error: {e}")
            return [], False, {}, "Todos los empleados"
        except Exception as e:
            logger.error(f"[VENTALIBRE] load_employee_options: Error loading employees: {e}")
            return [], False, {}, "Todos los empleados"

    # ============================================================================
    # 1. CALLBACK CARGAR DATOS INICIALES
    # ============================================================================
    @app.callback(
        [
            Output("ventalibre-data-store", "data"),
            Output("ventalibre-kpis-store", "data"),
            Output("ventalibre-load-interval", "disabled"),
            Output("ventalibre-treemap-container", "children"),  # REGLA #11: Renderizar treemap aquí
        ],
        [
            Input("url", "pathname"),
            Input("ventalibre-date-range", "start_date"),
            Input("ventalibre-date-range", "end_date"),
            Input("ventalibre-employee-filter", "value"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=False,
    )
    def load_initial_data(
        pathname,
        start_date,
        end_date,
        employee_ids,
        auth_state,
        auth_tokens
    ):
        """
        Cargar datos iniciales cuando se navega a /ventalibre.

        REGLA #7.6: Restaurar tokens antes de API calls en Render multi-worker.
        """
        # Guard: Solo ejecutar en la página correcta
        if pathname != "/ventalibre":
            raise PreventUpdate

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[VENTALIBRE] User not authenticated - skipping data load")
            raise PreventUpdate

        # REGLA #7.6: Obtener auth headers desde tokens store para multi-worker
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[VENTALIBRE] No auth headers available - skipping data load")
            raise PreventUpdate

        logger.info("[VENTALIBRE] Loading initial data")

        try:
            # Usar fechas del DatePickerRange directamente
            date_from, date_to = start_date, end_date

            # Preparar parámetros de query
            params = {}
            if date_from:
                params["date_from"] = date_from
            if date_to:
                params["date_to"] = date_to
            if employee_ids:
                params["employee_ids"] = employee_ids

            # Llamada 1: Datos para treemap
            treemap_response = request_coordinator.make_request(
                method="GET",
                endpoint="/api/v1/ventalibre/sales-by-necesidad",
                params=params,
                auth_headers=auth_headers,  # REGLA #7.6: Pasar headers explícitos
            )
            treemap_data = treemap_response if treemap_response else {"nodes": [], "total_sales": 0}

            # ================================================================
            # AGRUPACIÓN POR NecesidadPrincipal (18 categorías principales)
            # En vez de ~180 categorías específicas, agrupamos por las 18 principales
            # para una visualización más manejable y útil.
            # ================================================================
            raw_nodes = treemap_data.get("nodes", [])
            total_sales = treemap_data.get("total_sales", 0) or sum(n.get("sales", 0) for n in raw_nodes)

            # Agrupar por categoría principal
            grouped_nodes = _aggregate_by_principal(raw_nodes)

            logger.info(
                f"[VENTALIBRE] Grouped {len(raw_nodes)} specific categories "
                f"into {len(grouped_nodes)} principal categories"
            )

            # Actualizar treemap_data con nodos agrupados
            treemap_data = {
                "nodes": grouped_nodes,        # 18 categorías principales
                "other_nodes": [],             # Ya no hay "otras" - todas están agrupadas
                "total_sales": total_sales,
                "raw_nodes": raw_nodes,        # Guardar originales por si se necesitan
            }
            # ================================================================

            # Llamada 2: KPIs
            kpis_response = request_coordinator.make_request(
                method="GET",
                endpoint="/api/v1/ventalibre/kpis",
                params=params,
                auth_headers=auth_headers,  # REGLA #7.6: Pasar headers explícitos
            )
            kpis_data = kpis_response if kpis_response else {
                "total_sales": 0,
                "total_products": 0,
                "categories_count": 0,
                "coverage_percent": 0,
            }

            logger.info(
                "[VENTALIBRE] Data loaded successfully",
                extra={
                    "total_sales": kpis_data.get("total_sales"),
                    "categories": len(treemap_data.get("nodes", [])),
                }
            )

            # REGLA #11: Renderizar treemap aquí en lugar de callback separado
            treemap_component = create_necesidad_treemap(treemap_data, height=400)

            return (
                treemap_data,      # Store treemap
                kpis_data,         # Store kpis (mantener como trigger para products.py)
                True,              # Deshabilitar interval (ya cargó)
                treemap_component, # REGLA #11: Treemap renderizado
            )

        except Exception as e:
            logger.error(f"[VENTALIBRE] Error loading data: {e}")
            empty_treemap_data = {"nodes": [], "total_sales": 0}
            return (
                empty_treemap_data,
                {"total_sales": 0, "total_products": 0, "categories_count": 0, "coverage_percent": 0},
                True,
                create_necesidad_treemap(empty_treemap_data, height=400),  # Treemap vacío
            )


def register_context_treemap_callbacks(app):
    """Registrar callback para treemap de contexto."""

    @app.callback(
        Output("ventalibre-context-treemap", "figure"),
        Input("ventalibre-data-store", "modified_timestamp"),
        [
            State("ventalibre-data-store", "data"),
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_context_treemap(ts, data_store, start_date, end_date, auth_state, auth_tokens):
        """
        Actualizar treemap de contexto mostrando proporción Venta Libre vs Prescripción.

        Jerarquía (similar a /prescription y /generics):
        - Ventas Totales Farmacia
          - Venta Libre (destacado - es la página actual)
          - Prescripción

        Llama al endpoint /api/v1/analysis/context-treemap para obtener el total
        de la farmacia y calcula las proporciones.
        """
        from utils.config import BACKEND_URL
        from utils.pharmacy_context import get_current_pharmacy_id

        # Guard: Verificar datos de venta libre
        if not data_store:
            return _create_empty_context_treemap("Sin datos disponibles")

        venta_libre_sales = data_store.get("total_sales", 0)
        if venta_libre_sales == 0:
            return _create_empty_context_treemap("Sin datos de ventas")

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            return _create_empty_context_treemap("Sin sesión activa")

        # REGLA #7.6: Obtener auth headers
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            return _create_empty_context_treemap("Sesión expirada")

        try:
            pharmacy_id = get_current_pharmacy_id()
            if not pharmacy_id:
                return _create_empty_context_treemap("Farmacia no configurada")

            # Preparar parámetros de fecha
            params = {}
            if start_date:
                params["start_date"] = start_date[:10] if isinstance(start_date, str) else start_date
            if end_date:
                params["end_date"] = end_date[:10] if isinstance(end_date, str) else end_date

            # Llamar al endpoint de contexto para obtener total farmacia
            response = request_coordinator.make_request(
                method="GET",
                endpoint=f"/api/v1/analysis/context-treemap/{pharmacy_id}",
                params=params,
                auth_headers=auth_headers,
            )

            if not response:
                logger.warning("[VENTALIBRE] Context treemap: No response from API")
                # Fallback: mostrar solo venta libre
                return _create_simple_context_treemap(venta_libre_sales)

            hierarchy = response.get("hierarchy", {})
            total_farmacia = hierarchy.get("value", 0)

            if total_farmacia <= 0:
                return _create_simple_context_treemap(venta_libre_sales)

            # Calcular prescripción = total - venta libre
            prescription_sales = max(0, total_farmacia - venta_libre_sales)

            logger.info(
                f"[VENTALIBRE] Context treemap: Total={total_farmacia:.2f}€, "
                f"VentaLibre={venta_libre_sales:.2f}€, Prescription={prescription_sales:.2f}€"
            )

            return _create_hierarchical_context_treemap(
                total_sales=total_farmacia,
                venta_libre_sales=venta_libre_sales,
                prescription_sales=prescription_sales,
            )

        except Exception as e:
            logger.error(f"[VENTALIBRE] Context treemap error: {e}")
            return _create_simple_context_treemap(venta_libre_sales)


def _create_empty_context_treemap(message: str) -> go.Figure:
    """Crear treemap vacío con mensaje."""
    fig = go.Figure()
    fig.add_annotation(
        text=message,
        showarrow=False,
        font={"size": 14, "color": "#6c757d"},
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
    )
    fig.update_layout(
        margin={"t": 10, "b": 10, "l": 10, "r": 10},
        height=280,
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )
    return fig


def _create_simple_context_treemap(venta_libre_sales: float) -> go.Figure:
    """
    Fallback: Crear treemap simple solo con venta libre (sin contexto de farmacia).
    """
    fig = go.Figure(
        go.Treemap(
            labels=["Venta Libre"],
            parents=[""],
            values=[venta_libre_sales],
            text=[format_currency(venta_libre_sales)],
            marker={"colors": ["#28a745"]},  # Verde para venta libre
            textinfo="label+text",
            textfont={"size": 14},
            hovertemplate="<b>%{label}</b><br>%{text}<extra></extra>",
        )
    )
    fig.update_layout(
        margin={"t": 5, "b": 5, "l": 5, "r": 5},
        height=280,
        paper_bgcolor="rgba(0,0,0,0)",
        font={"family": "Arial, sans-serif"},
    )
    return fig


def _create_hierarchical_context_treemap(
    total_sales: float,
    venta_libre_sales: float,
    prescription_sales: float,
) -> go.Figure:
    """
    Crear treemap jerárquico mostrando contexto de ventas.

    Jerarquía:
    - Ventas Totales (gris oscuro)
      - Venta Libre (verde - destacado, es la página actual)
      - Prescripción (azul)

    Similar al patrón de /prescription y /generics.
    """
    # Calcular porcentajes
    pct_vl = (venta_libre_sales / total_sales * 100) if total_sales > 0 else 0
    pct_rx = (prescription_sales / total_sales * 100) if total_sales > 0 else 0

    # Arrays para treemap
    labels = ["Ventas Totales", "Venta Libre", "Prescripción"]
    parents = ["", "Ventas Totales", "Ventas Totales"]
    values = [total_sales, venta_libre_sales, prescription_sales]
    colors = [
        "#343a40",  # Gris oscuro para total
        "#28a745",  # Verde para venta libre (destacado)
        "#2c7be5",  # Azul para prescripción
    ]
    texts = [
        format_currency(total_sales),
        f"{format_currency(venta_libre_sales)}<br>{pct_vl:.1f}%",
        f"{format_currency(prescription_sales)}<br>{pct_rx:.1f}%",
    ]

    fig = go.Figure(
        go.Treemap(
            labels=labels,
            parents=parents,
            values=values,
            text=texts,
            marker={"colors": colors},
            textinfo="label+text",
            textfont={"size": 12},
            hovertemplate="<b>%{label}</b><br>%{text}<extra></extra>",
            branchvalues="total",
            maxdepth=-1,  # Expandir todos los niveles
        )
    )

    fig.update_layout(
        margin={"t": 5, "b": 5, "l": 5, "r": 5},
        height=280,
        paper_bgcolor="rgba(0,0,0,0)",
        font={"family": "Arial, sans-serif"},
    )

    return fig


def register_date_slider_callbacks(app):
    """
    Registrar callbacks para sincronizar el RangeSlider con el DatePickerRange.

    Similar al patrón de /prescription.
    """

    @app.callback(
        [
            Output("ventalibre-date-range", "start_date"),
            Output("ventalibre-date-range", "end_date"),
            Output("ventalibre-date-range", "min_date_allowed"),
            Output("ventalibre-date-range", "max_date_allowed"),
            Output("ventalibre-date-slider", "min"),
            Output("ventalibre-date-slider", "max"),
            Output("ventalibre-date-slider", "marks"),
            Output("ventalibre-date-slider", "value"),
        ],
        [
            Input("url", "pathname"),
            Input("auth-ready", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=False,
    )
    def initialize_date_range(pathname, auth_ready, auth_state, auth_tokens):
        """
        Inicializar DatePickerRange y RangeSlider con datos de la farmacia.

        Obtiene el rango de fechas disponibles desde el backend.
        """
        # Guard: Solo ejecutar en /ventalibre
        if pathname != "/ventalibre":
            raise PreventUpdate

        # Guard: Esperar auth-ready
        if not auth_ready:
            raise PreventUpdate

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Obtener auth headers
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            raise PreventUpdate

        try:
            # Obtener rango de fechas desde el backend
            response = request_coordinator.make_request(
                method="GET",
                endpoint="/api/v1/ventalibre/date-range",
                auth_headers=auth_headers,
            )

            if not response:
                # Fallback: últimos 90 días
                today = datetime.now().date()
                min_date = today - timedelta(days=365)
                max_date = today
            else:
                min_date_str = response.get("min_date")
                max_date_str = response.get("max_date")

                if min_date_str and max_date_str:
                    min_date = datetime.fromisoformat(min_date_str).date()
                    max_date = datetime.fromisoformat(max_date_str).date()
                else:
                    today = datetime.now().date()
                    min_date = today - timedelta(days=365)
                    max_date = today

            # Calcular rango en días
            total_days = (max_date - min_date).days

            # Por defecto, últimos 90 días
            default_start = max(min_date, max_date - timedelta(days=90))

            # Crear marks para el slider (cada 3 meses aproximadamente)
            marks = {}
            current = min_date
            while current <= max_date:
                day_index = (current - min_date).days
                marks[day_index] = current.strftime("%b %y")
                # Avanzar aproximadamente 3 meses
                if current.month <= 9:
                    current = current.replace(month=current.month + 3)
                else:
                    current = current.replace(year=current.year + 1, month=(current.month + 3) % 12 or 12)

            # Valores del slider
            slider_start = (default_start - min_date).days
            slider_end = total_days

            logger.info(
                f"[VENTALIBRE] Date range initialized: {min_date} to {max_date}, "
                f"default: {default_start} to {max_date}"
            )

            return (
                default_start.isoformat(),  # start_date
                max_date.isoformat(),        # end_date
                min_date.isoformat(),        # min_date_allowed
                max_date.isoformat(),        # max_date_allowed
                0,                           # slider min
                total_days,                  # slider max
                marks,                       # slider marks
                [slider_start, slider_end],  # slider value
            )

        except Exception as e:
            logger.error(f"[VENTALIBRE] Error initializing date range: {e}")
            # Fallback
            today = datetime.now().date()
            min_date = today - timedelta(days=365)
            max_date = today
            default_start = max_date - timedelta(days=90)
            total_days = 365

            return (
                default_start.isoformat(),
                max_date.isoformat(),
                min_date.isoformat(),
                max_date.isoformat(),
                0,
                total_days,
                {},
                [total_days - 90, total_days],
            )

    @app.callback(
        [
            Output("ventalibre-date-range", "start_date", allow_duplicate=True),
            Output("ventalibre-date-range", "end_date", allow_duplicate=True),
        ],
        [Input("ventalibre-date-slider", "value")],
        [
            State("ventalibre-date-range", "min_date_allowed"),
        ],
        prevent_initial_call=True,
    )
    def sync_slider_to_datepicker(slider_value, min_date_allowed):
        """Sincronizar slider con DatePicker cuando se mueve el slider."""
        if not slider_value or not min_date_allowed:
            raise PreventUpdate

        try:
            min_dt = datetime.fromisoformat(min_date_allowed).date()
            start_dt = min_dt + timedelta(days=slider_value[0])
            end_dt = min_dt + timedelta(days=slider_value[1])
            return start_dt.isoformat(), end_dt.isoformat()
        except Exception as e:
            logger.debug(f"[VENTALIBRE] Slider sync error: {e}")
            raise PreventUpdate
