"""
Context Navigation Callbacks (Issue #512)

Callbacks para navegación contextual desde menú derecho del treemap.
Permite navegación directa a otras secciones manteniendo contexto de filtros.

Flujo:
1. JavaScript (context_menu.js) captura right-click/long-press
2. JavaScript actualiza ventalibre-context-click-store
3. Callback open_context_menu() detecta cambio y abre modal
4. Usuario selecciona opción
5. Callback handle_navigation() navega a tab + aplica filtro

Sigue REGLA #7.6: Restaurar tokens desde auth-tokens-store antes de API calls.
Sigue REGLA #11: Un Input → Un callback.
"""

import logging
from datetime import datetime, timezone
from urllib.parse import urlencode

from dash import Input, Output, State, callback, ctx, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.context_menu import create_context_category_badge
from components.ventalibre.categories import get_principal_info
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.config import BACKEND_URL
from utils.pharmacy_context import get_current_pharmacy_id
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_context_navigation_callbacks(app):
    """Register context navigation callbacks."""

    # =========================================================================
    # CALLBACK 1: Detectar click derecho (desde JavaScript) y abrir modal
    # =========================================================================
    @app.callback(
        [
            Output("ventalibre-context-menu-modal", "is_open"),
            Output("ventalibre-context-menu-title", "children"),
            Output("ventalibre-context-category-badge", "children"),
            Output("ventalibre-navigation-context-store", "data"),
        ],
        Input("ventalibre-context-interval", "n_intervals"),
        [
            State("ventalibre-context-click-store", "data"),
            State("ventalibre-context-menu-modal", "is_open"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def poll_context_click(n_intervals, context_data, is_open, auth_state):
        """
        Poll for context click events from JavaScript.

        JavaScript stores click data in context-click-store.
        This callback detects changes and opens the context menu modal.

        Args:
            n_intervals: Interval counter
            context_data: Click data from JavaScript
            is_open: Current modal state
            auth_state: Auth state

        Returns:
            Tuple: (is_open, title, badge, navigation_context)
        """
        # Guard: Skip if not authenticated
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Guard: No context data
        if not context_data:
            raise PreventUpdate

        # Guard: Check timestamp to avoid reprocessing old clicks
        timestamp = context_data.get("timestamp", 0)
        now = datetime.now(timezone.utc).timestamp() * 1000  # JS timestamp is in ms
        age_ms = now - timestamp

        # Ignore clicks older than 500ms (already processed)
        if age_ms > 500:
            raise PreventUpdate

        # Guard: Skip aggregated categories (e.g., "Otras Necesidades")
        if context_data.get("is_aggregated", False):
            logger.debug("[CONTEXT] Skipping aggregated category")
            raise PreventUpdate

        # Extract category info
        category_id = context_data.get("category_id", "")
        category_label = context_data.get("category_label", "")
        count = context_data.get("count", 0)
        percentage = context_data.get("percentage", 0)

        # Get display name from categories
        principal_info = get_principal_info(category_id)
        display_name = principal_info.get("name", category_label or category_id)

        logger.info(
            "context_navigation.open_menu",
            extra={
                "category_id": category_id,
                "category_label": display_name,
                "count": count,
            },
        )

        # Create badge component
        badge = create_context_category_badge(
            category_label=display_name,
            count=count,
            percentage=percentage,
        )

        # Build navigation context for use by navigation callback
        nav_context = {
            "category_id": category_id,
            "category_label": display_name,
            "count": count,
            "percentage": percentage,
            "timestamp": timestamp,
        }

        return True, f"Acciones: {display_name}", badge, nav_context

    # =========================================================================
    # CALLBACK 2: Manejar selección de opción de navegación
    # =========================================================================
    @app.callback(
        [
            Output("ventalibre-tabs", "active_tab"),
            Output("ventalibre-context-menu-modal", "is_open", allow_duplicate=True),
            Output("url", "search", allow_duplicate=True),
            # Navigation outputs for different tabs
            Output("ventalibre-selected-category-store", "data", allow_duplicate=True),
            Output("ventalibre-brands-necesidad-dropdown", "value", allow_duplicate=True),
        ],
        [
            Input("ventalibre-context-option-products", "n_clicks"),
            Input("ventalibre-context-option-evolution", "n_clicks"),
            Input("ventalibre-context-option-brands", "n_clicks"),
        ],
        [
            State("ventalibre-navigation-context-store", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_navigation(n_products, n_evolution, n_brands, nav_context, auth_state):
        """
        Handle navigation option selection from context menu.

        Navigates to the appropriate tab and applies category filter.

        Args:
            n_products: Click on "Ver productos" option
            n_evolution: Click on "Ver tendencias" option
            n_brands: Click on "Ver análisis de marcas" option
            nav_context: Navigation context from context-click-store
            auth_state: Auth state

        Returns:
            Tuple: (active_tab, modal_is_open, url_search, category_store, brands_dropdown)
        """
        # Guard: No trigger
        if not ctx.triggered_id:
            raise PreventUpdate

        # Guard: Not authenticated
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Guard: No navigation context
        if not nav_context:
            logger.warning("[CONTEXT] handle_navigation: No nav_context")
            raise PreventUpdate

        category_id = nav_context.get("category_id", "")
        category_label = nav_context.get("category_label", "")

        logger.info(
            "context_navigation.navigate",
            extra={
                "trigger": ctx.triggered_id,
                "category_id": category_id,
            },
        )

        # Common: Close modal
        close_modal = False

        # Build URL search params for deep-linking
        def build_search(tab: str, category: str) -> str:
            params = {"tab": tab, "category": category}
            return "?" + urlencode(params)

        # Handle each option
        if ctx.triggered_id == "ventalibre-context-option-products":
            # Navigate to Tab 4 (Producto y Surtido) + set category filter
            category_data = {"level": 1, "l1": category_id, "l2": None}
            return (
                "tab-producto",
                close_modal,
                build_search("producto", category_id),
                category_data,
                no_update,
            )

        elif ctx.triggered_id == "ventalibre-context-option-evolution":
            # Navigate to Tab 2 (Brands) + set category dropdown
            # Tab 2 has evolution chart within brands analysis
            return (
                "tab-brands",
                close_modal,
                build_search("brands", category_id),
                no_update,
                category_id,  # Set brands dropdown
            )

        elif ctx.triggered_id == "ventalibre-context-option-brands":
            # Navigate to Tab 2 (Brands) + set category dropdown
            return (
                "tab-brands",
                close_modal,
                build_search("brands", category_id),
                no_update,
                category_id,  # Set brands dropdown
            )

        else:
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 3: Exportar CSV de productos de la categoría
    # =========================================================================
    @app.callback(
        [
            Output("ventalibre-context-csv-download", "data"),
            Output("ventalibre-context-menu-modal", "is_open", allow_duplicate=True),
        ],
        Input("ventalibre-context-option-export", "n_clicks"),
        [
            State("ventalibre-navigation-context-store", "data"),
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def export_category_csv(n_clicks, nav_context, start_date, end_date, auth_state, auth_tokens):
        """
        Export products of selected category as CSV.

        Calls backend endpoint to generate CSV with products.

        Args:
            n_clicks: Click on export option
            nav_context: Navigation context with category
            start_date: Start date filter
            end_date: End date filter
            auth_state: Auth state
            auth_tokens: Auth tokens for API call

        Returns:
            Tuple: (download_data, modal_is_open)
        """
        import requests

        if not n_clicks:
            raise PreventUpdate

        # Guard: Not authenticated
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Guard: No navigation context
        if not nav_context:
            logger.warning("[CONTEXT] export_csv: No nav_context")
            raise PreventUpdate

        # REGLA #7.6: Get auth headers for multi-worker
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[CONTEXT] export_csv: No auth headers")
            raise PreventUpdate

        category_id = nav_context.get("category_id", "")
        category_label = nav_context.get("category_label", category_id)

        try:
            pharmacy_id = get_current_pharmacy_id()
        except ValueError:
            logger.warning("[CONTEXT] export_csv: No pharmacy_id")
            raise PreventUpdate

        logger.info(
            "context_navigation.export_csv",
            extra={
                "category_id": category_id,
                "pharmacy_id": pharmacy_id,
            },
        )

        try:
            # Call backend export endpoint directly with requests for file download
            response = requests.get(
                f"{BACKEND_URL}/api/v1/ventalibre/products/export",
                params={
                    "category": category_id,
                    "date_from": start_date,
                    "date_to": end_date,
                    "format": "csv",
                },
                headers=auth_headers,
                timeout=60,
            )

            if response.status_code != 200:
                logger.error(f"[CONTEXT] export_csv: API returned {response.status_code}")
                raise PreventUpdate

            # Build filename
            safe_category = category_label.replace(" ", "_").lower()[:30]
            filename = f"productos_{safe_category}_{start_date}_{end_date}.csv"

            # Return download data and close modal
            return {
                "content": response.text,
                "filename": filename,
                "type": "text/csv",
            }, False

        except requests.RequestException as e:
            logger.error(
                "context_navigation.export_csv_error",
                extra={"error": str(e)},
                exc_info=True,
            )
            raise PreventUpdate

    # =========================================================================
    # CALLBACK 4: Parse URL params for deep-linking (page load)
    # =========================================================================
    @app.callback(
        [
            Output("ventalibre-tabs", "active_tab", allow_duplicate=True),
            Output("ventalibre-selected-category-store", "data", allow_duplicate=True),
            Output("ventalibre-brands-necesidad-dropdown", "value", allow_duplicate=True),
        ],
        Input("url", "search"),
        State("url", "pathname"),
        prevent_initial_call=True,
    )
    def parse_url_params(search, pathname):
        """
        Parse URL query parameters for deep-linking.

        Supports URLs like:
        - /ventalibre?tab=brands&category=dermocosmetica
        - /ventalibre?tab=producto&category=respiratorio

        Args:
            search: URL query string (e.g., "?tab=brands&category=...")
            pathname: Current pathname

        Returns:
            Tuple: (active_tab, category_store, brands_dropdown)
        """
        # Guard: Only on /ventalibre
        if pathname != "/ventalibre":
            raise PreventUpdate

        # Guard: No search params
        if not search or search == "?":
            raise PreventUpdate

        # Parse query string
        from urllib.parse import parse_qs

        try:
            params = parse_qs(search.lstrip("?"))
            tab = params.get("tab", [None])[0]
            category = params.get("category", [None])[0]
        except Exception:
            raise PreventUpdate

        # Guard: No valid params
        if not tab:
            raise PreventUpdate

        logger.info(
            "context_navigation.parse_url",
            extra={"tab": tab, "category": category},
        )

        # Map tab param to tab_id
        tab_map = {
            "analisis": "tab-analisis",
            "brands": "tab-brands",
            "inventario": "tab-inventario",
            "producto": "tab-producto",
            "insights": "tab-insights",
        }

        active_tab = tab_map.get(tab)
        if not active_tab:
            raise PreventUpdate

        # Apply category filter if provided
        if category:
            if tab == "producto":
                category_data = {"level": 1, "l1": category, "l2": None}
                return active_tab, category_data, no_update
            elif tab == "brands":
                return active_tab, no_update, category
            else:
                return active_tab, no_update, no_update
        else:
            return active_tab, no_update, no_update

    # =========================================================================
    # CALLBACK 5: Clear context store after processing
    # =========================================================================
    @app.callback(
        Output("ventalibre-context-click-store", "data", allow_duplicate=True),
        Input("ventalibre-context-menu-modal", "is_open"),
        prevent_initial_call=True,
    )
    def clear_context_store(is_open):
        """
        Clear context click store when modal is closed.

        Prevents reprocessing of old click events.
        """
        if not is_open:
            return None
        raise PreventUpdate
