"""
Callbacks para Tab "Categorías y Marcas" (Issue #493)

Análisis de marcas por categoría NECESIDAD con métricas consultoría:
- HHI (concentración de mercado)
- Cuadrante de Valor (Margen vs Volumen)
- Brand Duel Mode (comparación lado a lado)
- Evolución de cuota de mercado
- Boxplot de precios (canibalización)

Sigue REGLA #7.6: Restaurar tokens desde auth-tokens-store antes de API calls.
Sigue REGLA #11: Un Input → Un Callback (consolidado en _register_brands_data_callback).
"""

import logging

import plotly.express as px
import plotly.graph_objects as go
from dash import Input, Output, State, ctx, no_update
from dash.exceptions import PreventUpdate

from components.ventalibre.brand_analysis import (
    create_brand_duel_display,
    create_brands_table,
    create_hhi_display,
    create_hhi_matrix_figure,
    create_hhi_matrix_table,
    create_market_evolution_figure,
    create_price_boxplot_figure,
    create_value_quadrant_figure,
)
from components.ventalibre.categories import PRINCIPAL_CATEGORIES
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.pharmacy_context import get_current_pharmacy_id
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_brands_callbacks(app):
    """
    Registrar todos los callbacks del tab Categorías y Marcas.

    Args:
        app: Instancia de la aplicación Dash
    """
    _register_category_dropdown_callback(app)
    _register_chart_toggle_callback(app)
    _register_brands_data_callback(app)  # CONSOLIDADO: Un callback, múltiples outputs
    _register_brand_duel_toggle_callback(app)
    _register_brand_duel_comparison_callback(app)
    _register_hhi_modal_callback(app)  # Modal educativo HHI
    _register_hhi_matrix_callback(app)  # Scatter plot HHI × Margen (Issue #539)


def _register_category_dropdown_callback(app):
    """Callback para cargar opciones del dropdown de categorías NECESIDAD."""

    @app.callback(
        Output("ventalibre-brands-necesidad-dropdown", "options"),
        Input("ventalibre-tabs", "active_tab"),
        State("auth-state", "data"),
        State("auth-tokens-store", "data"),
        prevent_initial_call=True,
    )
    def load_category_options(active_tab, auth_state, auth_tokens):
        """Cargar opciones del dropdown cuando se activa la tab."""
        if active_tab != "tab-brands":
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            logger.debug("[BRANDS] User not authenticated - skipping dropdown load")
            raise PreventUpdate

        # No se necesita restaurar tokens porque este callback no hace API calls
        # Solo usa datos locales de PRINCIPAL_CATEGORIES

        # Usar categorías principales definidas
        options = [
            {"label": info["name"], "value": key}
            for key, info in PRINCIPAL_CATEGORIES.items()
            if key not in ["otros", "interno_no_venta"]  # Excluir categorías no útiles
        ]

        # Ordenar alfabéticamente
        options.sort(key=lambda x: x["label"])

        return options


def _register_chart_toggle_callback(app):
    """Callback para toggle treemap/barras en gráfico de marcas."""

    @app.callback(
        [
            Output("ventalibre-brands-treemap-btn", "outline"),
            Output("ventalibre-brands-bars-btn", "outline"),
            Output("ventalibre-brands-chart-type-store", "data"),
        ],
        [
            Input("ventalibre-brands-treemap-btn", "n_clicks"),
            Input("ventalibre-brands-bars-btn", "n_clicks"),
        ],
        State("ventalibre-brands-chart-type-store", "data"),
        prevent_initial_call=True,
    )
    def toggle_chart_type(treemap_clicks, bars_clicks, current_type):
        """Toggle entre treemap y barras."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        if trigger_id == "ventalibre-brands-treemap-btn":
            return False, True, "treemap"
        elif trigger_id == "ventalibre-brands-bars-btn":
            return True, False, "bars"

        return no_update, no_update, no_update


def _register_brands_data_callback(app):
    """
    Callback CONSOLIDADO para cargar todos los datos de marcas.

    REGLA #11: Un Input → Un Callback.
    El Input "ventalibre-brands-necesidad-dropdown" solo aparece aquí.
    Este callback hace las 4 llamadas API y actualiza todos los gráficos.
    """

    @app.callback(
        [
            # Outputs del callback original de brands
            Output("ventalibre-brands-data-store", "data"),
            Output("ventalibre-hhi-indicator", "children"),
            Output("ventalibre-brands-chart", "figure"),
            Output("ventalibre-brands-table-container", "children"),
            Output("ventalibre-brand-duel-dropdown-a", "options"),
            Output("ventalibre-brand-duel-dropdown-b", "options"),
            # Outputs adicionales consolidados (antes en callbacks separados)
            Output("ventalibre-value-quadrant-chart", "figure"),
            Output("ventalibre-market-evolution-chart", "figure"),
            Output("ventalibre-price-boxplot-chart", "figure"),
        ],
        [
            Input("ventalibre-brands-necesidad-dropdown", "value"),
            # Issue #537 FIX: Añadir fechas como Inputs para recargar cuando cambian
            Input("ventalibre-date-range", "start_date"),
            Input("ventalibre-date-range", "end_date"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("ventalibre-brands-chart-type-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_all_brands_data(necesidad, start_date, end_date, auth_state, auth_tokens, chart_type):
        """
        Cargar todos los datos de marcas para la categoría seleccionada.

        Este callback consolidado hace 4 llamadas API:
        1. /brands/{necesidad} - Datos principales + HHI
        2. /value-quadrant/{necesidad} - Cuadrante de valor
        3. /market-share-evolution/{necesidad} - Evolución temporal
        4. /price-distribution/{necesidad} - Distribución de precios
        """
        # Empty outputs tuple for early returns
        empty_outputs = _create_empty_outputs()

        if not necesidad:
            return empty_outputs

        if not is_user_authenticated(auth_state):
            logger.debug("[BRANDS] User not authenticated - skipping data load")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRANDS] No auth headers available - skipping API calls")
            return empty_outputs

        pharmacy_id = get_current_pharmacy_id()
        if not pharmacy_id:
            raise PreventUpdate

        # Preparar parámetros comunes
        params = {"pharmacy_id": str(pharmacy_id)}
        if start_date:
            params["date_from"] = start_date
        if end_date:
            params["date_to"] = end_date

        try:
            # API Call 1: Brands + HHI
            brands_response = request_coordinator.make_request(
                f"/api/v1/ventalibre/brands/{necesidad}",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,
            )

            # API Call 2: Value Quadrant
            quadrant_response = request_coordinator.make_request(
                f"/api/v1/ventalibre/value-quadrant/{necesidad}",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,
            )

            # API Call 3: Market Evolution
            evolution_response = request_coordinator.make_request(
                f"/api/v1/ventalibre/market-share-evolution/{necesidad}",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,
            )

            # API Call 4: Price Distribution
            price_response = request_coordinator.make_request(
                f"/api/v1/ventalibre/price-distribution/{necesidad}",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,
            )

            # Process brands response
            if not brands_response or "error" in brands_response:
                logger.warning(f"[BRANDS] Error loading brands: {brands_response}")
                return empty_outputs

            brands = brands_response.get("brands", [])
            hhi = brands_response.get("hhi", 0)
            hhi_interpretation = brands_response.get("hhi_interpretation", {})

            # HHI display con diagnóstico cruzado (HHI × Margen)
            hhi_data = {"hhi": hhi, "hhi_interpretation": hhi_interpretation}
            hhi_display = create_hhi_display(hhi_data, brands_data=brands)

            # Chart (treemap o barras)
            chart_type = chart_type or "treemap"
            if chart_type == "treemap":
                brands_fig = _create_brands_treemap(brands)
            else:
                brands_fig = _create_brands_bar_chart(brands)

            # Table
            table = create_brands_table(brands)

            # Brand options for duel
            brand_options = [
                {"label": b["brand"].title(), "value": b["brand"]}
                for b in brands[:20]  # Limitar a top 20
            ]

            # Process quadrant response
            if quadrant_response and "error" not in quadrant_response:
                quadrant_fig = create_value_quadrant_figure(quadrant_response)
            else:
                quadrant_fig = _create_empty_figure("Error cargando cuadrante")

            # Process evolution response
            if evolution_response and "error" not in evolution_response:
                evolution_fig = create_market_evolution_figure(evolution_response)
            else:
                evolution_fig = _create_empty_figure("Error cargando evolución")

            # Process price response
            if price_response and "error" not in price_response:
                price_fig = create_price_boxplot_figure(price_response)
            else:
                price_fig = _create_empty_figure("Error cargando precios")

            return (
                brands_response,      # Store data
                hhi_display,          # HHI indicator
                brands_fig,           # Brands chart
                table,                # Brands table
                brand_options,        # Duel dropdown A
                brand_options,        # Duel dropdown B
                quadrant_fig,         # Value quadrant
                evolution_fig,        # Market evolution
                price_fig,            # Price boxplot
            )

        except Exception as e:
            logger.exception(f"[BRANDS] Exception loading brands data: {e}")
            return empty_outputs


def _create_empty_outputs():
    """Crear tuple de outputs vacíos para early returns."""
    empty_fig = _create_empty_figure("Selecciona una categoría para ver el análisis")
    empty_hhi = create_hhi_display({})
    empty_table = create_brands_table([])

    return (
        None,           # Store data
        empty_hhi,      # HHI indicator
        empty_fig,      # Brands chart
        empty_table,    # Brands table
        [],             # Duel dropdown A
        [],             # Duel dropdown B
        empty_fig,      # Value quadrant
        empty_fig,      # Market evolution
        empty_fig,      # Price boxplot
    )


def _register_brand_duel_toggle_callback(app):
    """Callback para toggle del collapse Brand Duel."""

    @app.callback(
        Output("ventalibre-brand-duel-collapse", "is_open"),
        Input("ventalibre-brand-duel-toggle-btn", "n_clicks"),
        State("ventalibre-brand-duel-collapse", "is_open"),
        prevent_initial_call=True,
    )
    def toggle_brand_duel(n_clicks, is_open):
        """Toggle collapse de Brand Duel."""
        if n_clicks:
            return not is_open
        return is_open


def _register_brand_duel_comparison_callback(app):
    """Callback para cargar comparación de marcas."""

    @app.callback(
        Output("ventalibre-brand-duel-content", "children"),
        [
            Input("ventalibre-brand-duel-dropdown-a", "value"),
            Input("ventalibre-brand-duel-dropdown-b", "value"),
            # Issue #537 FIX: Añadir fechas como Inputs para recargar cuando cambian
            Input("ventalibre-date-range", "start_date"),
            Input("ventalibre-date-range", "end_date"),
        ],
        [
            State("ventalibre-brands-necesidad-dropdown", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_brand_duel(brand_a, brand_b, start_date, end_date, necesidad, auth_state, auth_tokens):
        """Cargar comparación de dos marcas."""
        if not brand_a or not brand_b:
            return create_brand_duel_display({})

        if brand_a == brand_b:
            return create_brand_duel_display({})

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[BRANDS] No auth headers available for brand duel")
            return create_brand_duel_display({})

        pharmacy_id = get_current_pharmacy_id()
        if not pharmacy_id:
            raise PreventUpdate

        params = {
            "pharmacy_id": str(pharmacy_id),
            "brand_a": brand_a,
            "brand_b": brand_b,
            "necesidad": necesidad,
        }
        if start_date:
            params["date_from"] = start_date
        if end_date:
            params["date_to"] = end_date

        try:
            response = request_coordinator.make_request(
                "/api/v1/ventalibre/brand-duel",
                method="GET",
                params=params,
                timeout=30,
                auth_headers=auth_headers,
            )

            if not response or "error" in response:
                return create_brand_duel_display({})

            return create_brand_duel_display(response)

        except Exception as e:
            logger.exception(f"[BRANDS] Exception loading duel: {e}")
            return create_brand_duel_display({})


# =============================================================================
# Helper Functions
# =============================================================================


def _create_empty_figure(message: str) -> go.Figure:
    """Crear figura vacía con mensaje."""
    fig = go.Figure()
    fig.add_annotation(
        text=message,
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )
    fig.update_layout(
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        plot_bgcolor="white",
        margin=dict(t=20, b=20, l=20, r=20),
    )
    return fig


def _create_brands_treemap(brands: list) -> go.Figure:
    """Crear treemap de marcas."""
    if not brands:
        return _create_empty_figure("No hay datos de marcas")

    # Preparar datos
    labels = [b["brand"].title() for b in brands[:15]]  # Limitar a 15
    values = [b.get("sales", 0) for b in brands[:15]]
    parents = [""] * len(labels)

    # Texto personalizado
    custom_text = [
        f"{b.get('share', 0):.1f}%<br>{b.get('sales', 0):,.0f}€"
        for b in brands[:15]
    ]

    fig = go.Figure(
        go.Treemap(
            labels=labels,
            parents=parents,
            values=values,
            textinfo="label+text",
            text=custom_text,
            textfont=dict(size=12),
            marker=dict(
                colors=px.colors.qualitative.Set2[:len(labels)],
                line=dict(width=2, color="white"),
            ),
            hovertemplate="<b>%{label}</b><br>Ventas: %{value:,.0f}€<br>Cuota: %{text}<extra></extra>",
        )
    )

    fig.update_layout(
        margin=dict(t=30, b=10, l=10, r=10),
        showlegend=False,
    )

    return fig


def _create_brands_bar_chart(brands: list) -> go.Figure:
    """Crear gráfico de barras horizontal de marcas."""
    if not brands:
        return _create_empty_figure("No hay datos de marcas")

    # Preparar datos (invertir para que el top esté arriba)
    brands_top = brands[:15][::-1]
    labels = [b["brand"].title() for b in brands_top]
    values = [b.get("sales", 0) for b in brands_top]

    fig = go.Figure(
        go.Bar(
            x=values,
            y=labels,
            orientation="h",
            text=[f"{v:,.0f}€" for v in values],
            textposition="outside",
            marker=dict(color=px.colors.qualitative.Set2[0]),
            hovertemplate="<b>%{y}</b><br>Ventas: %{x:,.0f}€<extra></extra>",
        )
    )

    fig.update_layout(
        xaxis_title="Ventas (€)",
        yaxis_title="",
        margin=dict(t=30, b=50, l=100, r=50),
        plot_bgcolor="white",
        xaxis=dict(gridcolor="#eee", zeroline=False),
        showlegend=False,
    )

    return fig


def _register_hhi_modal_callback(app):
    """
    Callback para abrir/cerrar el modal educativo de HHI.

    El modal explica el diagnóstico cruzado HHI × Margen con:
    - Los 5 diagnósticos posibles
    - El trade-off Generalista vs Especialista
    - El checklist del especialista
    """

    @app.callback(
        Output("ventalibre-hhi-education-modal", "is_open"),
        [
            Input("ventalibre-hhi-modal-open", "n_clicks"),
            Input("ventalibre-hhi-modal-close", "n_clicks"),
        ],
        State("ventalibre-hhi-education-modal", "is_open"),
        prevent_initial_call=True,
    )
    def toggle_hhi_modal(open_clicks, close_clicks, is_open):
        """Abrir o cerrar el modal educativo de HHI."""
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        if trigger_id == "ventalibre-hhi-modal-open":
            return True
        elif trigger_id == "ventalibre-hhi-modal-close":
            return False

        return is_open


def _register_hhi_matrix_callback(app):
    """
    Callback para cargar el scatter plot HHI × Margen (Issue #539).

    Carga datos de todas las categorías con su HHI y margen promedio
    para visualizar la estrategia de concentración de la farmacia.
    """

    @app.callback(
        [
            Output("ventalibre-hhi-matrix-chart", "figure"),
            Output("ventalibre-hhi-matrix-table", "children"),
        ],
        Input("ventalibre-tabs", "active_tab"),
        [
            State("ventalibre-date-range", "start_date"),
            State("ventalibre-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_hhi_matrix(active_tab, start_date, end_date, auth_state, auth_tokens):
        """
        Cargar datos del HHI Matrix cuando se activa la tab de marcas.

        Se carga al entrar a la tab (no en cada cambio de categoría)
        porque muestra datos de TODAS las categorías.
        """
        # Solo cargar cuando se activa la tab de marcas
        if active_tab != "tab-brands":
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            logger.debug("[HHI-MATRIX] User not authenticated - skipping load")
            raise PreventUpdate

        # REGLA #7.6: Multi-Worker Token Restoration - pasar auth_headers explícitamente
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[HHI-MATRIX] No auth headers available - skipping API calls")
            empty_fig = _create_empty_figure("Sesión no válida")
            empty_table = create_hhi_matrix_table({})
            return empty_fig, empty_table

        pharmacy_id = get_current_pharmacy_id()
        if not pharmacy_id:
            raise PreventUpdate

        # Preparar parámetros
        params = {"pharmacy_id": str(pharmacy_id)}
        if start_date:
            params["date_from"] = start_date
        if end_date:
            params["date_to"] = end_date

        try:
            # Llamar al endpoint HHI Matrix
            response = request_coordinator.make_request(
                "/api/v1/ventalibre/hhi-matrix",
                method="GET",
                params=params,
                timeout=45,  # Puede tardar un poco más al calcular todas las categorías
                auth_headers=auth_headers,
            )

            if not response or "error" in response:
                logger.warning(f"[HHI-MATRIX] Error loading data: {response}")
                empty_fig = _create_empty_figure("Error cargando datos")
                empty_table = create_hhi_matrix_table({})
                return empty_fig, empty_table

            # Crear visualizaciones
            figure = create_hhi_matrix_figure(response)
            table = create_hhi_matrix_table(response)

            logger.info(
                "[HHI-MATRIX] Data loaded: %d categories, avg_hhi=%.1f",
                len(response.get("categories", [])),
                response.get("summary", {}).get("avg_hhi", 0),
            )

            return figure, table

        except Exception as e:
            logger.exception(f"[HHI-MATRIX] Exception loading data: {e}")
            empty_fig = _create_empty_figure("Error cargando datos")
            empty_table = create_hhi_matrix_table({})
            return empty_fig, empty_table
