"""
Callbacks para Dashboard Venta Libre (Issue #461, #471, #491, #493, #494, #509, #511, #512, #514)

Modulos:
- data_loading: Carga inicial de datos y KPIs (Tab Análisis)
- treemap: Interaccion con treemap de NECESIDAD (Tab Análisis)
- products: Tabla de productos con paginacion y busqueda (Tab Análisis)
- feedback: Modal de reporte de errores (Tab Análisis)
- evolution: Evolución temporal de ventas + YoY + Top Contributors (Tab Análisis - #491)
- brands: Análisis de marcas por categoría (Tab Categorías y Marcas - #493)
- product_analysis: Ficha producto, MBA, recomendaciones (Tab Producto y Surtido - #494)
- executive_summary: Resumen ejecutivo generado (NLG) (Issue #509)
- reports: Sistema de Doble Reporte - PDF Dirección + Excel Suelo (Issue #511)
- context_navigation: Navegación contextual desde menú derecho treemap (Issue #512)
- roi_tracker: Widget ROI Tracker - Feedback Loop de Acciones (Issue #514)

Nota: Tab Gestión eliminado (Issue #488) - funcionalidades movidas a /admin.
"""

from callbacks.ventalibre.brands import register_brands_callbacks
from callbacks.ventalibre.evolution import register_evolution_callbacks
from callbacks.ventalibre.context_navigation import register_context_navigation_callbacks
from callbacks.ventalibre.data_loading import (
    register_context_treemap_callbacks,
    register_data_loading_callbacks,
    register_date_slider_callbacks,
)
from callbacks.ventalibre.executive_summary import register_executive_summary_callbacks
from callbacks.ventalibre.feedback import register_feedback_callbacks
from callbacks.ventalibre.l2_drilldown import register_l2_drilldown_callbacks
from callbacks.ventalibre.product_analysis import register_product_analysis_callbacks
from callbacks.ventalibre.products import register_products_callbacks
from callbacks.ventalibre.reports import register_reports_callbacks
from callbacks.ventalibre.roi_tracker import register_roi_tracker_callbacks
from callbacks.ventalibre.treemap import register_treemap_callbacks


def register_ventalibre_callbacks(app):
    """
    Registrar todos los callbacks del dashboard venta libre.

    Args:
        app: Instancia de la aplicacion Dash
    """
    # Tab Análisis
    register_data_loading_callbacks(app)
    register_context_treemap_callbacks(app)  # Treemap de contexto (cabecera)
    register_date_slider_callbacks(app)      # Sincronización slider <-> datepicker
    register_treemap_callbacks(app)
    register_products_callbacks(app)
    register_feedback_callbacks(app)

    # Evolución Temporal (Issue #491)
    register_evolution_callbacks(app)

    # L2 Drill-Down (Issue #505)
    register_l2_drilldown_callbacks(app)

    # Tab Categorías y Marcas (Issue #493)
    register_brands_callbacks(app)

    # Tab Producto y Surtido (Issue #494)
    register_product_analysis_callbacks(app)

    # Resumen Ejecutivo NLG (Issue #509)
    register_executive_summary_callbacks(app)

    # Sistema de Doble Reporte (Issue #511)
    register_reports_callbacks(app)

    # Navegación Contextual (Issue #512)
    register_context_navigation_callbacks(app)

    # ROI Tracker Widget (Issue #514)
    register_roi_tracker_callbacks(app)


__all__ = ["register_ventalibre_callbacks"]
