# frontend/callbacks/upload.py
"""
Callbacks para upload con monitoreo en tiempo real.
Incluye seguimiento de progreso, historia de uploads y actualización automática de estado.

Nota: Los componentes UI han sido extraídos a components/upload_progress.py
para mejorar modularidad y reducir tamaño del archivo.
"""

import base64
import logging
import os
from datetime import datetime

import dash_bootstrap_components as dbc
import requests
from components.empty_state import empty_upload_history_state
from components.toast_manager import error_toast, success_toast, warning_toast
# Componentes UI extraídos a módulo separado (Issue: refactorización archivos largos)
from components.upload_progress import (
    create_delete_preview,
    create_empty_history_message,
    create_enrichment_progress_component,
    create_enrichment_success_toast,
    create_error_progress_card,
    create_final_progress_summary,
    create_progress_card,
    format_timestamp,
    load_upload_history,
)
from dash import ALL, MATCH, Input, Output, State, ctx, html, no_update
from dash.exceptions import PreventUpdate
from utils.api_client import backend_client
from utils.helpers import format_currency, format_date_spanish, format_number
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

# Configuración
UPLOAD_CHECK_INTERVAL = 2000  # 2 segundos
HISTORY_REFRESH_INTERVAL = 30000  # 30 segundos
MAX_RETRIES = 3


# ========================================
# HELPER FUNCTIONS para monitor callback
# ========================================
# Movidas a module-level para testabilidad (Issue #300 - Code Review)


def detect_file_type_from_columns(decoded_content: bytes, filename: str) -> str:
    """
    Detecta automáticamente si el archivo es de INVENTARIO o VENTAS
    basándose en las columnas del archivo.

    Issue #476: Detección automática de tipo de archivo.

    Args:
        decoded_content: Contenido decodificado del archivo
        filename: Nombre del archivo

    Returns:
        "inventory" si es archivo de inventario, "sales" si es de ventas
    """
    import io
    import re

    try:
        # Intentar leer como CSV con diferentes encodings
        content_str = None
        for encoding in ["utf-8", "latin-1", "cp1252"]:
            try:
                content_str = decoded_content.decode(encoding)
                break
            except UnicodeDecodeError:
                continue

        if not content_str:
            logger.warning("[DETECT_TYPE] No se pudo decodificar el archivo")
            return "sales"  # Default a ventas

        # Leer solo las primeras líneas para detectar columnas
        lines = content_str.split("\n")[:5]
        if not lines:
            return "sales"

        # Detectar separador (punto y coma o coma)
        first_line = lines[0]
        separator = ";" if ";" in first_line else ","

        # Obtener nombres de columnas
        columns = [col.strip().strip('"').strip("'") for col in first_line.split(separator)]

        # Normalizar columnas (mayúsculas, sin acentos, sin espacios)
        def normalize(col: str) -> str:
            col = col.upper()
            col = re.sub(r"[^A-Z0-9]", "", col)
            # Remover acentos comunes
            replacements = {
                "Á": "A", "É": "E", "Í": "I", "Ó": "O", "Ú": "U",
                "Ñ": "N", "Ü": "U"
            }
            for accent, letter in replacements.items():
                col = col.replace(accent, letter)
            return col

        normalized_columns = [normalize(col) for col in columns]

        logger.info(f"[DETECT_TYPE] Columnas detectadas: {columns[:10]}...")
        logger.info(f"[DETECT_TYPE] Columnas normalizadas: {normalized_columns[:10]}...")

        # Indicadores de INVENTARIO (Farmanager: Informe_Articulos)
        # Columnas típicas: Código, Descripción, Stock, PCM, IVA/IGIC, Ultima Compra, Ultima venta, P.V.P.
        inventory_indicators = {
            "STOCK", "PCM", "PVP", "EXISTENCIAS", "UNIDADES",
            "ULTIMACOMPRA", "ULTIMAVENTA", "PRECIOCOSTE", "COSTE",
            "IVAIGIC", "PRECIOVENTA"
        }

        # Indicadores de VENTAS
        # Columnas típicas: Fecha, Código, Producto, Cantidad, Importe, PVP
        sales_indicators = {
            "FECHAVENTA", "FECHAHORA", "CANTIDAD", "IMPORTE", "IMPORTETOTAL",
            "TOTALIMPORTE", "UNIDADESVENDIDAS", "VENTASNETAS", "FECHAOPERACION",
            "NUMTICKET", "TICKET", "NUMVENTA", "OPERACION"
        }

        # Contar matches
        inventory_matches = sum(1 for col in normalized_columns if col in inventory_indicators)
        sales_matches = sum(1 for col in normalized_columns if col in sales_indicators)

        # También verificar patrones parciales
        for col in normalized_columns:
            # Patrones de inventario
            if "STOCK" in col or "EXISTENCIA" in col:
                inventory_matches += 1
            if "ULTIMACOMPRA" in col or "ULTIMAVENTA" in col:
                inventory_matches += 1
            # Patrones de ventas
            if "FECHA" in col and ("VENTA" in col or "HORA" in col or "OPERACION" in col):
                sales_matches += 1
            if "TICKET" in col or "NUMVENTA" in col:
                sales_matches += 1

        logger.info(f"[DETECT_TYPE] Matches - Inventario: {inventory_matches}, Ventas: {sales_matches}")

        # Determinar tipo basándose en matches
        # Si hay más indicadores de inventario que de ventas, es inventario
        if inventory_matches > sales_matches and inventory_matches >= 2:
            logger.info(f"[DETECT_TYPE] Archivo detectado como INVENTARIO (matches: {inventory_matches})")
            return "inventory"
        else:
            logger.info(f"[DETECT_TYPE] Archivo detectado como VENTAS (matches: {sales_matches})")
            return "sales"

    except Exception as e:
        logger.error(f"[DETECT_TYPE] Error detectando tipo de archivo: {e}")
        return "sales"  # Default a ventas en caso de error


def _handle_navigation_cleanup(pathname):
    """
    Helper: Maneja limpieza cuando usuario navega fuera de /upload.
    Returns tuple: (children, disabled, toast, state, style)
    """
    logger.info(f"[CLEANUP] Navegando fuera de upload ({pathname}). Deshabilitando polling.")
    return (
        html.Div(),  # Limpiar progreso
        True,  # DESHABILITAR interval
        no_update,  # No toast
        no_update,  # No cambiar upload-state
        {"display": "none"},  # Ocultar progress container
    )


def _handle_upload_initialization(upload_id):
    """
    Helper: Inicializa monitoreo cuando se inicia un nuevo upload.
    Returns tuple: (children, disabled, toast, state, style)
    """
    if not upload_id:
        return (no_update, no_update, no_update, no_update, no_update)

    logger.info(f"[INIT] Nuevo upload iniciado: {upload_id[:8]}. Activando monitoreo.")
    return (
        create_progress_card(filename="Iniciando...", status="processing", upload_id=upload_id),
        False,  # HABILITAR interval
        no_update,
        {"status": "initiated", "upload_id": upload_id},
        {"display": "block"},  # Mostrar progress container
    )


def _handle_progress_monitoring(upload_id, n_intervals, upload_state):
    """
    Helper: Monitorea progreso del upload consultando backend.
    Returns tuple: (children, disabled, toast, state, style)

    Args:
        upload_id: ID del upload actual
        n_intervals: Número de intervals transcurridos
        upload_state: Estado actual del upload (para tracking de timestamps)
    """
    if not upload_id or n_intervals == 0:
        raise PreventUpdate

    # Debug: log para detectar polling infinito
    logger.info(f"[MONITOR] Upload check #{n_intervals} for ID: {upload_id[:8]}...")

    # Si llevamos muchos checks, forzar stop para evitar polling infinito
    if n_intervals > 100:  # Más de 200 segundos (100 * 2s)
        logger.warning(f"[MONITOR] Upload {upload_id[:8]} excede límite de checks ({n_intervals}). FORZANDO STOP.")
        toast = warning_toast("Tiempo de espera agotado. El proceso puede haber finalizado.", title="Tiempo de Espera")
        return (html.Div(), True, toast, no_update, no_update)  # DESHABILITAR interval - FORZADO  # Mantener style

    try:
        # Consultar estado del upload
        response = backend_client.get_upload_status(upload_id)

        if not response.success:
            logger.warning(f"Error consultando estado upload {upload_id}: {response.message}")
            # Si el upload no existe, detener polling
            if response.status_code == 404:
                logger.info(f"[MONITOR] Upload {upload_id[:8]} no encontrado. DETENIENDO polling.")
                toast = warning_toast("Upload no encontrado en el servidor", title="Upload No Encontrado")
                return (
                    html.Div(),
                    True,  # DESHABILITAR interval
                    toast,
                    no_update,  # No cambiar upload-state
                    no_update,  # Mantener style
                )
            raise PreventUpdate

        upload_data = response.data
        status = upload_data.get("status", "unknown")
        percentage = upload_data.get("percentage", 0)  # ✅ Issue #274: Leer del backend
        enrichment_status = upload_data.get("enrichment_status", "pending")
        enrichment_progress = upload_data.get("enrichment_progress", {})

        # DETECCIÓN DE ENRIQUECIMIENTO (Issue #264 Fase 2)
        # Si el status de upload está completado pero el enriquecimiento está "processing", mostrar progreso
        if status == "completed" and enrichment_status == "processing":
            progress_component = create_enrichment_progress_component(
                filename=upload_data.get("filename", "archivo"), enrichment_progress=enrichment_progress
            )

            # Continuar monitoreando durante enriquecimiento
            return (progress_component, False, no_update, no_update, no_update)  # Continuar interval  # Mantener style

        # ENRIQUECIMIENTO COMPLETADO (Issue #264 Fase 2)
        # Si el status de upload está completado Y el enriquecimiento también, mostrar progreso completo
        # por 3 segundos más antes del resumen final (Issue #330 - mejor UX)
        if status == "completed" and enrichment_status == "completed":
            # Obtener timestamp de la última actualización
            current_state = upload_state or {}
            enrichment_completed_at = current_state.get("enrichment_completed_at")

            # Si es la primera vez que detectamos completado, guardar timestamp
            if not enrichment_completed_at:
                logger.info(f"[MONITOR] Upload {upload_id[:8]} y enriquecimiento terminados. Mostrando progreso final por 3s...")

                # Mostrar componente de enriquecimiento completo (100%)
                progress_component = create_enrichment_progress_component(
                    filename=upload_data.get("filename", "archivo"),
                    enrichment_progress=enrichment_progress
                )

                # Guardar timestamp para saber cuándo mostrar resumen
                new_state = {
                    "status": status,
                    "upload_id": upload_id,
                    "filename": upload_data.get("filename", "archivo"),
                    "enrichment_completed_at": datetime.now().isoformat()
                }

                # Continuar polling por 3 segundos más para que usuario vea progreso completo
                return (
                    progress_component,
                    False,  # Continuar interval por 3s más
                    no_update,
                    new_state,
                    no_update,
                )

            # Si ya pasaron 3 segundos desde que completó, mostrar resumen final
            completed_time = datetime.fromisoformat(enrichment_completed_at)
            elapsed_seconds = (datetime.now() - completed_time).total_seconds()

            if elapsed_seconds >= 3:
                logger.info("[MONITOR] 3 segundos transcurridos. Mostrando resumen final y DETENIENDO polling.")

                # ✅ ISSUE #287: Usar componente persistente en lugar de create_progress_card()
                final_summary_component = create_final_progress_summary(
                    filename=upload_data.get("filename", "archivo"),
                    status=status,
                    rows_total=upload_data.get("rows_total"),
                    rows_processed=upload_data.get("rows_processed"),
                    rows_duplicates=upload_data.get("rows_duplicates", 0),  # Issue #330
                    enrichment_progress=enrichment_progress,
                    upload_id=upload_id,
                    error_message=upload_data.get("error_message"),
                    warnings=upload_data.get("warnings"),  # Issue #412
                )

                # Toast final con estadísticas de enriquecimiento
                final_toast = create_enrichment_success_toast(
                    filename=upload_data.get("filename", "archivo"), enrichment_progress=enrichment_progress
                )

                return (
                    final_summary_component,  # ✅ Componente PERSISTENTE (Issue #287)
                    True,  # DESHABILITAR interval - CRÍTICO
                    final_toast,
                    {"status": status, "upload_id": upload_id, "filename": upload_data.get("filename", "archivo")},
                    no_update,  # Mantener style
                )
            else:
                # Todavía no han pasado 3s, continuar mostrando progreso completo
                progress_component = create_enrichment_progress_component(
                    filename=upload_data.get("filename", "archivo"),
                    enrichment_progress=enrichment_progress
                )
                return (progress_component, False, no_update, current_state, no_update)

        # Actualizar componente de progreso (procesamiento estándar)
        progress_component = create_progress_card(
            filename=upload_data.get("filename", "archivo"),
            status=status,
            upload_id=upload_id,
            rows_total=upload_data.get("rows_total"),
            rows_processed=upload_data.get("rows_processed"),
            rows_with_errors=upload_data.get("rows_with_errors"),
            error_message=upload_data.get("error_message"),
            processing_completed_at=upload_data.get("processing_completed_at"),
            processing_notes=upload_data.get("processing_notes"),
            percentage=percentage,  # ✅ Issue #274: Pasar percentage del backend
            warnings=upload_data.get("warnings"),  # Issue #412
        )

        # Si el procesamiento terminó (éxito o error), detener monitoreo
        if status in ["completed", "error", "failed"]:
            # Log para confirmar que se detiene el polling
            logger.info(f"[MONITOR] Upload {upload_id[:8]} terminado con estado '{status}'. DETENIENDO polling.")

            # ✅ ISSUE #287: Crear componente persistente para estado final
            final_summary_component = create_final_progress_summary(
                filename=upload_data.get("filename", "archivo"),
                status=status,
                rows_total=upload_data.get("rows_total"),
                rows_processed=upload_data.get("rows_processed"),
                rows_duplicates=upload_data.get("rows_duplicates", 0),  # Issue #330
                enrichment_progress=enrichment_progress,
                upload_id=upload_id,
                error_message=upload_data.get("error_message"),
                warnings=upload_data.get("warnings"),  # Issue #412
            )

            # Crear toast final según el resultado
            if status == "completed":
                rows_total = upload_data.get("rows_total", 0)
                rows_processed = upload_data.get("rows_processed", 0)
                rows_errors = upload_data.get("rows_with_errors", 0)

                message = f"Archivo '{upload_data.get('filename', 'archivo')}' procesado correctamente. "
                message += f"{format_number(rows_processed)} de {format_number(rows_total)} registros procesados"
                if rows_errors > 0:
                    message += f" ({format_number(rows_errors)} con errores)"

                final_toast = success_toast(message, title="Procesamiento Completado")
            else:
                final_toast = error_toast(
                    f"Error al procesar '{upload_data.get('filename', 'archivo')}': {upload_data.get('error_message', 'Error desconocido')}",
                    title="Error de Procesamiento",
                )

            return (
                final_summary_component,  # ✅ Componente PERSISTENTE (Issue #287)
                True,  # DESHABILITAR interval - CRÍTICO
                final_toast,
                {"status": status, "upload_id": upload_id, "filename": upload_data.get("filename", "archivo")},
                no_update,  # Mantener style
            )

        # Continuar monitoreando
        return (progress_component, False, no_update, no_update, no_update)  # Continuar interval  # Mantener style

    except Exception as e:
        logger.error(f"Error monitoreando progreso upload {upload_id}: {str(e)}")
        return (
            create_error_progress_card(upload_id, str(e)),
            True,  # Detener monitoreo en caso de error
            no_update,
            {"status": "error", "upload_id": upload_id},
            no_update,  # Mantener style
        )


def register_upload_callbacks(app):
    """
    Registrar callbacks de upload con monitoreo en tiempo real.

    Args:
        app: Instancia de la aplicación Dash
    """

    # ========================================================================
    # CALLBACK: MOSTRAR/OCULTAR OPCIONES DE INVENTARIO (Issue #476)
    # ========================================================================

    @app.callback(
        Output("inventory-options-container", "style"),
        Input("upload-type-selector", "value"),
        prevent_initial_call=True,
    )
    def toggle_inventory_options(upload_type):
        """
        Mostrar opciones de inventario cuando se selecciona tipo 'inventory'.

        Issue #476: Carga de ficheros de inventario desde ERPs.
        """
        if upload_type == "inventory":
            return {"display": "block"}
        return {"display": "none"}

    # ========================================================================
    # CALLBACK UNIFICADO: FEEDBACK INMEDIATO + PROCESAMIENTO
    # ========================================================================
    # Este callback combina feedback inmediato + validación + upload
    # para cumplir REGLA #11 (One Input One Callback).
    #
    # Issue #334: Fix duplicate Input violation
    # Antes había 2 callbacks escuchando 'upload-data.contents':
    # 1. show_immediate_upload_feedback (líneas 292-391, eliminado)
    # 2. handle_file_upload_with_validation (líneas 401-569, eliminado)
    #
    # Ahora: UN SOLO callback con TODOS los outputs
    # ========================================================================

    @app.callback(
        [
            Output("upload-immediate-feedback", "children"),
            Output("upload-immediate-feedback", "style"),
            Output("file-preview-container", "children"),
            Output("file-preview-container", "style"),
            Output("current-upload-id", "data"),
        ],
        Input("upload-data", "contents"),
        [
            State("upload-data", "filename"),
            State("auth-tokens-store", "data"),
            State("upload-type-selector", "value"),  # Issue #476: Ventas/Inventario
            State("inventory-snapshot-date", "date"),  # Issue #476: Fecha inventario
        ],
        prevent_initial_call=True,
    )
    def handle_file_upload_unified(contents, filename, auth_tokens, upload_type, snapshot_date):
        """
        Callback UNIFICADO: Feedback inmediato + Validación + Upload.

        IMPORTANTE: Combina 2 callbacks anteriores para cumplir REGLA #11 (One Input One Callback).

        Flujo:
        1. Mostrar feedback inmediato (< 100ms) con estimación de tiempo
        2. Validación de formato y tamaño
        3. Si falla → mostrar preview de error
        4. Si pasa → subir archivo + iniciar monitoreo

        Issue #334: Fix duplicate Input on 'upload-data.contents'
        Callbacks originales combinados:
        - show_immediate_upload_feedback() (eliminado)
        - handle_file_upload_with_validation() (eliminado)
        """

        # Retornos por defecto para TODOS los outputs
        default_return = (
            None,  # immediate-feedback children
            {"display": "none"},  # immediate-feedback style
            None,  # file-preview children
            {"display": "none"},  # file-preview style
            None,  # upload-id
        )

        if contents is None or filename is None:
            return default_return

        # ============================================
        # FASE 1: FEEDBACK INMEDIATO (< 100ms)
        # ============================================
        # Calcular tamaño del archivo (estimación rápida sin decodificar completamente)
        try:
            # El contenido viene como "data:mime/type;base64,ENCODED_DATA"
            content_parts = contents.split(",")
            if len(content_parts) > 1:
                # Tamaño base64 es ~33% más grande que el original
                # Tamaño aproximado = (len(base64) * 3) / 4
                base64_length = len(content_parts[1])
                file_size_bytes = (base64_length * 3) // 4
                file_size_mb = file_size_bytes / (1024 * 1024)

                # Estimación de tiempo basada en benchmarks empíricos:
                # - Archivos pequeños (< 1MB): ~2-5s
                # - Archivos medianos (1-10MB): ~5-15s
                # - Archivos grandes (10-50MB): ~15-60s
                # Fórmula: tiempo_estimado = 2s (base) + 1.2s por MB
                estimated_time_seconds = 2 + (file_size_mb * 1.2)

                # Formatear tiempo estimado
                if estimated_time_seconds < 10:
                    time_text = f"~{int(estimated_time_seconds)}s"
                elif estimated_time_seconds < 60:
                    time_text = f"~{int(estimated_time_seconds)}s"
                else:
                    minutes = int(estimated_time_seconds // 60)
                    seconds = int(estimated_time_seconds % 60)
                    time_text = f"~{minutes}m {seconds}s"

                # Mensaje con tamaño y tiempo estimado
                size_text = f"{file_size_mb:.1f} MB" if file_size_mb >= 0.1 else f"{file_size_bytes / 1024:.0f} KB"
                details_text = f"{filename} • {size_text} • Tiempo estimado: {time_text}"
            else:
                # Fallback si no podemos calcular tamaño
                details_text = filename
                size_text = "tamaño desconocido"
                file_size_mb = 0
                file_size_bytes = 0
        except Exception as e:
            logger.warning(f"[IMMEDIATE_FEEDBACK] Error calculando tamaño: {e}")
            details_text = filename
            size_text = "tamaño desconocido"
            file_size_mb = 0
            file_size_bytes = 0

        # Crear indicador con barra de progreso y tiempo estimado
        feedback_component = dbc.Card(
            [
                dbc.CardBody(
                    [
                        # Header con spinner y título
                        html.Div(
                            [
                                dbc.Spinner(color="primary", size="sm", spinnerClassName="me-2"),
                                html.Strong("Procesando archivo...", className="me-2"),
                            ],
                            className="d-flex align-items-center mb-2",
                        ),
                        # Detalles del archivo
                        html.Small(details_text, className="text-muted d-block mb-3"),
                        # Barra de progreso indeterminada (animada)
                        dbc.Progress(
                            value=100,
                            striped=True,
                            animated=True,
                            color="primary",
                            style={"height": "6px"},
                        ),
                    ],
                    className="p-3",
                )
            ],
            className="border-primary mb-0",
            style={"borderLeft": "4px solid #0d6efd"},
        )

        logger.info(f"[UPLOAD_UNIFIED] Archivo seleccionado: {filename} ({size_text}). Iniciando validación y procesamiento...")

        # ============================================
        # FASE 2: VALIDACIÓN DE ARCHIVO
        # ============================================

        # Validación de extensión (rápida - no bloquea)
        allowed_extensions = [".csv", ".xlsx", ".xls"]
        file_ext = os.path.splitext(filename)[1].lower()

        if file_ext not in allowed_extensions:
            preview_error = dbc.Alert(
                [
                    html.I(className="fas fa-exclamation-triangle me-2"),
                    html.Strong("Formato de archivo no válido"),
                    html.Br(),
                    html.Small(
                        f"El archivo '{filename}' tiene extensión '{file_ext}'. Solo se permiten: .csv, .xlsx, .xls"
                    ),
                ],
                color="danger",
                className="mb-0",
            )

            return (
                None,  # Ocultar feedback inmediato
                {"display": "none"},
                preview_error,  # Mostrar error
                {"display": "block"},
                None,  # No upload-id
            )

        # Validación de tamaño (50MB max)
        try:
            content_type, content_string = contents.split(",")
            decoded = base64.b64decode(content_string)
            file_size_bytes = len(decoded)
            file_size_mb = file_size_bytes / (1024 * 1024)

            if file_size_mb > 50:
                preview_error = dbc.Alert(
                    [
                        html.I(className="fas fa-exclamation-triangle me-2"),
                        html.Strong("Archivo demasiado grande"),
                        html.Br(),
                        html.Small(
                            f"El archivo '{filename}' pesa {file_size_mb:.2f} MB. El tamaño máximo permitido es 50 MB"
                        ),
                    ],
                    color="danger",
                    className="mb-0",
                )

                return (
                    None,  # Ocultar feedback inmediato
                    {"display": "none"},
                    preview_error,  # Mostrar error
                    {"display": "block"},
                    None,  # No upload-id
                )

            # ============================================
            # FASE 3: ARCHIVO VÁLIDO → PROCESAR UPLOAD
            # ============================================

            # Crear formulario para enviar al backend
            files = {"file": (filename or "upload.csv", decoded, "text/csv")}
            # ERP type se obtiene automáticamente de la configuración de la farmacia en el backend
            data = {}

            # Issue #476: Detección AUTOMÁTICA de tipo de archivo basada en columnas
            detected_type = detect_file_type_from_columns(decoded, filename)
            logger.info(f"[UPLOAD_UNIFIED] Tipo detectado automáticamente: {detected_type.upper()}")

            is_inventory = detected_type == "inventory"
            if is_inventory:
                # Endpoint de inventario
                backend_url = f"{backend_client.backend_url}/api/v1/upload/inventory"
                # Añadir snapshot_date si se especificó
                if snapshot_date:
                    data["snapshot_date"] = snapshot_date
                logger.info(f"[UPLOAD_UNIFIED] Enviando a endpoint INVENTARIO, snapshot_date: {snapshot_date}")
            else:
                # Endpoint de ventas (default)
                backend_url = f"{backend_client.backend_url}/api/v1/upload/"
                logger.info("[UPLOAD_UNIFIED] Enviando a endpoint VENTAS")

            # Obtener headers de autenticación
            auth_headers = {}

            # Usar JWT token del usuario autenticado
            if auth_tokens and "tokens" in auth_tokens:
                from utils.auth import auth_manager

                # Decrypt and restore tokens
                if auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"]):
                    token = auth_manager.get_access_token()
                    if token:
                        auth_headers["Authorization"] = f"Bearer {token}"

            # Si no hay token JWT, fallar con error claro
            if not auth_headers:
                raise Exception("Authentication required: Please log in to upload files")

            response = requests.post(backend_url, files=files, data=data, headers=auth_headers, timeout=60)

            if response.status_code == 200:
                result = response.json()
                upload_id = result.get("upload_id")

                # Preview de éxito
                preview_success = dbc.Alert(
                    [
                        html.Div(
                            [
                                html.I(className="fas fa-file me-2 text-success", style={"fontSize": "1.5rem"}),
                                html.Div(
                                    [
                                        html.Strong(filename, className="d-block"),
                                        html.Small(
                                            [f"Tamaño: {file_size_mb:.2f} MB | ", f"Tipo: {file_ext.upper()}"],
                                            className="text-muted",
                                        ),
                                    ],
                                    className="ms-2",
                                ),
                            ],
                            className="d-flex align-items-center mb-0",
                        )
                    ],
                    color="success",
                    className="mb-0",
                )

                return (
                    None,  # Ocultar feedback inmediato
                    {"display": "none"},
                    preview_success,  # Mostrar preview exitoso
                    {"display": "block"},
                    upload_id,  # Upload ID para iniciar monitoreo
                )
            else:
                error_detail = response.text
                logger.error(f"[UPLOAD_UNIFIED] Error HTTP {response.status_code}: {error_detail}")
                return default_return

        except Exception as e:
            logger.error(f"[UPLOAD_UNIFIED] Error en upload: {str(e)}")
            return default_return

    @app.callback(
        [
            Output("upload-progress-container", "children"),
            Output("upload-progress-interval", "disabled"),
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("upload-state", "data", allow_duplicate=True),
            Output("upload-progress-container", "style"),
        ],
        [
            Input("upload-progress-interval", "n_intervals"),
            Input("url", "pathname"),
            Input("current-upload-id", "data"),
        ],
        [
            State("auth-state", "data"),  # Issue #302: Agregar auth_state
            State("upload-state", "data"),  # Issue #330: Agregar upload_state para tracking timestamps
        ],
        prevent_initial_call=True,
    )
    def monitor_upload_progress_and_cleanup(n_intervals, pathname, upload_id, auth_state, upload_state):  # Issue #302, #330
        """
        CALLBACK COORDINADOR (Issue #300 - REGLA #11):
        Combina 3 responsabilidades usando helper functions para clarity.

        Maneja:
        1. Monitoreo de progreso del upload (Input: n_intervals) → _handle_progress_monitoring()
        2. Limpieza al navegar fuera de /upload (Input: pathname) → _handle_navigation_cleanup()
        3. Inicialización del monitoreo (Input: current-upload-id) → _handle_upload_initialization()

        REFACTORIZADO (Code Review): Cada responsabilidad delegada a helper function.

        Issue #302: Verificación proactiva de autenticación antes de llamadas API.
        """
        # Issue #302: Verificación proactiva de autenticación
        from utils.auth_helpers import is_user_authenticated

        # Identificar qué Input disparó el callback
        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # CASO 1: Navegación fuera de /upload → Limpiar estado (NO requiere auth)
        if trigger_id == "url" and pathname != "/upload":
            return _handle_navigation_cleanup(pathname)

        # CASO 2: Nuevo upload iniciado → Activar monitoreo (NO requiere auth check aquí)
        if trigger_id == "current-upload-id":
            return _handle_upload_initialization(upload_id)

        # CASO 3: Monitoreo de progreso (n_intervals) → REQUIERE AUTH
        if trigger_id == "upload-progress-interval":
            if not is_user_authenticated(auth_state):
                logger.debug("[UPLOAD_PROGRESS] User not authenticated - skipping progress check")
                raise PreventUpdate

        return _handle_progress_monitoring(upload_id, n_intervals, upload_state)

    @app.callback(
        [
            Output("upload-history-container", "children"),
            Output("upload-data", "contents", allow_duplicate=True),
            Output("upload-data", "filename", allow_duplicate=True),
        ],
        [Input("upload-history-container", "id"), Input("upload-state", "data")],
        [
            State("auth-state", "data"),  # Issue #333: Verificar autenticación
            State("auth-context-sync", "data"),  # Issue #333: JWT timing protection
        ],
        prevent_initial_call="initial_duplicate",
    )
    def load_and_refresh_upload_history(element_id, upload_state, auth_state, auth_context_sync):
        """
        CONSOLIDATED CALLBACK (Issue #300 - REGLA #11):
        Combina load_upload_history_on_page_load + handle_upload_completion.
        Issue #333: Protección contra JWT timing race condition.

        Maneja:
        1. Cargar historial cuando la app carga (Input: upload-history-container.id)
        2. Refrescar historial y resetear upload al completar (Input: upload-state.data)

        Args:
            element_id: ID del contenedor de historial
            upload_state: Estado actual del upload
            auth_state: Estado de autenticación del usuario
            auth_context_sync: Estado de sincronización del token JWT

        Returns:
            Tuple: (historial content, upload contents reset, upload filename reset)
        """
        from utils.auth_helpers import is_user_authenticated

        # Identificar qué Input disparó el callback
        if not ctx.triggered:
            # Initial call desde page load (element_id trigger)
            trigger_id = "upload-history-container"
        else:
            trigger_id = ctx.triggered[0]["prop_id"].split(".")[0]

        # Issue #333: Verificar autenticación ANTES de hacer llamadas API
        if not is_user_authenticated(auth_state):
            logger.debug("[load_and_refresh_upload_history] User not authenticated - showing empty history")
            return create_empty_history_message(), no_update, no_update

        # Issue #333: Verificar que el token JWT esté sincronizado
        # Previene errores HTTP 401 por race condition en auth_context_sync
        if not auth_context_sync or not auth_context_sync.get("synced"):
            logger.debug("[load_and_refresh_upload_history] Token not synced yet (auth-context-sync pending) - showing skeleton")
            return html.Div([html.I(className="fas fa-spinner fa-spin me-2"), "Cargando historial..."]), no_update, no_update

        # CASO 1: Upload completado o eliminación → Refrescar historial Y resetear upload component
        if trigger_id == "upload-state" and upload_state:
            status = upload_state.get("status")
            refresh_requested = upload_state.get("refresh", False)

            # Si el upload completó (éxito o error) o se solicitó refresh (eliminación)
            if status in ["completed", "error"] or refresh_requested:
                if refresh_requested:
                    logger.info("[UPLOAD_HISTORY] Refresh requested (post-delete). Reloading history.")
                else:
                    logger.info(f"[UPLOAD_COMPLETION] Status: {status}. Refreshing history and resetting upload component.")

                # Refrescar historial
                history_content = _fetch_upload_history()

                # Resetear componente de upload (solo si completó, no si fue delete)
                if status in ["completed", "error"]:
                    return history_content, None, None
                else:
                    return history_content, no_update, no_update

            return no_update, no_update, no_update

        # CASO 2: Page load → Cargar historial inicial (NO resetear upload component)
        # Usar pharmacy_id hardcodeado para evitar problemas con API de farmacias
        history_content = _fetch_upload_history()
        return history_content, no_update, no_update


    def _fetch_upload_history():
        """
        Helper function to fetch and format upload history.
        Centraliza la lógica de llamada al backend y formateo de UI.
        Issue #330: Usar pharmacy_id del contexto del usuario autenticado.
        Issue #XXX: Layout visual mejorado con métricas diarias.
        """
        try:
            from datetime import datetime
            from utils.pharmacy_context import get_current_pharmacy_id

            response = backend_client.get_upload_history(pharmacy_id=get_current_pharmacy_id(), limit=10)

            if not response.success:
                logger.warning(f"[UPLOAD_HISTORY] Error al obtener historial: {response.message} (status: {response.status_code})")
                return create_empty_history_message()

            uploads = response.data

            if not uploads:
                return create_empty_history_message()

            # Crear lista de uploads
            history_items = []

            for upload in uploads:
                status = upload.get("status", "unknown")
                # Normalizar status a minúsculas (BD puede retornar MAYÚSCULAS)
                status = status.lower() if isinstance(status, str) else "unknown"

                # Configurar estilo según estado
                status_config = {
                    "completed": ("success", "fas fa-check-circle", "Completado"),
                    "error": ("danger", "fas fa-times-circle", "Error"),
                    "processing": ("primary", "fas fa-spinner fa-spin", "Procesando"),
                    "pending": ("info", "fas fa-clock", "Pendiente"),
                }.get(status, ("secondary", "fas fa-question-circle", "Desconocido"))

                # Parsear fechas y calcular días
                fecha_desde = upload.get("fecha_desde")
                fecha_hasta = upload.get("fecha_hasta")
                desde_dt = None
                hasta_dt = None
                num_days = 0

                if fecha_desde and fecha_hasta:
                    try:
                        desde_dt = datetime.fromisoformat(fecha_desde.replace('Z', '+00:00'))
                        hasta_dt = datetime.fromisoformat(fecha_hasta.replace('Z', '+00:00'))
                        num_days = max(1, (hasta_dt.date() - desde_dt.date()).days + 1)
                    except (ValueError, TypeError):
                        pass

                # Totales de ventas
                total_ventas = upload.get("total_ventas", 0)
                ventas_prescripcion = upload.get("ventas_prescripcion", 0)
                ventas_libre = upload.get("ventas_libre", 0)

                # Información de líneas
                rows_total = upload.get("rows_total", 0)
                rows_processed = upload.get("rows_processed", 0)
                rows_duplicates = upload.get("rows_duplicates", 0)
                rows_new = rows_processed - rows_duplicates if rows_processed and rows_duplicates else rows_processed

                # Calcular métricas diarias
                venta_media_diaria = total_ventas / num_days if num_days > 0 else 0
                lineas_media_diaria = rows_processed / num_days if num_days > 0 else 0

                # Porcentajes prescripción/libre
                pct_prescripcion = (ventas_prescripcion / total_ventas * 100) if total_ventas > 0 else 0
                pct_libre = (ventas_libre / total_ventas * 100) if total_ventas > 0 else 0

                # ====== DISEÑO VISUAL MEJORADO ======
                history_items.append(
                    dbc.Card(
                        [
                            # === HEADER: Filename + Status + Delete ===
                            dbc.CardHeader(
                                html.Div(
                                    [
                                        html.Div(
                                            [
                                                html.I(className=f"{status_config[1]} me-2 text-{status_config[0]}"),
                                                html.Strong(
                                                    upload.get("filename", "Sin nombre"),
                                                    className="text-truncate",
                                                    style={"maxWidth": "200px"},
                                                    title=upload.get("filename", "Sin nombre"),
                                                ),
                                            ],
                                            className="d-flex align-items-center flex-grow-1",
                                        ),
                                        html.Div(
                                            [
                                                html.Span(
                                                    status_config[2],
                                                    className=f"badge bg-{status_config[0]} me-2",
                                                ),
                                                dbc.Button(
                                                    html.I(className="fas fa-trash-alt"),
                                                    id={"type": "delete-upload", "index": upload.get("upload_id")},
                                                    color="danger",
                                                    outline=True,
                                                    size="sm",
                                                    className="py-0 px-1",
                                                    title="Eliminar archivo y datos",
                                                ),
                                            ],
                                            className="d-flex align-items-center",
                                        ),
                                    ],
                                    className="d-flex justify-content-between align-items-center",
                                ),
                                className="py-2 px-3",
                            ),
                            # === BODY: Métricas visuales ===
                            dbc.CardBody(
                                [
                                    # --- ROW 1: Fechas con badge de días ---
                                    html.Div(
                                        [
                                            html.Div(
                                                [
                                                    html.I(className="fas fa-calendar-alt me-2 text-primary"),
                                                    html.Span(
                                                        format_date_spanish(desde_dt) if desde_dt else "Sin fecha",
                                                        className="fw-bold",
                                                    ),
                                                    html.I(className="fas fa-arrow-right mx-2 text-muted", style={"fontSize": "0.7rem"}),
                                                    html.Span(
                                                        format_date_spanish(hasta_dt) if hasta_dt else "",
                                                        className="fw-bold",
                                                    ),
                                                ],
                                                className="d-flex align-items-center",
                                            ),
                                            html.Span(
                                                f"{num_days} días",
                                                className="badge bg-secondary rounded-pill",
                                            ) if num_days > 0 else None,
                                        ],
                                        className="d-flex justify-content-between align-items-center mb-3",
                                    ) if desde_dt else None,

                                    # --- ROW 2: Grid de KPIs (4 métricas) ---
                                    dbc.Row(
                                        [
                                            # KPI 1: Total Ventas
                                            dbc.Col(
                                                html.Div(
                                                    [
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-coins text-success", style={"fontSize": "1.2rem"}),
                                                            ],
                                                            className="mb-1",
                                                        ),
                                                        html.Div(format_currency(total_ventas), className="fw-bold text-success"),
                                                        html.Small("Total", className="text-muted"),
                                                    ],
                                                    className="text-center p-2 border rounded bg-light",
                                                ),
                                                width=6,
                                                lg=3,
                                                className="mb-2",
                                            ),
                                            # KPI 2: Venta Media Diaria
                                            dbc.Col(
                                                html.Div(
                                                    [
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-chart-line text-primary", style={"fontSize": "1.2rem"}),
                                                            ],
                                                            className="mb-1",
                                                        ),
                                                        html.Div(format_currency(venta_media_diaria), className="fw-bold text-primary"),
                                                        html.Small("€/día", className="text-muted"),
                                                    ],
                                                    className="text-center p-2 border rounded bg-light",
                                                ),
                                                width=6,
                                                lg=3,
                                                className="mb-2",
                                            ),
                                            # KPI 3: Líneas Totales
                                            dbc.Col(
                                                html.Div(
                                                    [
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-list-ol text-info", style={"fontSize": "1.2rem"}),
                                                            ],
                                                            className="mb-1",
                                                        ),
                                                        html.Div(format_number(rows_processed), className="fw-bold text-info"),
                                                        html.Small("Líneas", className="text-muted"),
                                                    ],
                                                    className="text-center p-2 border rounded bg-light",
                                                ),
                                                width=6,
                                                lg=3,
                                                className="mb-2",
                                            ),
                                            # KPI 4: Líneas Media Diaria
                                            dbc.Col(
                                                html.Div(
                                                    [
                                                        html.Div(
                                                            [
                                                                html.I(className="fas fa-tasks text-warning", style={"fontSize": "1.2rem"}),
                                                            ],
                                                            className="mb-1",
                                                        ),
                                                        html.Div(f"{lineas_media_diaria:.0f}", className="fw-bold text-warning"),
                                                        html.Small("Líneas/día", className="text-muted"),
                                                    ],
                                                    className="text-center p-2 border rounded bg-light",
                                                ),
                                                width=6,
                                                lg=3,
                                                className="mb-2",
                                            ),
                                        ],
                                        className="g-2 mb-3",
                                    ) if total_ventas > 0 else None,

                                    # --- ROW 3: Barra Prescripción/Libre con porcentajes ---
                                    html.Div(
                                        [
                                            # Labels con porcentajes
                                            html.Div(
                                                [
                                                    html.Div(
                                                        [
                                                            html.Span(
                                                                [
                                                                    html.I(className="fas fa-prescription me-1"),
                                                                    f"Prescripción {pct_prescripcion:.0f}%",
                                                                ],
                                                                className="badge bg-primary me-2",
                                                            ),
                                                            html.Small(format_currency(ventas_prescripcion), className="text-muted"),
                                                        ],
                                                        className="d-flex align-items-center",
                                                    ),
                                                    html.Div(
                                                        [
                                                            html.Small(format_currency(ventas_libre), className="text-muted me-2"),
                                                            html.Span(
                                                                [
                                                                    f"Libre {pct_libre:.0f}%",
                                                                    html.I(className="fas fa-store ms-1"),
                                                                ],
                                                                className="badge bg-info",
                                                            ),
                                                        ],
                                                        className="d-flex align-items-center",
                                                    ),
                                                ],
                                                className="d-flex justify-content-between mb-2",
                                            ),
                                            # Barra de progreso stacked
                                            dbc.Progress(
                                                [
                                                    dbc.Progress(
                                                        value=pct_prescripcion,
                                                        color="primary",
                                                        bar=True,
                                                    ),
                                                    dbc.Progress(
                                                        value=pct_libre,
                                                        color="info",
                                                        bar=True,
                                                    ),
                                                ],
                                                style={"height": "10px"},
                                                className="rounded",
                                            ),
                                        ],
                                    ) if total_ventas > 0 else None,

                                    # --- ROW 4: Info duplicados (si hay) ---
                                    html.Div(
                                        [
                                            html.Hr(className="my-2"),
                                            html.Small(
                                                [
                                                    html.I(className="fas fa-info-circle me-1 text-muted"),
                                                    f"{format_number(rows_new)} nuevas · {format_number(rows_duplicates)} duplicadas",
                                                ],
                                                className="text-muted",
                                            ),
                                        ],
                                    ) if rows_duplicates > 0 else None,

                                    # --- Estado sin datos ---
                                    html.Div(
                                        [
                                            html.I(className="fas fa-inbox fa-2x text-muted mb-2"),
                                            html.P("Sin datos de ventas", className="text-muted mb-0 small"),
                                        ],
                                        className="text-center py-3",
                                    ) if total_ventas == 0 and status == "completed" else None,
                                ],
                                className="py-2 px-3",
                            ),
                        ],
                        className="mb-3 shadow-sm",
                    )
                )

            return html.Div(history_items)
        except Exception as e:
            logger.error(f"Error fetching history: {e}")
            return create_empty_history_message()

    # ❌ REMOVED: handle_upload_completion callback (consolidated in load_and_refresh_upload_history above)
    # This callback violated REGLA #11: ONE INPUT ONE CALLBACK
    # Outputs 'upload-history-container.children', 'upload-data.contents', 'upload-data.filename'
    # now handled by load_and_refresh_upload_history callback with multiple inputs (line 440)

    # ❌ REMOVED: cleanup_on_navigation callback (consolidated in monitor_upload_progress_and_cleanup above)
    # This callback violated REGLA #11: ONE INPUT ONE CALLBACK
    # Outputs 'upload-progress-interval.disabled' and 'current-upload-id.data' now handled
    # by monitor_upload_progress_and_cleanup callback with multiple inputs (line 228)

    @app.callback(
        Output({"type": "upload-summary-card", "index": MATCH}, "style"),
        Input({"type": "dismiss-upload-summary", "index": MATCH}, "n_clicks"),
        prevent_initial_call=True,
    )
    def dismiss_upload_summary(n_clicks):
        """
        Callback para cerrar el resumen final de upload cuando usuario hace clic en "X".
        Issue #287 - Permitir que usuario controle cuándo desaparece el resumen.

        RESPETA REGLA #11: Un Input (button click) → Un Output (hide card)
        Usa patrón MATCH para esconder el card específico con el botón.

        CORRECCIÓN: Ahora Output también tiene MATCH para coincidir con Input MATCH.
        Esto previene el error "extras beyond the Output(s)" de Dash.

        Args:
            n_clicks: Número de clicks del botón dismiss

        Returns:
            Dict con style para ocultar el card específico
        """
        if not ctx.triggered or not n_clicks:
            raise PreventUpdate

        # Ocultar el card específico
        logger.info("[DISMISS] Usuario cerró el resumen final de upload.")
        return {"display": "none"}

    @app.callback(
        Output("upload-system-status", "children"),
        [Input("app-load", "n_intervals"), Input("url", "pathname"), Input("auth-state", "data")],  # Issue #429: auth-state como Input (se dispara al login)
        [
            State("upload-state", "data"),  # Issue #429: Actualizar cuando termine upload
        ],
        prevent_initial_call=False,
    )
    def show_upload_summary_kpis(n_intervals, pathname, auth_state, upload_state):
        """
        Mostrar KPIs del resumen de cargas (Issue #429).
        Reemplaza "Configuración del Sistema" con métricas útiles del historial.

        Issue #330: Trigger inmediato al cargar página (pathname) + refresh periódico (interval).
        Issue #333: Protección contra JWT timing race condition.
        Issue #429: Mostrar KPIs útiles: Archivos subidos, Ventas procesadas, Tasa éxito, Última carga.

        Args:
            n_intervals: Número de intervalos transcurridos
            pathname: Ruta actual de la página
            upload_state: Estado del upload actual (para actualizar cuando termine)
            auth_state: Estado de autenticación del usuario
            auth_context_sync: Estado de sincronización del token JWT

        Returns:
            Componente con KPIs del resumen de cargas
        """
        from datetime import datetime
        from utils.auth_helpers import is_user_authenticated
        from utils.pharmacy_context import get_current_pharmacy_id

        logger.info(f"[show_upload_summary_kpis] CALLBACK EXECUTED - pathname: {pathname}, auth_state: {auth_state}")

        # Guard: Solo ejecutar en página /upload (pero permitir None en carga inicial)
        if pathname is not None and pathname != "/upload":
            logger.info("[show_upload_summary_kpis] Pathname guard - not /upload, raising PreventUpdate")
            raise PreventUpdate

        # Issue #333: Verificar autenticación ANTES de hacer llamadas API
        if not is_user_authenticated(auth_state):
            logger.info("[show_upload_summary_kpis] User not authenticated, raising PreventUpdate")
            raise PreventUpdate

        # Ya verificamos autenticación arriba con is_user_authenticated(auth_state)
        # Si llegamos aquí, el usuario está autenticado

        try:
            # Obtener historial de uploads (límite 10)
            response = backend_client.get_upload_history(pharmacy_id=get_current_pharmacy_id(), limit=10)

            if not response.success or not response.data:
                return html.Div(
                    [
                        html.I(className="fas fa-info-circle text-muted me-2"),
                        "No hay cargas registradas aún"
                    ],
                    className="text-center text-muted"
                )

            uploads = response.data

            # Calcular KPIs del historial
            total_uploads = len(uploads)
            successful_uploads = len([u for u in uploads if u.get("status", "").lower() == "completed"])
            total_ventas_procesadas = sum(u.get("total_ventas", 0) for u in uploads)
            success_rate = (successful_uploads / total_uploads * 100) if total_uploads > 0 else 0

            # Última carga
            if uploads:
                last_upload = uploads[0]  # Asumiendo que están ordenados por fecha desc
                last_upload_date = last_upload.get("uploaded_at")
                if last_upload_date:
                    try:
                        dt = datetime.fromisoformat(last_upload_date.replace('Z', '+00:00'))
                        days_ago = (datetime.now(dt.tzinfo) - dt).days
                        if days_ago == 0:
                            last_upload_text = "Hoy"
                        elif days_ago == 1:
                            last_upload_text = "Ayer"
                        else:
                            last_upload_text = f"Hace {days_ago} días"
                    except (ValueError, TypeError):
                        last_upload_text = "Desconocido"
                else:
                    last_upload_text = "Desconocido"
            else:
                last_upload_text = "Nunca"

            # Construir KPIs en grid
            return dbc.Row(
                [
                    # KPI 1: Archivos subidos
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-file-upload fa-lg text-primary mb-2"),
                                    html.H4(format_number(total_uploads), className="mb-0 text-primary"),
                                    html.Small("Archivos", className="text-muted"),
                                ],
                                className="text-center"
                            )
                        ],
                        width=6,
                        md=3,
                        className="mb-3 mb-md-0"
                    ),
                    # KPI 2: Ventas procesadas
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-euro-sign fa-lg text-success mb-2"),
                                    html.H4(format_currency(total_ventas_procesadas).split()[0], className="mb-0 text-success"),
                                    html.Small("Ventas", className="text-muted"),
                                ],
                                className="text-center"
                            )
                        ],
                        width=6,
                        md=3,
                        className="mb-3 mb-md-0"
                    ),
                    # KPI 3: Tasa de éxito
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(
                                        className=f"fas fa-{'check-circle' if success_rate >= 80 else 'exclamation-triangle'} fa-lg {'text-success' if success_rate >= 80 else 'text-warning'} mb-2"
                                    ),
                                    html.H4(f"{success_rate:.0f}%", className=f"mb-0 {'text-success' if success_rate >= 80 else 'text-warning'}"),
                                    html.Small("Tasa éxito", className="text-muted"),
                                ],
                                className="text-center"
                            )
                        ],
                        width=6,
                        md=3,
                        className="mb-3 mb-md-0"
                    ),
                    # KPI 4: Última carga
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-clock fa-lg text-info mb-2"),
                                    html.H6(last_upload_text, className="mb-0 text-info", style={"fontSize": "1rem"}),
                                    html.Small("Última carga", className="text-muted"),
                                ],
                                className="text-center"
                            )
                        ],
                        width=6,
                        md=3,
                        className="mb-0"
                    ),
                ],
                className="g-3"
            )

        except Exception as e:
            logger.error(f"[UPLOAD_SUMMARY_KPIS] Error al cargar resumen: {str(e)}")
            return html.Div(
                [
                    html.I(className="fas fa-exclamation-triangle text-warning me-2"),
                    "Error cargando resumen de cargas"
                ],
                className="text-center"
            )

    @app.callback(
        Output("erp-help-accordion-container", "children"),
        [Input("url", "pathname"), Input("auth-state", "data")],  # Issue #429: auth-state como Input (se dispara al login)
        prevent_initial_call=False,
    )
    def populate_erp_help_accordion(pathname, auth_state):
        """
        Poblar acordeón de ayuda ERP con el tipo de ERP del usuario (Issue #429).
        Expande automáticamente el acordeón correspondiente al ERP configurado.

        Args:
            pathname: Ruta actual
            auth_state: Estado de autenticación
            auth_context_sync: Sincronización de JWT

        Returns:
            Instrucciones de exportación para el ERP del usuario
        """
        from components.upload import create_erp_help_section
        from utils.auth_helpers import is_user_authenticated

        # Guard: Solo ejecutar en página /upload
        if pathname != "/upload":
            raise PreventUpdate

        # Verificar autenticación
        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        # Ya verificamos autenticación arriba - proceder con la carga

        try:
            # Obtener ERP del usuario desde la farmacia
            pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
            erp_type = None

            if pharmacy_id:
                pharmacy_response = backend_client.get_pharmacy_data(pharmacy_id)
                if pharmacy_response.success and pharmacy_response.data:
                    # Backend retorna {"success": true, "data": {...}}
                    # ApiResponse.data contiene todo el JSON, así que hay que acceder a "data" primero
                    pharmacy_data = pharmacy_response.data.get("data", {})
                    erp_type = pharmacy_data.get("erp_type", "").lower()
                    logger.info(f"[ERP_HELP] ERP type for pharmacy {pharmacy_id}: {erp_type}")

            # Crear sección de ayuda con instrucciones del ERP del usuario
            return create_erp_help_section(user_erp_type=erp_type)

        except Exception as e:
            logger.error(f"[ERP_HELP] Error: {str(e)}")
            # Retornar mensaje genérico en caso de error
            return create_erp_help_section()

    # =========================================================================
    # DELETE UPLOAD CALLBACKS
    # =========================================================================

    # Callback 1: Capturar click de botón delete (pattern-matching → Store)
    # NOTA: Pattern-matching inputs con MATCH no pueden tener outputs estáticos.
    # Solución: Usar ALL y guardar en Store, luego otro callback abre el modal.
    @app.callback(
        Output("delete-upload-target", "data"),
        Input({"type": "delete-upload", "index": ALL}, "n_clicks"),
        prevent_initial_call=True,
    )
    def capture_delete_click(all_clicks):
        """
        Captura el click del botón delete y guarda el upload_id en el store.

        Usa ALL en lugar de MATCH porque los outputs son estáticos.
        El upload_id se extrae del trigger_id.
        """
        logger.info(f"[DELETE_CLICK] Callback triggered! all_clicks={all_clicks}")
        logger.info(f"[DELETE_CLICK] ctx.triggered={ctx.triggered}")
        logger.info(f"[DELETE_CLICK] ctx.triggered_id={ctx.triggered_id}")

        if not ctx.triggered or not any(c for c in all_clicks if c):
            logger.info("[DELETE_CLICK] No valid click detected - PreventUpdate")
            raise PreventUpdate

        trigger_id = ctx.triggered_id

        # Verificar que es un botón delete
        if isinstance(trigger_id, dict) and trigger_id.get("type") == "delete-upload":
            upload_id = trigger_id.get("index")
            logger.info(f"[DELETE_CLICK] Delete button clicked! upload_id={upload_id}")
            if upload_id:
                return {"upload_id": upload_id, "trigger_time": datetime.now().isoformat()}

        logger.warning(f"[DELETE_CLICK] trigger_id not a delete button: {trigger_id}")
        raise PreventUpdate

    # Callback 2: Abrir modal cuando delete-upload-target cambia
    @app.callback(
        [
            Output("delete-upload-modal", "is_open"),
            Output("delete-upload-target", "data", allow_duplicate=True),
            Output("delete-upload-preview", "children"),
        ],
        [
            Input("delete-upload-target", "data"),
            Input("delete-upload-cancel", "n_clicks"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def handle_delete_upload_modal(target_data, cancel_clicks, auth_state, auth_tokens):
        """
        Abrir/cerrar modal de confirmación de eliminación de upload.

        - Cuando delete-upload-target cambia: Abre modal y carga preview
        - Al hacer clic en cancelar: Cierra modal y limpia datos

        REGLA #7.6: Restaura tokens antes de API calls para multi-worker.
        """
        from utils.auth import auth_manager
        from utils.auth_helpers import is_user_authenticated

        if not ctx.triggered:
            raise PreventUpdate

        trigger_id = ctx.triggered_id

        # Cancelar: cerrar modal
        if trigger_id == "delete-upload-cancel":
            return False, None, html.Div()

        # Target store changed (nuevo upload_id para eliminar)
        if trigger_id == "delete-upload-target" and target_data:
            upload_id = target_data.get("upload_id")

            if not upload_id:
                raise PreventUpdate

            # Verificar autenticación
            if not is_user_authenticated(auth_state):
                logger.warning("[DELETE_UPLOAD] Usuario no autenticado")
                raise PreventUpdate

            # REGLA #7.6: Restaurar tokens para multi-worker (Render)
            if auth_tokens and "tokens" in auth_tokens:
                try:
                    auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])
                except Exception as e:
                    logger.warning(f"[DELETE_UPLOAD] Error restaurando tokens: {e}")

            # Obtener preview del backend
            try:
                response = backend_client.get_upload_delete_preview(upload_id)

                if response.success and response.data:
                    data = response.data
                    preview_content = create_delete_preview(data)
                    # Enriquecer target_data con filename para el callback de confirmación
                    enriched_data = {**target_data, "filename": data.get("filename")}
                    return True, enriched_data, preview_content
                else:
                    logger.error(f"[DELETE_UPLOAD] Error obteniendo preview: {response.message}")
                    return (
                        True,
                        target_data,
                        dbc.Alert("Error cargando información del archivo", color="warning"),
                    )

            except Exception as e:
                logger.error(f"[DELETE_UPLOAD] Excepción obteniendo preview: {e}")
                return True, target_data, dbc.Alert(f"Error: {str(e)}", color="danger")

        raise PreventUpdate

    # ============================================================================
    # PATRÓN 2 FASES: Feedback inmediato + eliminación async
    # ============================================================================

    @app.callback(
        [
            Output("delete-upload-confirm", "children"),
            Output("delete-upload-confirm", "disabled"),
            Output("delete-upload-cancel", "disabled"),
            Output("delete-in-progress", "data"),
        ],
        Input("delete-upload-confirm", "n_clicks"),
        [
            State("delete-upload-target", "data"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=True,
    )
    def start_delete_phase1(n_clicks, target_data, auth_state):
        """
        FASE 1: Feedback inmediato al usuario.

        Muestra estado de carga y dispara la fase 2.
        """
        from utils.auth_helpers import is_user_authenticated

        if not n_clicks or not target_data:
            raise PreventUpdate

        upload_id = target_data.get("upload_id")
        filename = target_data.get("filename", "archivo")

        if not upload_id:
            raise PreventUpdate

        # Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.warning("[DELETE_PHASE1] Usuario no autenticado")
            raise PreventUpdate

        logger.info(f"[DELETE_PHASE1] Starting delete for {filename} ({upload_id[:8]}...)")

        # Mostrar spinner y deshabilitar botones
        loading_content = html.Span([
            dbc.Spinner(size="sm", color="light", spinner_class_name="me-2"),
            "Eliminando..."
        ])

        return (
            loading_content,  # Cambiar texto del botón
            True,  # Deshabilitar botón confirmar
            True,  # Deshabilitar botón cancelar
            {"upload_id": upload_id, "filename": filename, "timestamp": datetime.now().isoformat()},
        )

    @app.callback(
        [
            Output("delete-upload-modal", "is_open", allow_duplicate=True),
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("upload-state", "data", allow_duplicate=True),
            Output("delete-upload-confirm", "children", allow_duplicate=True),
            Output("delete-upload-confirm", "disabled", allow_duplicate=True),
            Output("delete-upload-cancel", "disabled", allow_duplicate=True),
        ],
        Input("delete-in-progress", "data"),
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # REGLA #7.6
        ],
        prevent_initial_call=True,
    )
    def execute_delete_phase2(delete_data, auth_state, auth_tokens):
        """
        FASE 2: Ejecutar eliminación real con LOOP INTERNO para datasets grandes.

        CHUNKED DELETE: El backend elimina en batches de 5K registros.
        Este callback hace un LOOP síncrono llamando al backend repetidamente
        hasta que status='completed'. No usa re-triggering de callbacks.

        REGLA #7.6: Restaura tokens antes de API calls para multi-worker.
        """
        from utils.auth import auth_manager
        from utils.auth_helpers import is_user_authenticated
        import time

        # Contenido original del botón para restaurar
        original_button = html.Span([html.I(className="fas fa-trash-alt me-2"), "Eliminar"])

        if not delete_data:
            raise PreventUpdate

        upload_id = delete_data.get("upload_id")
        filename = delete_data.get("filename", "archivo")

        if not upload_id:
            raise PreventUpdate

        logger.info(f"[DELETE_PHASE2] Starting chunked delete for {filename} ({upload_id[:8]}...)")

        # Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.warning("[DELETE_PHASE2] Usuario no autenticado para eliminar")
            return (
                False,
                error_toast("Sesión expirada. Inicia sesión de nuevo."),
                no_update,
                original_button,
                False,
                False,
            )

        # REGLA #7.6: Restaurar tokens para multi-worker (Render)
        if auth_tokens and "tokens" in auth_tokens:
            try:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])
                logger.info("[DELETE_PHASE2] Tokens restored successfully")
            except Exception as e:
                logger.warning(f"[DELETE_PHASE2] Error restaurando tokens: {e}")

        # LOOP de eliminación en batches (síncrono, sin re-triggering)
        iteration = 0
        max_iterations = 100  # Safety limit (~500K registros con batches de 5K)
        total_deleted = 0

        try:
            while iteration < max_iterations:
                iteration += 1
                logger.info(f"[DELETE_PHASE2] Iteration {iteration} - calling backend")

                response = backend_client.delete_upload(upload_id)

                if not response.success:
                    logger.error(f"[DELETE_PHASE2] Error del backend: {response.message}")
                    return (
                        False,
                        error_toast(f"Error eliminando: {response.message}"),
                        no_update,
                        original_button,
                        False,
                        False,
                    )

                status = response.data.get("status", "completed")

                if status == "completed":
                    # Todo eliminado exitosamente
                    logger.info(
                        f"[DELETE_PHASE2] Upload '{filename}' eliminado exitosamente "
                        f"(iterations={iteration}, total_deleted={total_deleted})"
                    )
                    return (
                        False,  # Cerrar modal
                        success_toast(f"'{filename}' eliminado correctamente"),
                        {"refresh": True, "timestamp": datetime.now().isoformat()},
                        original_button,
                        False,
                        False,
                    )

                # status == "in_progress" - continuar el loop
                progress = response.data.get("progress", {})
                deleted_this_batch = progress.get("deleted_this_batch", 0)
                remaining = progress.get("total_remaining", 0)
                phase = progress.get("phase", "unknown")

                total_deleted += deleted_this_batch

                logger.info(
                    f"[DELETE_PHASE2] Iteration {iteration}: phase={phase}, "
                    f"deleted={deleted_this_batch}, remaining={remaining}"
                )

                # Pequeña pausa entre batches (200ms)
                time.sleep(0.2)

            # Si llegamos aquí, excedimos max_iterations (safety limit)
            logger.error(f"[DELETE_PHASE2] Exceeded max iterations ({max_iterations})")
            return (
                False,
                error_toast("Eliminación incompleta. Intenta de nuevo."),
                {"refresh": True, "timestamp": datetime.now().isoformat()},
                original_button,
                False,
                False,
            )

        except Exception as e:
            logger.error(f"[DELETE_PHASE2] Excepción eliminando upload: {e}")
            return (
                False,
                error_toast(f"Error: {str(e)}"),
                no_update,
                original_button,
                False,
                False,
            )


