# frontend/callbacks/storage_usage.py
"""
Callbacks para FREE tier UX improvements (Issue #420).
- Storage indicator in sidebar
- Date restriction banners in date pickers
- Disabled controls for FREE users (date picker, employee filter)

Issue #435: Fixed auth-ready dependency for storage usage callback.
"""
import logging
from datetime import datetime, timedelta, timezone

import dash_bootstrap_components as dbc
from dash import Input, Output, State, html, no_update

from utils.auth import auth_manager
from utils.auth_helpers import is_user_authenticated
from utils.helpers import format_month_year
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def _create_date_restriction_banner():
    """
    Create the FREE tier date restriction banner component.

    Returns:
        dbc.Alert with info about 3-month data restriction
    """
    return dbc.Alert(
        [
            html.I(className="fas fa-info-circle me-2"),
            html.Span("Plan FREE: Datos limitados a últimos 3 meses desde tu última venta"),
        ],
        color="info",
        className="mt-2 mb-0 py-2",
        style={"fontSize": "0.8rem"},
    )


def register_storage_usage_callbacks(app):
    """
    Registrar callback para storage usage.

    Args:
        app: Instancia de la aplicación Dash
    """

    @app.callback(
        Output("storage-usage-store", "data"),
        [Input("url", "pathname"), Input("auth-ready", "data")],
        [State("auth-state", "data"), State("auth-tokens-store", "data")],
        prevent_initial_call=False,
    )
    def fetch_storage_usage(pathname, auth_ready, auth_state, auth_tokens):
        """
        Fetch storage usage SOLO para usuarios FREE al cargar cualquier página.

        Args:
            pathname: Ruta actual (trigger en navegación)
            auth_ready: Flag indicando que auth está listo
            auth_state: Estado de autenticación del usuario
            auth_tokens: Tokens encriptados desde localStorage (Issue #442)

        Returns:
            Dict con storage data o None si no es FREE user

        Issue #302: Verificación proactiva de autenticación.
        Issue #420: FREE tier experience - Storage indicator.
        Issue #435: Wait for auth-ready before making API calls.
        Issue #442: Multi-worker token restoration - Restore tokens in THIS worker.
        """
        # Issue #435: Wait for auth-ready signal
        if not auth_ready:
            logger.debug("[STORAGE] Auth not ready yet - returning None")
            return None  # Return None (not no_update) to initialize store

        # Issue #302: Verificación proactiva
        if not is_user_authenticated(auth_state):
            logger.debug("[STORAGE] User not authenticated - skipping storage fetch")
            return None  # Return None to initialize store (sidebar depends on this)

        # Extraer plan de suscripción
        user_data = auth_state.get("user") or {}
        subscription_plan = user_data.get("subscription_plan", "").lower()

        # Solo mostrar para usuarios FREE
        if subscription_plan != "free":
            logger.debug(f"[STORAGE] Plan '{subscription_plan}' - storage indicator disabled")
            return None  # None = no mostrar indicador

        # Fetch storage usage desde API
        try:
            # Issue #442: Multi-worker token restoration
            # In Render multi-worker environment, each worker has its own auth_manager.
            # sync_auth_context may have run in a DIFFERENT worker, so this worker's
            # auth_manager might not have the tokens. We restore them here to ensure
            # this worker can authenticate.
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])
                logger.debug("[STORAGE] Restored tokens from auth-tokens-store in this worker")

            # Issue #441: Pass explicit auth_headers to avoid race condition
            access_token = auth_manager.get_access_token()
            if not access_token:
                logger.debug("[STORAGE] No access token available yet - will retry on next trigger")
                return None

            auth_headers = {"Authorization": f"Bearer {access_token}"}
            response = request_coordinator.make_request(
                "/api/v1/auth/storage-usage",
                method="GET",
                timeout=10,
                auth_headers=auth_headers,
            )

            if response and "error" not in response:
                # Extraer datos del response
                storage_data = {
                    "total_used_mb": response.get("total_used_mb", 0),
                    "limit_mb": response.get("limit_mb", 100),
                    "percentage": response.get("percentage", 0),
                    "plan": response.get("plan", "free"),
                }

                logger.info(
                    f"[STORAGE] Fetched usage: {storage_data['total_used_mb']:.1f}/"
                    f"{storage_data['limit_mb']} MB ({storage_data['percentage']:.1f}%)"
                )

                return storage_data

            # Error en API - no bloquear UI
            logger.warning(f"[STORAGE] API error: {response.get('error', 'Unknown') if response else 'No response'}")
            return None

        except Exception as e:
            logger.error(f"[STORAGE] Exception fetching storage: {str(e)}")
            return None  # Fail silently, no bloquear navegación

    # =========================================================================
    # Issue #420 Phase 2: Date restriction banners for FREE users
    # =========================================================================

    @app.callback(
        Output("generics-date-restriction-banner", "children"),
        Output("generics-date-restriction-banner", "style"),
        [Input("url", "pathname")],
        [
            State("generics-date-range", "start_date"),  # Issue #435: State (no Input) - evita error skeleton
            State("auth-state", "data"),
        ],
        prevent_initial_call=False,
    )
    def show_generics_date_banner(pathname, start_date, auth_state):
        """
        Show date restriction banner in generics page for FREE users.

        Issue #435: Banner dinámico basado en fecha seleccionada (UX improvement).
        - Muestra banner INFO solo cuando usuario FREE selecciona fecha fuera de rango
        - NO bloquea ni deshabilita controles (enfoque informativo, no restrictivo)
        - Banner es dismissable

        Args:
            pathname: Current URL path
            start_date: Selected start date from date range picker
            auth_state: User authentication state

        Returns:
            Tuple of (banner_children, banner_style)
        """
        from datetime import datetime, timezone, timedelta

        # Only show on generics page
        if not pathname or not pathname.startswith("/generics"):
            return None, {"display": "none"}

        # Check if user is authenticated
        if not is_user_authenticated(auth_state):
            return None, {"display": "none"}

        # Check subscription plan
        user_data = auth_state.get("user") or {}
        subscription_plan = user_data.get("subscription_plan", "").lower()

        # Solo usuarios FREE tienen restricción
        if subscription_plan != "free":
            return None, {"display": "none"}

        # Issue #435: Verificar si fecha está fuera de rango (3 meses)
        if not start_date:
            # Sin fecha seleccionada, no mostrar banner
            return None, {"display": "none"}

        try:
            # Calcular límite (3 meses desde hoy, aproximación)
            date_limit = datetime.now(timezone.utc) - timedelta(days=90)

            # Parsear fecha seleccionada
            if isinstance(start_date, str):
                selected_date = datetime.fromisoformat(start_date.replace("Z", "+00:00"))
            else:
                selected_date = start_date

            # Solo mostrar banner si fecha está FUERA de rango
            if selected_date < date_limit:
                date_limit_str = date_limit.strftime("%d/%m/%Y")
                logger.debug(f"[DATE_BANNER] FREE user selected old date ({selected_date.date()}) - showing banner")

                banner = dbc.Alert(
                    [
                        html.I(className="fas fa-info-circle me-2"),
                        html.Span(
                            f"Tu plan gratuito incluye datos desde {date_limit_str}. "
                            "Los datos mostrados corresponden a ese período."
                        ),
                    ],
                    color="info",
                    dismissable=True,
                    className="mt-2 mb-0 py-2",
                    style={"fontSize": "0.85rem"},
                )
                return banner, {"display": "block"}

            # Fecha dentro del rango, no mostrar banner
            return None, {"display": "none"}

        except Exception as e:
            logger.error(f"[DATE_BANNER] Error checking date: {e}")
            return None, {"display": "none"}

    @app.callback(
        Output("prescription-date-restriction-banner", "children"),
        Output("prescription-date-restriction-banner", "style"),
        [Input("url", "pathname")],
        [State("auth-state", "data")],
        prevent_initial_call=False,
    )
    def show_prescription_date_banner(pathname, auth_state):
        """
        Show date restriction banner in prescription page for FREE users.

        Este callback actualiza el elemento dentro del layout de prescription.py.
        El placeholder del skeleton fue ELIMINADO para evitar IDs duplicados
        que causaban un gap de layout entre el sidebar y el contenido.

        Args:
            pathname: Current URL path
            auth_state: User authentication state

        Returns:
            Tuple of (banner_children, banner_style)
        """
        # Only show on prescription page
        if not pathname or not pathname.startswith("/prescription"):
            return None, {"display": "none"}

        # Check if user is authenticated
        if not is_user_authenticated(auth_state):
            return None, {"display": "none"}

        # Check subscription plan
        user_data = auth_state.get("user") or {}
        subscription_plan = user_data.get("subscription_plan", "").lower()

        if subscription_plan == "free":
            logger.debug("[DATE_BANNER] Showing prescription date restriction banner for FREE user")
            return _create_date_restriction_banner(), {"display": "block"}

        return None, {"display": "none"}

    # =========================================================================
    # Issue #420 Phase 3: Disable controls for FREE users (read-only mode)
    # =========================================================================

    @app.callback(
        Output("generics-date-range", "disabled"),
        Output("generics-date-slider", "disabled"),
        [Input("url", "pathname")],
        [State("auth-state", "data")],
        prevent_initial_call=False,
    )
    def disable_generics_date_controls(pathname, auth_state):
        """
        Disable date picker and slider for FREE users - they can only see the allowed range.

        FREE users have data restricted to 3 months from last sale.
        The DatePicker and Slider show the range but are not interactive.
        """
        if not pathname or pathname != "/generics":
            return False, False

        if not is_user_authenticated(auth_state):
            return False, False

        user_data = auth_state.get("user") or {}
        subscription_plan = user_data.get("subscription_plan", "").lower()

        if subscription_plan == "free":
            logger.debug("[DATE_CONTROLS] Disabling date picker and slider for FREE user")
            return True, True

        return False, False

    # NOTE: Employee filter disabled/placeholder is handled by
    # callbacks/generics.py:load_employee_options() to avoid duplicate Outputs

    @app.callback(
        Output("generics-date-range-store", "data"),
        [Input("url", "pathname")],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def fetch_generics_date_range(pathname, auth_state, auth_tokens):
        """Obtiene el rango de fechas disponible para generics."""
        from dash.exceptions import PreventUpdate
        from utils.auth_helpers import get_auth_headers_from_tokens

        if pathname != "/generics":
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            raise PreventUpdate

        user = auth_state.get("user") or {}
        pharmacy_id = user.get("pharmacy_id")

        if not pharmacy_id:
            raise PreventUpdate

        response = request_coordinator.make_request(
            endpoint=f"/api/v1/sales/date-range/{pharmacy_id}",
            method="GET",
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response or "error" in response:
            # Fallback: último año
            end_date = datetime.now(timezone.utc).date()
            start_date = end_date - timedelta(days=365)
            return {
                "min_date": start_date.isoformat(),
                "max_date": end_date.isoformat(),
                "ready": True
            }

        min_date = response.get("min_date") or response.get("oldest_date")
        max_date = response.get("max_date") or response.get("latest_date")

        if not min_date or not max_date:
            end_date = datetime.now(timezone.utc).date()
            start_date = end_date - timedelta(days=365)
            return {
                "min_date": start_date.isoformat(),
                "max_date": end_date.isoformat(),
                "ready": True
            }

        logger.info(f"[fetch_generics_date_range] Date range: {min_date} to {max_date}")
        return {"min_date": min_date, "max_date": max_date, "ready": True}

    @app.callback(
        Output("generics-date-range", "start_date"),
        Output("generics-date-range", "end_date"),
        Output("generics-date-range", "min_date_allowed"),
        Output("generics-date-range", "max_date_allowed"),
        Output("generics-date-slider", "min"),
        Output("generics-date-slider", "max"),
        Output("generics-date-slider", "marks"),
        Output("generics-date-slider", "value"),
        [Input("generics-date-range-store", "data")],
        [
            State("generics-date-range", "start_date"),
            State("generics-date-range", "end_date"),
            State("auth-state", "data"),
        ],
        prevent_initial_call=False,
    )
    def apply_generics_date_range(store_data, current_start, current_end, auth_state):
        """
        Aplica el rango de fechas al DatePicker y Slider.

        - Fecha de inicio por defecto: 1 de enero del año actual
        - FREE tier: últimos 90 días
        """
        from dash.exceptions import PreventUpdate

        if not store_data or not store_data.get("ready"):
            raise PreventUpdate

        min_date = store_data.get("min_date")
        max_date = store_data.get("max_date")

        if not min_date or not max_date:
            raise PreventUpdate

        max_dt = datetime.fromisoformat(max_date)
        min_dt = datetime.fromisoformat(min_date)

        # Verificar si es FREE tier
        user_data = (auth_state.get("user") or {}) if auth_state else {}
        subscription_plan = user_data.get("subscription_plan", "").lower()

        if subscription_plan == "free":
            # FREE tier: últimos 90 días
            end_date = max_dt
            start_date = max(min_dt, max_dt - timedelta(days=90))
            # Restringir el rango permitido
            min_date_allowed = start_date.strftime("%Y-%m-%d")
            max_date_allowed = max_date
        else:
            # PRO/MAX: 1 de enero del año actual
            current_year = datetime.now().year
            default_start = datetime(current_year, 1, 1)

            # Asegurar que esté dentro del rango de datos
            if default_start < min_dt:
                default_start = min_dt
            if default_start > max_dt:
                default_start = max_dt - timedelta(days=90)

            # Verificar fecha guardada
            if current_start:
                try:
                    saved_start = datetime.fromisoformat(current_start)
                    if saved_start < default_start:
                        start_date = default_start
                    else:
                        start_date = saved_start
                except (ValueError, TypeError):
                    start_date = default_start
            else:
                start_date = default_start

            end_date = datetime.fromisoformat(current_end) if current_end else max_dt
            min_date_allowed = min_date
            max_date_allowed = max_date

        # Calcular slider
        total_days = (max_dt - min_dt).days
        marks = {}
        current = min_dt
        idx = 0
        while current <= max_dt:
            if current.day == 1:
                marks[idx] = format_month_year(current)
            idx += 1
            current += timedelta(days=1)

        if len(marks) > 12:
            step = len(marks) // 6
            marks = {k: v for i, (k, v) in enumerate(marks.items()) if i % step == 0}

        # Calcular valores del slider
        slider_start = (start_date - min_dt).days if isinstance(start_date, datetime) else 0
        slider_end = (end_date - min_dt).days if isinstance(end_date, datetime) else total_days
        slider_start = max(0, min(slider_start, total_days))
        slider_end = max(0, min(slider_end, total_days))

        start_date_str = start_date.strftime("%Y-%m-%d") if isinstance(start_date, datetime) else str(start_date)
        end_date_str = end_date.strftime("%Y-%m-%d") if isinstance(end_date, datetime) else str(end_date)

        logger.info(f"[apply_generics_date_range] Applied: {start_date_str} to {end_date_str}")

        return (
            start_date_str,
            end_date_str,
            min_date_allowed,
            max_date_allowed,
            0,
            total_days,
            marks,
            [slider_start, slider_end],
        )

    @app.callback(
        Output("generics-date-range", "start_date", allow_duplicate=True),
        Output("generics-date-range", "end_date", allow_duplicate=True),
        [Input("generics-date-slider", "value")],
        [State("generics-date-range-store", "data")],
        prevent_initial_call=True,
    )
    def sync_generics_slider_to_datepicker(slider_value, store_data):
        """Sincronizar slider con DatePicker."""
        from dash.exceptions import PreventUpdate

        if not slider_value or not store_data:
            raise PreventUpdate

        min_date = store_data.get("min_date")
        if not min_date:
            raise PreventUpdate

        try:
            min_dt = datetime.fromisoformat(min_date)
            start_dt = min_dt + timedelta(days=slider_value[0])
            end_dt = min_dt + timedelta(days=slider_value[1])
            return start_dt.strftime("%Y-%m-%d"), end_dt.strftime("%Y-%m-%d")
        except Exception as e:
            logger.debug(f"[sync_generics_slider_to_datepicker] Error: {e}")
            raise PreventUpdate
