# frontend/callbacks/settings.py
"""
Callbacks para la página de Ajustes (Issue #155).
Maneja cambio de tabs, carga y guardado de perfil/farmacia/preferencias.
"""

import logging

import dash_bootstrap_components as dbc
from components.toast_manager import error_toast, success_toast, warning_toast
from dash import Input, Output, State, no_update
from dash import html

from pages.settings_page import (
    PLAN_CONFIG,
    create_pharmacy_tab,
    create_preferences_tab,
    create_profile_tab,
    create_subscription_tab,
)
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_settings_callbacks(app):
    """
    Registrar todos los callbacks de la página de ajustes.

    Args:
        app: Instancia de la aplicación Dash
    """

    # Callback para cambiar tabs (REGLA #11: Un Input → Un Callback)
    # Este callback actualiza TANTO el contenido como el store intermedio
    @app.callback(
        [Output("settings-tab-content", "children"), Output("settings-active-tab-store", "data")],
        Input("settings-tabs", "active_tab"),
        prevent_initial_call=False,
    )
    def change_settings_tab(active_tab):
        """
        Cambia el contenido según el tab activo y actualiza store intermedio.

        Args:
            active_tab: ID del tab activo ('perfil', 'farmacia', 'preferencias', 'suscripcion')

        Returns:
            Tuple: (Layout del tab correspondiente, Store data con tab activo)
        """
        if active_tab == "perfil":
            return create_profile_tab(), {"active_tab": "perfil"}
        elif active_tab == "farmacia":
            return create_pharmacy_tab(), {"active_tab": "farmacia"}
        elif active_tab == "preferencias":
            return create_preferences_tab(), {"active_tab": "preferencias"}
        elif active_tab == "suscripcion":
            # Issue #420: FREE tier experience - Subscription tab
            return create_subscription_tab(), {"active_tab": "suscripcion"}

        # Default: perfil
        return create_profile_tab(), {"active_tab": "perfil"}

    # Callback para cargar datos de perfil al abrir la página
    @app.callback(
        [
            Output("profile-full-name", "value", allow_duplicate=True),
            Output("profile-email", "value", allow_duplicate=True),
            Output("profile-phone", "value", allow_duplicate=True),
            Output("profile-username", "value", allow_duplicate=True),
            Output("profile-notifications", "value", allow_duplicate=True),
        ],
        Input("url", "pathname"),
        [
            State("auth-state", "data"),  # Issue #302: Agregar auth_state
            State("auth-tokens-store", "data"),  # REGLA #7.6: Multi-worker token restoration
        ],
        prevent_initial_call='initial_duplicate',  # Dash 3.x: Permite carga inicial + duplicados
    )
    def load_profile_data(pathname, auth_state, auth_tokens):  # Issue #302 + REGLA #7.6
        """
        Carga datos del perfil del usuario actual.

        Args:
            pathname: Ruta actual (detecta cuando entra a /ajustes)
            auth_state: Estado de autenticación (Issue #302)
            auth_tokens: Tokens de autenticación (REGLA #7.6)

        Returns:
            Valores para campos del formulario de perfil

        Issue #302: Verificación proactiva de autenticación antes de llamadas API.
        REGLA #7.6: Multi-worker token restoration para Render.
        """
        # Issue #302: Verificación proactiva de autenticación
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[SETTINGS_PROFILE] User not authenticated - skipping profile load")
            return no_update, no_update, no_update, no_update, no_update

        # Solo cargar datos si está en página de ajustes
        if not pathname or not pathname.startswith("/ajustes"):
            return no_update, no_update, no_update, no_update, no_update

        # REGLA #7.6: Multi-worker token restoration - EXPLICIT HEADERS
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        try:
            # Llamar a API para obtener datos del usuario actual
            # Fix Problema #4: Corregir ruta del endpoint (falta /auth)
            # REGLA #7.6: Pass explicit auth_headers for multi-worker support
            response = request_coordinator.make_request(
                "/api/v1/auth/me", method="GET", timeout=10, auth_headers=auth_headers
            )

            if response and "error" not in response:
                user_data = response

                # Mapear datos de usuario a campos
                full_name = user_data.get("full_name", "")
                email = user_data.get("email", "")
                phone = user_data.get("phone", "")
                username = user_data.get("username", "")

                # Notificaciones (mapear boolean a lista)
                notifications = []
                if user_data.get("notify_uploads", True):
                    notifications.append("uploads")
                if user_data.get("notify_errors", True):
                    notifications.append("errors")

                return full_name, email, phone, username, notifications

            logger.warning(f"No se pudieron cargar datos de perfil: {response.get('error', 'Unknown')}")
            return "", "", "", "", ["uploads", "errors"]

        except Exception as e:
            logger.error(f"Error cargando datos de perfil: {str(e)}")
            return "", "", "", "", ["uploads", "errors"]

    # CALLBACKS SEPARADOS para cada tab (necesario por reemplazo dinámico de contenido)
    # Cada callback solo escucha componentes de su propio tab

    # Callback para guardar perfil
    @app.callback(
        [
            Output("profile-save-feedback", "children", allow_duplicate=True),
            Output("toast-trigger-store", "data", allow_duplicate=True),
        ],
        Input("save-profile-btn", "n_clicks"),
        [
            State("profile-full-name", "value"),
            State("profile-phone", "value"),
            State("profile-current-password", "value"),
            State("profile-new-password", "value"),
            State("profile-confirm-password", "value"),
            State("profile-notifications", "value"),
            State("auth-tokens-store", "data"),  # Issue #442: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def save_profile_settings(n_clicks, full_name, phone, current_password, new_password, confirm_password, notifications, auth_tokens):
        """Callback para guardar cambios de perfil."""
        if n_clicks is None or n_clicks == 0:
            return no_update, no_update

        # Issue #442: Multi-worker token restoration - EXPLICIT HEADERS
        # Use centralized helper for token extraction (REGLA #7.6)
        from utils.auth_helpers import get_auth_headers_from_tokens

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        feedback, toast = _save_profile_logic(
            full_name, phone, current_password, new_password, confirm_password, notifications, auth_headers
        )
        return feedback, toast

    # Callback para guardar farmacia
    @app.callback(
        [
            Output("pharmacy-save-feedback", "children", allow_duplicate=True),
            Output("toast-trigger-store", "data", allow_duplicate=True),
        ],
        Input("save-pharmacy-btn", "n_clicks"),
        [
            State("pharmacy-name", "value"),
            State("pharmacy-code", "value"),
            State("pharmacy-address", "value"),
            State("pharmacy-city", "value"),
            State("pharmacy-phone", "value"),
            State("primary-erp-type", "value"),
            State("erp-version", "value"),
            State("auth-tokens-store", "data"),  # Issue #442: Multi-worker token restoration
        ],
        prevent_initial_call=True,
    )
    def save_pharmacy_settings(n_clicks, pharmacy_name, pharmacy_code, pharmacy_address, pharmacy_city, pharmacy_phone, erp_type, erp_version, auth_tokens):
        """Callback para guardar cambios de farmacia."""
        if n_clicks is None or n_clicks == 0:
            return no_update, no_update

        # Issue #442: Multi-worker token restoration - EXPLICIT HEADERS
        # Use centralized helper for token extraction (REGLA #7.6)
        from utils.auth_helpers import get_auth_headers_from_tokens

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        feedback, toast = _save_pharmacy_logic(
            pharmacy_name, pharmacy_code, pharmacy_address, pharmacy_city, pharmacy_phone, erp_type, erp_version, auth_headers
        )
        return feedback, toast

    # Callback para guardar preferencias
    @app.callback(
        [
            Output("preferences-save-feedback", "children", allow_duplicate=True),
            Output("toast-trigger-store", "data", allow_duplicate=True),
        ],
        Input("save-preferences-btn", "n_clicks"),
        [
            State("refresh-interval", "value"),
            State("system-notifications", "value"),
            State("notification-email", "value"),
        ],
        prevent_initial_call=True,
    )
    def save_preferences_settings(n_clicks, refresh_interval, system_notifications, notification_email):
        """Callback para guardar cambios de preferencias."""
        if n_clicks is None or n_clicks == 0:
            return no_update, no_update

        feedback, toast = _save_preferences_logic(refresh_interval, system_notifications, notification_email)
        return feedback, toast

    def _save_profile_logic(full_name, phone, current_password, new_password, confirm_password, notifications, auth_headers=None):
        """Helper para lógica de guardado de perfil (extraída del callback original).

        Args:
            auth_headers: Dict with Authorization header (Issue #442 multi-worker fix)
        """

        # Validar cambio de contraseña si se intenta
        if new_password or confirm_password:
            if not current_password:
                return (
                    dbc.Alert("Debes ingresar tu contraseña actual", color="warning", className="mt-3"),
                    warning_toast("Falta contraseña actual", "Validación"),
                )

            if new_password != confirm_password:
                return (
                    dbc.Alert("Las contraseñas nuevas no coinciden", color="warning", className="mt-3"),
                    warning_toast("Las contraseñas no coinciden", "Validación"),
                )

            if len(new_password) < 8:
                return (
                    dbc.Alert("La contraseña debe tener al menos 8 caracteres", color="warning", className="mt-3"),
                    warning_toast("Contraseña muy corta", "Validación"),
                )

        try:
            # Preparar payload para API
            payload = {}

            if full_name:
                payload["full_name"] = full_name
            if phone:
                payload["phone"] = phone

            # Notificaciones
            payload["notify_uploads"] = "uploads" in (notifications or [])
            payload["notify_errors"] = "errors" in (notifications or [])

            # Cambio de contraseña (si aplica)
            if new_password:
                payload["current_password"] = current_password
                payload["new_password"] = new_password

            # Llamar a API para actualizar perfil
            # Issue #442: Pass auth_headers EXPLICITLY for multi-worker support
            # return_error_details=True para obtener mensajes de validación específicos
            response = request_coordinator.make_request(
                "/api/v1/auth/me", method="PUT", data=payload, timeout=15,
                auth_headers=auth_headers, return_error_details=True
            )

            # Éxito: response es dict sin campo "error"
            if response and not response.get("error"):
                return (
                    dbc.Alert("Perfil actualizado correctamente", color="success", className="mt-3", duration=4000),
                    success_toast("Cambios guardados exitosamente", "Perfil Actualizado"),
                )

            # Error: extraer mensaje detallado
            if response and response.get("error"):
                error_msg = response.get("message", "Error de validación")
                # Si es 422, dar mensaje más útil sobre requisitos de contraseña
                if response.get("status_code") == 422 and new_password:
                    error_msg = ("La contraseña debe tener al menos 8 caracteres y contener "
                                "3 de 4: mayúsculas, minúsculas, números, caracteres especiales (!@#$%^&*)")
            else:
                error_msg = "Error de conexión"
            logger.error(f"Error guardando perfil: {error_msg}")

            return (
                dbc.Alert(f"Error al guardar: {error_msg}", color="danger", className="mt-3"),
                error_toast(f"No se pudieron guardar los cambios: {error_msg}", "Error"),
            )

        except Exception as e:
            logger.error(f"Excepción guardando perfil: {str(e)}")
            return (
                dbc.Alert(f"Error inesperado: {str(e)}", color="danger", className="mt-3"),
                error_toast("Error inesperado al guardar", "Error"),
            )

    def _save_pharmacy_logic(
        pharmacy_name, pharmacy_code, pharmacy_address, pharmacy_city, pharmacy_phone, erp_type, erp_version, auth_headers=None
    ):
        """Helper para lógica de guardado de farmacia (extraída del callback original).

        Args:
            auth_headers: Dict with Authorization header (Issue #442 multi-worker fix)
        """
        try:
            # Obtener pharmacy_id del usuario actual
            # Issue #442: Pass auth_headers EXPLICITLY for multi-worker support
            user_response = request_coordinator.make_request(
                "/api/v1/auth/me", method="GET", timeout=10, auth_headers=auth_headers
            )

            if not user_response or "error" in user_response:
                return (
                    dbc.Alert("Error al obtener información del usuario", color="danger", className="mt-3"),
                    error_toast("No se pudo identificar la farmacia", "Error"),
                )

            pharmacy_id = user_response.get("pharmacy_id")
            if not pharmacy_id:
                return (
                    dbc.Alert("Usuario sin farmacia asignada", color="warning", className="mt-3"),
                    warning_toast("No hay farmacia asociada", "Advertencia"),
                )

            # Preparar payload
            payload = {}
            if pharmacy_name:
                payload["name"] = pharmacy_name
            if pharmacy_code:
                payload["code"] = pharmacy_code
            if pharmacy_address:
                payload["address"] = pharmacy_address
            if pharmacy_city:
                payload["city"] = pharmacy_city
            if pharmacy_phone:
                payload["phone"] = pharmacy_phone
            if erp_type:
                payload["erp_type"] = erp_type
            if erp_version:
                payload["erp_version"] = erp_version

            # Llamar a API para actualizar farmacia
            # Issue #442: Pass auth_headers EXPLICITLY for multi-worker support
            response = request_coordinator.make_request(
                f"/api/v1/pharmacy/{pharmacy_id}", method="PUT", data=payload, timeout=15, auth_headers=auth_headers
            )

            if response and "error" not in response:
                return (
                    dbc.Alert(
                        "Datos de farmacia actualizados correctamente", color="success", className="mt-3", duration=4000
                    ),
                    success_toast("Cambios guardados exitosamente", "Farmacia Actualizada"),
                )

            error_msg = response.get("error", "Error desconocido") if response else "Error de conexión"
            logger.error(f"Error guardando farmacia: {error_msg}")

            return (
                dbc.Alert(f"Error al guardar: {error_msg}", color="danger", className="mt-3"),
                error_toast(f"No se pudieron guardar los cambios: {error_msg}", "Error"),
            )

        except Exception as e:
            logger.error(f"Excepción guardando farmacia: {str(e)}")
            return (
                dbc.Alert(f"Error inesperado: {str(e)}", color="danger", className="mt-3"),
                error_toast("Error inesperado al guardar", "Error"),
            )

    def _save_preferences_logic(refresh_interval, system_notifications, notification_email):
        """Helper para lógica de guardado de preferencias (extraída del callback original)."""
        # TODO: Implementar guardado de preferencias en backend cuando exista endpoint
        # Por ahora, solo mostrar mensaje de éxito

        logger.info(
            f"Preferencias guardadas (solo local): refresh={refresh_interval}, notifications={system_notifications}"
        )

        return (
            dbc.Alert("Preferencias guardadas correctamente", color="success", className="mt-3", duration=4000),
            success_toast("Preferencias actualizadas", "Preferencias"),
        )

    # Callback para cancelar cambios de perfil
    @app.callback(
        [
            Output("profile-current-password", "value"),
            Output("profile-new-password", "value"),
            Output("profile-confirm-password", "value"),
        ],
        Input("cancel-profile-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def cancel_profile_changes(n_clicks):
        """
        Limpia campos de contraseña al cancelar.

        Args:
            n_clicks: Número de clics en botón cancelar

        Returns:
            Limpia campos de contraseña
        """
        if not n_clicks:
            return no_update, no_update, no_update

        # Limpiar campos de contraseña
        return "", "", ""

    # ==========================================================================
    # REGLA #11 FIX: Callback unificado para cargar datos por tab
    # Combina load_pharmacy_data + load_subscription_data en UN solo callback
    # ==========================================================================
    @app.callback(
        [
            # Outputs de farmacia (9)
            Output("pharmacy-name", "value", allow_duplicate=True),
            Output("pharmacy-code", "value", allow_duplicate=True),
            Output("pharmacy-address", "value", allow_duplicate=True),
            Output("pharmacy-city", "value", allow_duplicate=True),
            Output("pharmacy-phone", "value", allow_duplicate=True),
            Output("primary-erp-type", "value", allow_duplicate=True),
            Output("erp-version", "value", allow_duplicate=True),
            Output("date-format", "value", allow_duplicate=True),
            Output("decimal-separator", "value", allow_duplicate=True),
            # Outputs de suscripción (3) - Issue #420
            Output("subscription-current-plan-content", "children", allow_duplicate=True),
            Output("subscription-storage-content", "children", allow_duplicate=True),
            Output("subscription-limits-content", "children", allow_duplicate=True),
        ],
        Input("settings-active-tab-store", "data"),
        State("auth-state", "data"),
        prevent_initial_call="initial_duplicate",
    )
    def load_tab_data(store_data, auth_state):
        """
        Callback unificado para cargar datos según el tab activo.

        REGLA #11 Fix: Un Input (settings-active-tab-store) -> Un callback.
        Combina la lógica de farmacia y suscripción en un solo callback.

        Args:
            store_data: Datos del store con el tab activo
            auth_state: Estado de autenticación

        Returns:
            Tuple con valores para farmacia (9) + suscripción (3)
        """
        from utils.auth_helpers import is_user_authenticated

        # Outputs por defecto (no_update para todos)
        pharmacy_defaults = (no_update,) * 9
        subscription_defaults = (no_update,) * 3

        if not is_user_authenticated(auth_state):
            logger.debug("[SETTINGS] User not authenticated - skipping tab load")
            return pharmacy_defaults + subscription_defaults

        if not store_data:
            return pharmacy_defaults + subscription_defaults

        active_tab = store_data.get("active_tab")

        # Tab Farmacia
        if active_tab == "farmacia":
            pharmacy_values = _load_pharmacy_data_impl(auth_state)
            return pharmacy_values + subscription_defaults

        # Tab Suscripción (Issue #420)
        if active_tab == "suscripcion":
            subscription_values = _load_subscription_data_impl(auth_state)
            return pharmacy_defaults + subscription_values

        # Otro tab - no hacer nada
        return pharmacy_defaults + subscription_defaults

    def _load_pharmacy_data_impl(auth_state):
        """Helper: Carga datos de farmacia."""
        try:
            user_response = request_coordinator.make_request("/api/v1/auth/me", method="GET", timeout=10)

            if not user_response or "error" in user_response:
                logger.warning("No se pudo obtener información del usuario")
                return ("", "", "", "", "", "farmatic", "", "dd/mm/yyyy", "comma")

            pharmacy_id = user_response.get("pharmacy_id")
            if not pharmacy_id:
                logger.warning("Usuario sin pharmacy_id asignado")
                return ("", "", "", "", "", "farmatic", "", "dd/mm/yyyy", "comma")

            pharmacy_response = request_coordinator.make_request(
                f"/api/v1/pharmacy/{pharmacy_id}", method="GET", timeout=10
            )

            if pharmacy_response and "error" not in pharmacy_response:
                pharmacy_data = pharmacy_response.get("data", pharmacy_response) if "data" in pharmacy_response else pharmacy_response

                return (
                    pharmacy_data.get("name", ""),
                    pharmacy_data.get("code", ""),
                    pharmacy_data.get("address", ""),
                    pharmacy_data.get("city", ""),
                    pharmacy_data.get("phone", ""),
                    pharmacy_data.get("erp_type", "farmatic"),
                    pharmacy_data.get("erp_version", ""),
                    "dd/mm/yyyy",
                    "comma",
                )

            logger.warning(f"No se pudieron cargar datos de farmacia: {pharmacy_response.get('error', 'Unknown')}")
            return ("", "", "", "", "", "farmatic", "", "dd/mm/yyyy", "comma")

        except Exception as e:
            logger.error(f"Error cargando datos de farmacia: {str(e)}")
            return ("", "", "", "", "", "farmatic", "", "dd/mm/yyyy", "comma")

    def _load_subscription_data_impl(auth_state):
        """Helper: Carga datos de suscripción (Issue #420)."""
        try:
            user_data = auth_state.get("user", {})
            subscription_plan = user_data.get("subscription_plan", "free")
            plan_config = PLAN_CONFIG.get(subscription_plan, PLAN_CONFIG["free"])

            plan_content = _build_plan_badge_content(subscription_plan, plan_config)

            storage_response = request_coordinator.make_request(
                "/api/v1/auth/storage-usage", method="GET", timeout=10
            )

            if storage_response and "error" not in storage_response:
                storage_used_mb = storage_response.get("total_used_mb", 0)
                storage_limit_mb = storage_response.get("limit_mb", plan_config["storage_mb"])
            else:
                logger.warning(f"[SUBSCRIPTION] Storage API failed: {storage_response}")
                storage_used_mb = 0
                storage_limit_mb = plan_config["storage_mb"]

            storage_content = _build_storage_progress(storage_used_mb, storage_limit_mb)
            limits_content = _build_limits_content(subscription_plan, plan_config)

            return (plan_content, storage_content, limits_content)

        except Exception as e:
            logger.error(f"Error cargando datos de suscripción: {str(e)}")
            error_content = dbc.Alert(
                "Error al cargar información de suscripción",
                color="danger",
                className="mb-0",
            )
            return (error_content, error_content, error_content)

    # Callback para restaurar preferencias por defecto
    @app.callback(
        [
            Output("refresh-interval", "value", allow_duplicate=True),
            Output("system-notifications", "value", allow_duplicate=True),
            Output("notification-email", "value", allow_duplicate=True),
        ],
        Input("reset-preferences-btn", "n_clicks"),
        prevent_initial_call=True,
    )
    def reset_preferences(n_clicks):
        """
        Restaura preferencias a valores por defecto.

        Args:
            n_clicks: Número de clics en botón restaurar

        Returns:
            Valores por defecto para preferencias
        """
        if not n_clicks:
            return no_update, no_update, no_update

        # Valores por defecto
        return "30", ["system"], ""

# ==========================================================================
# Issue #420: Helper functions for Subscription Tab
# ==========================================================================


def _build_plan_badge_content(plan: str, config: dict):
    """
    Build the current plan display with badge.

    Args:
        plan: Current subscription plan ('free', 'pro', 'max')
        config: Plan configuration dict

    Returns:
        Dash html.Div component with plan badge
    """
    return html.Div(
        [
            dbc.Badge(
                config["name"],
                color=config["badge_color"],
                className="fs-3 px-4 py-2 mb-3",
            ),
            html.P(
                [
                    "Actualmente estás en el plan ",
                    html.Strong(config["name"]),
                    ".",
                ],
                className="mb-2",
            ),
            html.P(
                [
                    html.I(className="fas fa-database me-2"),
                    f"Almacenamiento: {config['storage_mb']} MB",
                ],
                className="text-muted small mb-1",
            ),
            html.P(
                [
                    html.I(className="fas fa-headset me-2"),
                    f"Soporte: {config['support']}",
                ],
                className="text-muted small mb-0",
            ),
        ],
        className="text-center",
    )


def _build_storage_progress(used_mb: float, limit_mb: int):
    """
    Build storage progress bar.

    Args:
        used_mb: Storage used in MB
        limit_mb: Storage limit in MB

    Returns:
        Dash html.Div component with progress bar
    """
    percentage = min((used_mb / limit_mb) * 100, 100) if limit_mb > 0 else 0

    # Color based on usage
    if percentage > 90:
        color = "danger"
        message = "¡Almacenamiento casi lleno!"
    elif percentage > 70:
        color = "warning"
        message = "Considera actualizar tu plan pronto"
    else:
        color = "success"
        message = "Almacenamiento disponible"

    return html.Div(
        [
            html.Div(
                [
                    html.Span(f"{used_mb:.1f} MB", className="fw-bold"),
                    html.Span(f" / {limit_mb} MB", className="text-muted"),
                ],
                className="d-flex justify-content-between mb-2",
            ),
            dbc.Progress(
                value=percentage,
                color=color,
                striped=percentage > 70,
                animated=percentage > 90,
                className="mb-2",
                style={"height": "24px"},
            ),
            html.Div(
                [
                    html.Small(f"{percentage:.1f}% utilizado", className="text-muted"),
                    html.Small(f" - {message}", className=f"text-{color}"),
                ]
            ),
        ]
    )


def _build_limits_content(plan: str, config: dict):
    """
    Build limits info content.

    Args:
        plan: Current subscription plan ('free', 'pro', 'max')
        config: Plan configuration dict

    Returns:
        Dash html.Div component with limits info
    """
    if plan == "free":
        return html.Div(
            [
                html.P(
                    [
                        html.I(className="fas fa-exclamation-triangle me-2 text-warning"),
                        "Con el plan FREE solo puedes acceder a los datos de los ",
                        html.Strong("últimos 3 meses"),
                        " desde tu última venta registrada.",
                    ],
                    className="mb-2",
                ),
                html.P(
                    [
                        html.I(className="fas fa-info-circle me-2 text-info"),
                        "Actualiza a PRO o MAX para acceso ilimitado a tu historial.",
                    ],
                    className="mb-0 small text-muted",
                ),
            ]
        )
    else:
        return html.P(
            [
                html.I(className="fas fa-check-circle me-2 text-success"),
                "Tienes acceso ",
                html.Strong("ilimitado"),
                " a todo tu historial de datos.",
            ],
            className="mb-0",
        )
