# frontend/callbacks/security_local.py
"""
Security Callbacks for kaiFarma Local (Pivot 2026).

Handles lock screen interactions:
- Polling backend for lock status
- Unlock with PIN
- Manual lock
- Auto-lock detection

Only active when KAIFARMA_LOCAL=true
"""

import logging
import os

import requests
from dash import Input, Output, State, ctx, no_update
from dash.exceptions import PreventUpdate

logger = logging.getLogger(__name__)

# Backend URL (same machine in local mode)
BACKEND_URL = os.getenv("BACKEND_URL", "http://127.0.0.1:8000")


def register_security_local_callbacks(app):
    """
    Register all security-related callbacks for local mode.

    Called from callbacks/__init__.py when KAIFARMA_LOCAL=true
    """

    @app.callback(
        Output("lock-screen-modal", "is_open"),
        Output("lock-pharmacy-name", "children"),
        Output("pin-error", "children"),
        Output("local-auth-state", "data"),
        Input("security-pulse", "n_intervals"),
        Input("btn-unlock", "n_clicks"),
        State("pin-input", "value"),
        State("local-auth-state", "data"),
        prevent_initial_call=True,  # Wait for interval/button - modal starts visible
    )
    def manage_lock_screen(n_intervals, n_clicks, pin, auth_state):
        """
        Main lock screen controller.

        Handles:
        1. Polling backend for lock status (every 5s via security-pulse interval)
        2. Unlock attempts with PIN
        3. Detecting auto-lock events

        Note: prevent_initial_call=True is safe because:
        - Lock screen modal starts with is_open=True (always visible initially)
        - First interval fires within 5s and updates state from backend
        - This avoids DASH006 anti-pattern (API calls on initial page load)

        Returns:
            is_open: Whether modal should be visible
            pharmacy_name: Name to display in modal
            error_message: Error text (empty if no error)
            auth_state: Updated auth state
        """
        trigger_id = ctx.triggered_id

        # Default auth state if None
        if auth_state is None:
            auth_state = {"unlocked": False, "pharmacy": "", "initialized": False}

        # Case 1: Unlock attempt
        if trigger_id == "btn-unlock" and pin:
            try:
                resp = requests.post(
                    f"{BACKEND_URL}/api/v1/auth/local/unlock",
                    json={"pin": pin},
                    timeout=5,
                )

                if resp.status_code == 200:
                    logger.info("[SECURITY_LOCAL] Terminal unlocked successfully")
                    new_state = {
                        **auth_state,
                        "unlocked": True,
                        "initialized": True,
                    }
                    return False, no_update, "", new_state

                else:
                    # Wrong PIN
                    logger.warning("[SECURITY_LOCAL] Invalid PIN attempt")
                    return True, no_update, "PIN incorrecto", auth_state

            except requests.Timeout:
                logger.error("[SECURITY_LOCAL] Backend timeout on unlock")
                return True, no_update, "Timeout - intenta de nuevo", auth_state

            except requests.RequestException as e:
                logger.error(f"[SECURITY_LOCAL] Connection error: {e}")
                return True, no_update, "Error de conexion", auth_state

        # Case 2: Status polling (heartbeat)
        try:
            resp = requests.get(
                f"{BACKEND_URL}/api/v1/auth/local/status",
                timeout=3,
            )

            if resp.status_code == 200:
                data = resp.json()
                is_unlocked = data.get("is_unlocked", False)
                pharmacy = data.get("pharmacy_name", "Farmacia")
                state = data.get("state", "locked")

                # Update auth state
                new_state = {
                    "unlocked": is_unlocked,
                    "pharmacy": pharmacy,
                    "initialized": state != "uninitialized",
                }

                # Show modal if locked, hide if unlocked
                should_show_modal = not is_unlocked

                # Clear error on successful poll
                return should_show_modal, pharmacy, "", new_state

            else:
                logger.warning(f"[SECURITY_LOCAL] Unexpected status code: {resp.status_code}")
                return True, "Error de estado", "", auth_state

        except requests.Timeout:
            # On timeout, keep current state (don't lock user out due to slow response)
            logger.debug("[SECURITY_LOCAL] Status poll timeout")
            return no_update, no_update, no_update, no_update

        except requests.RequestException as e:
            logger.error(f"[SECURITY_LOCAL] Status poll failed: {e}")
            # On connection error, show lock screen with error
            return True, "Conexion perdida", "", auth_state

    @app.callback(
        Output("pin-input", "value"),
        Input("lock-screen-modal", "is_open"),
        prevent_initial_call=True,
    )
    def clear_pin_on_close(is_open):
        """Clear PIN input when modal opens/closes."""
        if is_open:
            return ""
        return no_update

    # Optional: Manual lock button callback
    @app.callback(
        Output("local-auth-state", "data", allow_duplicate=True),
        Input("btn-manual-lock", "n_clicks"),
        State("local-auth-state", "data"),
        prevent_initial_call=True,
    )
    def manual_lock(n_clicks, auth_state):
        """
        Handle manual lock button click (if present in navbar).

        Calls backend to lock terminal immediately.
        """
        if not n_clicks:
            raise PreventUpdate

        try:
            resp = requests.post(
                f"{BACKEND_URL}/api/v1/auth/local/lock",
                timeout=5,
            )

            if resp.status_code == 200:
                logger.info("[SECURITY_LOCAL] Terminal locked manually")
                return {**auth_state, "unlocked": False}

        except requests.RequestException as e:
            logger.error(f"[SECURITY_LOCAL] Manual lock failed: {e}")

        raise PreventUpdate
