# frontend/callbacks/routing.py
"""
Callbacks de routing para navegación multi-página.

Pivot 2026: En modo local (KAIFARMA_LOCAL=true), el routing NO verifica
autenticación JWT. La seguridad se maneja via lock screen modal con PIN.
"""

import os

from components.common import create_empty_state
from dash import Input, Output, State, html
from layouts import get_dashboard_layout, get_upload_layout
from layouts.admin import create_admin_layout
from layouts.auth import create_forgot_password_layout, create_login_layout
from layouts.generics import get_generics_layout
from layouts.homepage import create_homepage_layout
from layouts.landing import create_landing_layout
from layouts.prescription import get_prescription_layout  # Issue #400
# DEPRECATED (ADR-004): Clustering dinámico reemplazado por grupos curados
from layouts.clustering import get_clustering_layout  # Issue #458 - DEPRECADO
from layouts.ventalibre import get_ventalibre_layout  # Issue #461
from pages.settings_page import create_settings_layout  # Issue #155
from pages.test_design_system import create_test_design_system_layout
from utils.auth_helpers import is_user_authenticated

# Pivot 2026: Detect local mode for auth bypass
IS_LOCAL_MODE = os.getenv("KAIFARMA_LOCAL", "").lower() == "true"

# Constants for interval states (Issue #147)
# These control viewport-interval, health-check-interval (system-status-interval eliminado)
INTERVALS_DISABLED = (True, True)  # For static pages (landing, auth, 404)
INTERVALS_ENABLED = (False, False)  # For protected pages with monitoring


def register_routing_callbacks(app):
    """
    Registrar callbacks de routing.

    Args:
        app: Instancia de la aplicación Dash
    """

    @app.callback(
        [
            Output("page-content", "children"),
            Output("viewport-interval", "disabled"),
            Output("health-check-interval", "disabled"),
        ],
        [Input("url", "pathname"), Input("auth-state", "data"), Input("auth-ready", "data")],  # Issue #429: auth-ready Input para esperar sync_auth_context
        # NOTE: prevent_initial_call=False (default) is REQUIRED here
        # The callback MUST fire on initial page load to replace the loading spinner
        # with actual content. Without this, users see a loading spinner indefinitely.
    )
    def display_page(pathname, auth_state, auth_ready):
        """
        Mostrar contenido de página según la URL con verificación de autenticación.

        Issue #429: Agregar auth_ready como Input para esperar a que sync_auth_context
        complete la sincronización del token antes de verificar autenticación.

        Pivot 2026: En modo local (KAIFARMA_LOCAL=true), se salta la verificación
        JWT. La seguridad se maneja via lock screen modal con PIN.

        Args:
            pathname: Ruta de la URL
            auth_state: Estado de autenticación del usuario
            auth_ready: Flag que indica si sync_auth_context completó (True/False)

        Returns:
            Layout de la página correspondiente o redirección a login
        """
        import logging

        logger = logging.getLogger(__name__)
        logger.info(f"[ROUTING] display_page called with pathname: {pathname}, auth_ready: {auth_ready}")

        # Normalizar pathname
        if pathname is None:
            pathname = "/"

        pathname = pathname.strip().lower()
        logger.info(f"[ROUTING] Normalized pathname: {pathname}")

        # ═══════════════════════════════════════════════════════════════
        # PIVOT 2026: Modo local bypass auth JWT
        # ═══════════════════════════════════════════════════════════════
        # En modo local:
        # - No hay JWT/OAuth - autenticación via PIN (lock screen modal)
        # - El lock screen modal bloquea interacción hasta PIN correcto
        # - Routing siempre considera usuario "autenticado"
        # - Root (/) redirige a /home (no hay landing page en local)
        # ═══════════════════════════════════════════════════════════════
        if IS_LOCAL_MODE:
            logger.info("[ROUTING] LOCAL MODE - bypassing JWT auth check")
            is_authenticated = True  # Lock screen handles real security

            # En modo local, root siempre va a /home
            if pathname == "/":
                logger.info("[ROUTING] LOCAL MODE - redirecting root to /home")
                from dash import dcc
                return (
                    html.Div([dcc.Location(id="redirect-to-home", pathname="/home", refresh=True)]),
                    *INTERVALS_ENABLED,
                )

            # Skip auth pages in local mode (no login/register needed)
            if pathname in ["/auth/login", "/login", "/auth/register", "/auth/forgot-password"]:
                logger.info("[ROUTING] LOCAL MODE - auth pages not needed, redirect to /home")
                from dash import dcc
                return (
                    html.Div([dcc.Location(id="redirect-to-home", pathname="/home", refresh=True)]),
                    *INTERVALS_ENABLED,
                )

        else:
            # ═══════════════════════════════════════════════════════════════
            # CLOUD MODE: Verificación de autenticación PER-BROWSER
            # ═══════════════════════════════════════════════════════════════
            # IMPORTANTE: Usar auth_state (per-browser via session storage), NO
            # auth_manager.is_authenticated() que es un singleton compartido entre
            # TODOS los browsers del servidor.
            #
            # Bug fix: auth_manager singleton causaba que si Chrome se autenticaba,
            # Edge también aparecía "autenticado" porque compartían el mismo
            # auth_manager.access_token en memoria del servidor Python.
            #
            # auth_ready indica si sync_auth_context ya completó:
            # - auth_ready = False/None → Sync aún no ejecutó, mostrar loading
            # - auth_ready = True → Sync completó, auth_state es confiable
            # ═══════════════════════════════════════════════════════════════

            # Si sync_auth_context aún no completó, mostrar loading para evitar flash
            if not auth_ready:
                logger.info(f"[ROUTING] auth_ready={auth_ready} - waiting for sync_auth_context to complete")
                import dash_bootstrap_components as dbc
                return (
                    html.Div(
                        [
                            dbc.Spinner(color="primary", size="lg"),
                            html.P(
                                "Cargando...",
                                className="text-center text-muted mt-3"
                            )
                        ],
                        className="d-flex flex-column align-items-center justify-content-center",
                        style={"minHeight": "100vh"}
                    ),
                    *INTERVALS_DISABLED,
                )

            # auth_ready = True → sync completó, ahora auth_state es confiable
            is_authenticated = is_user_authenticated(auth_state)
            logger.info(f"[ROUTING] auth_ready=True, is_authenticated={is_authenticated}")

        # Landing page pública (ruta raíz)
        if pathname == "/":
            # FIX #2: Verificar autenticación ANTES de decidir qué mostrar
            # Si usuario autenticado en root, redirigir a /home (NO solo retornar layout)
            if is_authenticated:
                logger.info("[ROUTING] Authenticated user at root - redirecting to /home")
                # Usar dcc.Location para redirect REAL (no solo retornar homepage layout)
                from dash import dcc

                return (
                    html.Div([dcc.Location(id="redirect-to-home", pathname="/home", refresh=True)]),
                    *INTERVALS_ENABLED,
                )

            # Usuario no autenticado: mostrar landing page
            logger.info("[ROUTING] Returning landing page for unauthenticated user")
            return create_landing_layout(), *INTERVALS_DISABLED

        # Authentication pages (always accessible)
        # Soportar /login como alias de /auth/login (backward compatibility)
        if pathname in ["/auth/login", "/login"]:
            return create_login_layout(), *INTERVALS_DISABLED
        elif pathname == "/auth/register":
            # Registro deshabilitado - redirigir a landing
            logger.info("[ROUTING] Registration disabled - redirecting to landing")
            return create_landing_layout(), *INTERVALS_DISABLED
        elif pathname == "/auth/forgot-password":
            return create_forgot_password_layout(), *INTERVALS_DISABLED

        # Development/Testing pages (always accessible in development)
        if pathname == "/test-design-system":
            # Página de prueba del Design System (DEVELOPMENT ONLY - NO AUTH REQUIRED)
            logger.info("[ROUTING] Returning test design system page (public)")
            try:
                layout = create_test_design_system_layout()
                logger.info(f"[ROUTING] Test design system layout created successfully: {type(layout)}")
                return layout, *INTERVALS_DISABLED
            except Exception as e:
                logger.error(f"[ROUTING] Error creating test design system layout: {str(e)}", exc_info=True)
                return (
                    create_empty_state(
                        title="Error al cargar Design System",
                        message=f"Error: {str(e)}",
                        icon="fas fa-exclamation-triangle",
                    ),
                    *INTERVALS_DISABLED,
                )

        # Check authentication for protected routes
        # Issue #301: Redirección REAL con dcc.Location para evitar callbacks innecesarios
        if not is_authenticated:
            logger.info(f"[ROUTING] User not authenticated accessing {pathname} - redirecting to root")
            # Usar dcc.Location para redirect REAL (no solo retornar layout)
            # Esto evita que callbacks de la página protegida se ejecuten
            from dash import dcc
            import dash_bootstrap_components as dbc

            return (
                html.Div(
                    [
                        dcc.Location(id="redirect-to-login", pathname="/", refresh=True),
                        # Loader visible mientras redirige (previene flash de contenido protegido)
                        html.Div(
                            [
                                dbc.Spinner(color="primary", size="lg"),
                                html.P(
                                    "Verificando autenticación...",
                                    className="text-center text-muted mt-3"
                                )
                            ],
                            className="d-flex flex-column align-items-center justify-content-center",
                            style={"minHeight": "100vh"}
                        ),
                    ]
                ),
                *INTERVALS_DISABLED,
            )

        # Routing con Homepage como página principal (ruta protegida)
        # Habilitar intervalos en páginas protegidas (necesarios para monitoring)
        if pathname == "/home":
            logger.info("[ROUTING] Returning homepage layout")
            layout = create_homepage_layout()  # Homepage con panel de estado del sistema
            logger.info(f"[ROUTING] Homepage layout type: {type(layout)}")
            return layout, *INTERVALS_ENABLED
        elif pathname == "/dashboard":
            return get_dashboard_layout(), *INTERVALS_ENABLED
        elif pathname == "/upload":
            return get_upload_layout(), *INTERVALS_ENABLED
        elif pathname == "/config":
            # Redirect legacy /config to /ajustes/perfil
            from dash import dcc

            return (
                html.Div([dcc.Location(id="redirect-to-settings", pathname="/ajustes/perfil", refresh=True)]),
                *INTERVALS_ENABLED,
            )
        elif pathname.startswith("/ajustes"):
            # Página de ajustes con tabs (Issue #155)
            # Extraer tab activo de la ruta
            if pathname == "/ajustes" or pathname == "/ajustes/perfil":
                return create_settings_layout(active_tab="perfil"), *INTERVALS_ENABLED
            elif pathname == "/ajustes/farmacia":
                return create_settings_layout(active_tab="farmacia"), *INTERVALS_ENABLED
            elif pathname == "/ajustes/preferencias":
                return create_settings_layout(active_tab="preferencias"), *INTERVALS_ENABLED
            else:
                # Cualquier otra ruta /ajustes/* → tab perfil por defecto
                return create_settings_layout(active_tab="perfil"), *INTERVALS_ENABLED
        elif pathname == "/prescription":
            # Dashboard de análisis de prescripción (Issue #400)
            return get_prescription_layout(), *INTERVALS_ENABLED
        elif pathname == "/generics":
            # Página de genéricos con análisis completo
            return get_generics_layout(), *INTERVALS_ENABLED
        elif pathname == "/clustering":
            # DEPRECATED (ADR-004): Dashboard de clustering/taxonomía (Issue #458)
            # Reemplazado por grupos curados en /admin?tab=curated-groups
            return get_clustering_layout(), *INTERVALS_ENABLED
        elif pathname == "/ventalibre":
            # Dashboard de venta libre - Análisis por NECESIDAD (Issue #461)
            return get_ventalibre_layout(), *INTERVALS_ENABLED
        elif pathname == "/admin":
            # Panel de administración con dashboard de salud
            return create_admin_layout(), *INTERVALS_ENABLED
        else:
            # Página 404 - deshabilitar intervalos (página estática de error)
            return (
                create_empty_state(
                    title="Página no encontrada",
                    message="La página que buscas no existe.",
                    icon="fas fa-exclamation-triangle",
                    action_button={"text": "Ir al Inicio", "href": "/"},
                ),
                *INTERVALS_DISABLED,
            )

    @app.callback(
        [
            Output("context-store", "clear_data"),
            Output("analysis-store", "clear_data"),
            Output("partners-selection-store", "clear_data"),
            Output("codes-cache-store", "clear_data"),
            # ❌ REMOVED: laboratory-codes-cache eliminado (era duplicado de laboratory-cache-store)
            # laboratory-cache-store usa storage_type="session" - persiste en sesión, permite refresh entre sesiones
            Output("homogeneous-expansion-state", "clear_data"),
            Output("temporal-drill-store", "clear_data"),
            Output("homogeneous-drill-store", "clear_data"),
        ],
        Input("url", "pathname"),
        prevent_initial_call=True,
    )
    def cleanup_stores_on_navigation(pathname):
        """
        Limpiar stores pesados cuando usuario sale de /home.

        Issue: Memory leak en producción (14 MB/min)
        Root cause: Stores con storage_type="session" acumulan datos sin límite
        Stores afectados:
        - context-store: ~500 KB (universo sustituible completo)
        - analysis-store: ~1-2 MB (análisis dinámico con 1000 grupos)
        - codes-cache-store: ~100 KB (códigos laboratorios cacheados)
        - homogeneous-expansion-state: ~50 KB (estado UI)
        - temporal-drill-store: ~200 KB (drill-down temporal)
        - homogeneous-drill-store: ~200 KB (drill-down homogéneo)

        Total estimado: ~2.4 MB por sesión activa

        Nota: laboratory-cache-store (local) NO se limpia porque persiste en localStorage

        Solución: Limpiar stores cuando usuario navega fuera de /home
        Impacto: Memory leak → 0 MB/min
        """
        import logging

        logger = logging.getLogger(__name__)

        # Si navegamos fuera de /home o /generics, limpiar stores
        if pathname not in ["/home", "/generics"]:
            logger.info(f"[ROUTING] Clearing stores - navigated from /home or /generics to {pathname}")
            return [True, True, True, True, True, True, True]

        # Si estamos en /home o /generics, mantener stores
        return [False, False, False, False, False, False, False]

    # COMENTADO: Navbar collapse ya no existe - ahora usamos sidebar fijo
    # @app.callback(
    #     Output('navbar-collapse', 'is_open'),
    #     [Input('navbar-toggler', 'n_clicks')],
    #     prevent_initial_call=True
    # )
    # def toggle_navbar_collapse(n_clicks):
    #     """
    #     Toggle del navbar colapsable para móvil.
    #     OBSOLETO: Ahora usamos sidebar fijo
    #     """
    #     pass
