"""
Callbacks for prescription seasonality Tab 3.

Issue #489: Tendencias y Estacionalidad.
Issue #498: STL Decomposition + Forecast.
Issue #500: Stock-out Risk Matrix.
Issue #501: Anomaly Detection + Alerts.
Issue #502: Export CSV + Badges.
Issue #532: Replace category patterns with ATC filter.

Handles:
- Loading seasonality data from backend with ATC filter
- Rendering heatmap and monthly index charts
- STL decomposition chart
- Forecast chart with confidence intervals
- KPIs display
- Stock-out risk matrix
- Anomaly alerts table
- CSV export for forecast
- Tab badges (anomalies, next peak, stockout)
"""

import logging
from datetime import datetime
from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
from dash import Input, Output, State, callback, ctx, html, no_update
from dash.exceptions import PreventUpdate

from utils.helpers import MESES_ESPANOL_CORTO

from components.prescription.heatmap_hourly import (
    create_hourly_heatmap_chart,
    create_peak_info_text,
)
from components.prescription.monthly_index_chart import (
    create_monthly_index_chart,
    create_data_quality_alert,
    create_partial_month_note,  # Issue #529
)
from components.prescription.trend_decomposition_chart import (
    create_decomposition_chart,
    create_forecast_chart,
    create_decomposition_summary_card,
    create_forecast_summary_card,
    create_data_quality_alert_stl,
)
# Issue #532: category_patterns_chart removed - replaced by ATC filter
from components.prescription.stockout_risk_table import (
    create_stockout_risk_summary_badges,
    create_stockout_risk_table,
    create_empty_inventory_message,
)
from components.prescription.anomaly_alerts import (
    create_anomaly_summary_badges,
    create_anomaly_alerts_table,
    create_no_data_message,
)
from utils.auth_helpers import get_auth_headers_from_tokens, is_user_authenticated
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def register_seasonality_callbacks(app):
    """
    Registra callbacks para Tab 3 Seasonality.

    Callbacks:
    1. load_seasonality_data: Carga datos de heatmap, índice mensual y KPIs
    2. update_hourly_heatmap: Renderiza el heatmap
    3. update_monthly_index: Renderiza el gráfico de índice mensual
    """

    @app.callback(
        [
            Output("seasonality-heatmap-store", "data"),
            Output("seasonality-monthly-index-store", "data"),
            Output("seasonality-kpis-store", "data"),
            Output("seasonality-decomposition-store", "data"),  # Issue #498
            Output("seasonality-forecast-store", "data"),  # Issue #498
            # Issue #532: category-patterns-store removed - replaced by ATC filter
        ],
        [
            Input("url", "pathname"),
            Input("prescription-apply-filters-btn", "n_clicks"),
            Input("prescription-date-range-store", "data"),
            Input("auth-ready", "data"),
            Input("prescription-tabs", "active_tab"),  # Solo cargar cuando Tab 3 activo
            Input("seasonality-atc-filter", "value"),  # Issue #532: ATC filter
            # Issue #537 FIX: Añadir fechas como Inputs para recargar cuando cambian
            Input("prescription-date-range", "start_date"),
            Input("prescription-date-range", "end_date"),
        ],
        [
            State("prescription-categories-filter", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_seasonality_data(
        pathname: str,
        n_clicks: int,
        date_range_store: Optional[Dict],
        auth_ready: Optional[bool],
        active_tab: str,
        atc_filter: Optional[str],  # Issue #532: ATC filter
        start_date: str,
        end_date: str,
        categories: Optional[List[str]],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> tuple:
        """
        Carga datos de estacionalidad cuando Tab 3 está activo.

        Issue #489: Heatmap, monthly index, KPIs
        Issue #498: STL decomposition, Holt-Winters forecast
        Issue #532: ATC filter for all endpoints

        REGLA #7.6: Restaurar tokens multi-worker.
        REGLA #11: Solo un callback por Input.
        """
        logger.debug(f"[load_seasonality_data] Triggered - active_tab: {active_tab}")

        # Guard 1: Solo ejecutar en /prescription
        if pathname != "/prescription":
            raise PreventUpdate

        # Guard 2: Solo ejecutar cuando Tab 3 (seasonality) está activo
        if active_tab != "tab-tendencias":
            logger.debug("[load_seasonality_data] Tab 3 not active - skipping")
            raise PreventUpdate

        # Guard 3: Esperar auth-ready
        if not auth_ready:
            raise PreventUpdate

        # Guard 4: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[load_seasonality_data] User not authenticated - skipping")
            raise PreventUpdate

        # Guard 5: Verificar fechas
        if not start_date or not end_date:
            logger.debug("[load_seasonality_data] No date range - skipping")
            raise PreventUpdate

        # Guard 6: Verificar date_range_store ready
        if not date_range_store or not date_range_store.get("ready"):
            raise PreventUpdate

        # Obtener headers de autenticación (REGLA #7.6)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[load_seasonality_data] No auth headers - skipping")
            raise PreventUpdate

        # Obtener pharmacy_id
        pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_seasonality_data] No pharmacy_id - skipping")
            raise PreventUpdate

        logger.info(
            f"[load_seasonality_data] Loading data for pharmacy {pharmacy_id}, "
            f"dates: {start_date} to {end_date}, atc_filter: {atc_filter}"
        )

        # Parámetros comunes
        common_params = {
            "date_from": start_date,
            "date_to": end_date,
        }

        # Añadir categorías si se especifican
        if categories:
            common_params["categories"] = categories

        # Issue #532: Añadir filtro ATC si se especifica
        if atc_filter:
            common_params["atc_code"] = atc_filter

        # 1. Cargar heatmap
        heatmap_data = None
        try:
            heatmap_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/hourly-heatmap",
                method="GET",
                params=common_params,
                timeout=30,
                auth_headers=auth_headers,
            )
            if heatmap_response and "error" not in heatmap_response:
                heatmap_data = heatmap_response
                logger.info(
                    f"[load_seasonality_data] Heatmap loaded: "
                    f"{len(heatmap_data.get('heatmap_data', []))} cells"
                )
        except Exception as e:
            logger.error(f"[load_seasonality_data] Heatmap error: {e}")

        # 2. Cargar índice mensual
        monthly_index_data = None
        try:
            monthly_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/monthly-index",
                method="GET",
                params=common_params,
                timeout=30,
                auth_headers=auth_headers,
            )
            if monthly_response and "error" not in monthly_response:
                monthly_index_data = monthly_response
                logger.info(
                    f"[load_seasonality_data] Monthly index loaded: "
                    f"{len(monthly_index_data.get('monthly_index', []))} months"
                )
        except Exception as e:
            logger.error(f"[load_seasonality_data] Monthly index error: {e}")

        # 3. Cargar KPIs
        kpis_data = None
        try:
            kpis_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/kpis",
                method="GET",
                params=common_params,
                timeout=30,
                auth_headers=auth_headers,
            )
            if kpis_response and "error" not in kpis_response:
                kpis_data = kpis_response
                logger.info(
                    f"[load_seasonality_data] KPIs loaded: trend_yoy={kpis_data.get('trend_yoy')}"
                )
        except Exception as e:
            logger.error(f"[load_seasonality_data] KPIs error: {e}")

        # 4. Cargar descomposición STL (Issue #498)
        # IMPORTANTE: STL requiere 12+ meses. NO usar filtro de fechas del usuario.
        # Enviar solo date_to para usar todo el historial disponible.
        decomposition_data = None
        try:
            stl_params = {"date_to": end_date}  # Sin date_from = todo el historial
            # Issue #532: Añadir filtro ATC
            if atc_filter:
                stl_params["atc_code"] = atc_filter
            decomposition_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/trend-decomposition",
                method="GET",
                params=stl_params,
                timeout=60,  # STL puede tardar más
                auth_headers=auth_headers,
            )
            if decomposition_response and "error" not in decomposition_response:
                decomposition_data = decomposition_response
                logger.info(
                    f"[load_seasonality_data] STL decomposition loaded: "
                    f"{len(decomposition_data.get('decomposition', []))} points"
                )
        except Exception as e:
            logger.error(f"[load_seasonality_data] STL decomposition error: {e}")

        # 5. Cargar forecast Holt-Winters (Issue #498)
        # IMPORTANTE: Forecast requiere 12+ meses. NO usar filtro de fechas del usuario.
        # Enviar solo date_to para usar todo el historial disponible.
        forecast_data = None
        try:
            forecast_params = {
                "date_to": end_date,  # Sin date_from = todo el historial
                "periods_ahead": 3,
                "confidence": 0.95,
            }
            # Issue #532: Añadir filtro ATC
            if atc_filter:
                forecast_params["atc_code"] = atc_filter

            forecast_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/forecast",
                method="GET",
                params=forecast_params,
                timeout=60,  # Forecast puede tardar más
                auth_headers=auth_headers,
            )
            if forecast_response and "error" not in forecast_response:
                forecast_data = forecast_response
                logger.info(
                    f"[load_seasonality_data] Forecast loaded: "
                    f"{len(forecast_data.get('forecast', []))} periods"
                )
        except Exception as e:
            logger.error(f"[load_seasonality_data] Forecast error: {e}")

        # Issue #532: Category patterns removed - replaced by ATC filter

        return (
            heatmap_data,
            monthly_index_data,
            kpis_data,
            decomposition_data,
            forecast_data,
        )

    @app.callback(
        [
            Output("seasonality-hourly-heatmap", "figure"),
            Output("heatmap-peak-info", "children"),
        ],
        Input("seasonality-heatmap-store", "data"),
        prevent_initial_call=True,
    )
    def update_hourly_heatmap(heatmap_data: Optional[Dict]) -> tuple:
        """
        Actualiza el gráfico de heatmap cuando hay nuevos datos.
        """
        if not heatmap_data:
            fig = create_hourly_heatmap_chart([], None)
            return fig, "Sin datos disponibles"

        heatmap_cells = heatmap_data.get("heatmap_data", [])
        summary = heatmap_data.get("summary", {})

        fig = create_hourly_heatmap_chart(heatmap_cells, summary)
        peak_info = create_peak_info_text(summary)

        return fig, peak_info

    @app.callback(
        [
            Output("seasonality-monthly-index-chart", "figure"),
            Output("monthly-index-data-quality-alert", "children"),
            Output("monthly-index-partial-note", "children"),  # Issue #529
        ],
        Input("seasonality-monthly-index-store", "data"),
        prevent_initial_call=True,
    )
    def update_monthly_index(monthly_data: Optional[Dict]) -> tuple:
        """
        Actualiza el gráfico de índice mensual cuando hay nuevos datos.

        Issue #529: Añade nota informativa sobre meses parciales.
        """
        if not monthly_data:
            fig = create_monthly_index_chart([], None)
            return fig, None, None

        monthly_index = monthly_data.get("monthly_index", [])
        summary = monthly_data.get("summary", {})
        data_quality = monthly_data.get("data_quality", {})

        fig = create_monthly_index_chart(monthly_index, summary)
        quality_alert = create_data_quality_alert(data_quality)
        # Issue #529: Nota sobre meses parciales
        partial_note = create_partial_month_note(monthly_index)

        return fig, quality_alert, partial_note

    @app.callback(
        [
            Output("seasonality-trend-yoy-value", "children"),
            Output("seasonality-trend-direction-icon", "className"),
            Output("seasonality-next-peak-value", "children"),
            Output("seasonality-anomalies-value", "children"),
        ],
        Input("seasonality-kpis-store", "data"),
        prevent_initial_call=True,
    )
    def update_seasonality_kpis(kpis_data: Optional[Dict]) -> tuple:
        """
        Actualiza los KPIs del header de Tab 3.

        Issue #530: Si datos insuficientes (<18 meses), muestra MoM o mensaje.
        """
        from dash import html

        if not kpis_data:
            return "--", "mdi mdi-minus", "--", "0"

        # Issue #530: Verificar datos insuficientes
        insufficient_data = kpis_data.get("insufficient_data", False)
        data_span_months = kpis_data.get("data_span_months", 0)
        trend_direction = kpis_data.get("trend_direction", "stable")

        if insufficient_data:
            # Verificar si hay alternativa MoM disponible
            trend_mom = kpis_data.get("trend_mom")

            if trend_mom is not None:
                # Mostrar MoM como alternativa
                if trend_mom > 0:
                    trend_text = html.Span([
                        f"+{trend_mom:.1f}%",
                        html.Small(" (MoM)", className="text-muted ms-1")
                    ])
                else:
                    trend_text = html.Span([
                        f"{trend_mom:.1f}%",
                        html.Small(" (MoM)", className="text-muted ms-1")
                    ])
            else:
                # Mostrar mensaje informativo
                trend_text = html.Span([
                    html.Small(
                        f"Requiere +{18 - data_span_months} meses",
                        className="text-muted",
                        title=f"Se necesitan 18 meses para calcular YoY. Datos actuales: {data_span_months} meses."
                    )
                ])
        else:
            # Tendencia YoY normal
            trend_yoy = kpis_data.get("trend_yoy", 0) or 0

            if trend_yoy > 0:
                trend_text = f"+{trend_yoy:.1f}%"
            else:
                trend_text = f"{trend_yoy:.1f}%"

        # Icono de dirección
        if trend_direction == "up":
            icon_class = "mdi mdi-trending-up text-success"
        elif trend_direction == "down":
            icon_class = "mdi mdi-trending-down text-danger"
        else:
            icon_class = "mdi mdi-minus text-muted"

        # Próximo pico
        next_peak_category = kpis_data.get("next_peak_category", "")
        next_peak_months = kpis_data.get("next_peak_months", "")
        days_until = kpis_data.get("days_until_peak", 0)

        if next_peak_category:
            peak_text = f"{next_peak_category} ({next_peak_months}) - {days_until} días"
        else:
            peak_text = "--"

        # Anomalías
        anomalies = kpis_data.get("anomalies_count", 0)
        anomalies_text = str(anomalies)

        return trend_text, icon_class, peak_text, anomalies_text

    # =========================================================================
    # Issue #498: STL Decomposition + Forecast Callbacks
    # =========================================================================

    @app.callback(
        [
            Output("seasonality-decomposition-chart", "figure"),
            Output("stl-summary-container", "children"),
            Output("stl-data-quality-alert", "children"),
        ],
        Input("seasonality-decomposition-store", "data"),
        prevent_initial_call=True,
    )
    def update_decomposition_chart(decomposition_data: Optional[Dict]) -> tuple:
        """
        Actualiza el gráfico de descomposición STL cuando hay nuevos datos.

        Issue #498: STL Decomposition visualization.
        """
        if not decomposition_data:
            fig = create_decomposition_chart([], None)
            return fig, None, None

        decomposition = decomposition_data.get("decomposition", [])
        summary = decomposition_data.get("summary", {})
        data_quality = decomposition_data.get("data_quality", {})

        fig = create_decomposition_chart(decomposition, summary)
        summary_card = create_decomposition_summary_card(summary) if decomposition else None
        quality_alert = create_data_quality_alert_stl(data_quality)

        return fig, summary_card, quality_alert

    @app.callback(
        [
            Output("seasonality-forecast-chart", "figure"),
            Output("forecast-summary-container", "children"),
        ],
        Input("seasonality-forecast-store", "data"),
        prevent_initial_call=True,
    )
    def update_forecast_chart(forecast_data: Optional[Dict]) -> tuple:
        """
        Actualiza el gráfico de forecast cuando hay nuevos datos.

        Issue #498: Holt-Winters Forecast visualization.
        """
        if not forecast_data:
            fig = create_forecast_chart([], None, None)
            return fig, None

        forecast = forecast_data.get("forecast", [])
        historical = forecast_data.get("historical", [])
        summary = forecast_data.get("summary", {})
        model_info = forecast_data.get("model_info", {})
        confidence_level = forecast_data.get("confidence_level", 0.95)

        fig = create_forecast_chart(forecast, historical, summary, confidence_level)
        summary_card = create_forecast_summary_card(summary, model_info) if forecast else None

        return fig, summary_card

    # =========================================================================
    # Issue #499: Category Patterns Callbacks - REMOVED (Issue #532)
    # Replaced by ATC filter in the seasonality tab header
    # =========================================================================

    # =========================================================================
    # Issue #500: Stock-out Risk Matrix Callbacks
    # =========================================================================

    @app.callback(
        Output("seasonality-stockout-risk-store", "data"),
        [
            Input("url", "pathname"),
            Input("prescription-tabs", "active_tab"),
            Input("stockout-days-restock", "value"),
            Input("auth-ready", "data"),
            Input("seasonality-atc-filter", "value"),  # Issue #532: ATC filter
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_stockout_risk_data(
        pathname: str,
        active_tab: str,
        days_restock: int,
        auth_ready: Optional[bool],
        atc_filter: Optional[str],  # Issue #532: ATC filter
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> Optional[Dict]:
        """
        Carga datos de riesgo de stock cuando Tab 3 está activo.

        Issue #500: Stock-out Risk Matrix.
        Issue #532: ATC filter support.
        REGLA #7.6: Restaurar tokens multi-worker.
        """
        logger.debug(f"[load_stockout_risk] Triggered - active_tab: {active_tab}")

        # Guard 0: Solo ejecutar en /prescription
        if pathname != "/prescription":
            raise PreventUpdate

        # Guard 1: Solo ejecutar cuando Tab 3 (seasonality) está activo
        if active_tab != "tab-tendencias":
            logger.debug("[load_stockout_risk] Tab 3 not active - skipping")
            raise PreventUpdate

        # Guard 2: Esperar auth-ready
        if not auth_ready:
            raise PreventUpdate

        # Guard 3: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[load_stockout_risk] User not authenticated - skipping")
            raise PreventUpdate

        # Obtener headers de autenticación (REGLA #7.6)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[load_stockout_risk] No auth headers - skipping")
            raise PreventUpdate

        # Obtener pharmacy_id
        pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_stockout_risk] No pharmacy_id - skipping")
            raise PreventUpdate

        # Días de reposición
        days = days_restock or 7

        logger.info(
            f"[load_stockout_risk] Loading data for pharmacy {pharmacy_id}, "
            f"days_restock={days}, atc_filter={atc_filter}"
        )

        # Build params with optional ATC filter (Issue #532)
        stockout_params = {"days_until_restock": days, "top_n": 50}
        if atc_filter:
            stockout_params["atc_code"] = atc_filter

        try:
            stockout_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/stockout-risk",
                method="GET",
                params=stockout_params,
                timeout=60,
                auth_headers=auth_headers,
            )
            if stockout_response and "error" not in stockout_response:
                logger.info(
                    f"[load_stockout_risk] Loaded: "
                    f"{len(stockout_response.get('risk_items', []))} items"
                )
                return stockout_response
        except Exception as e:
            logger.error(f"[load_stockout_risk] Error: {e}")

        return None

    @app.callback(
        [
            Output("stockout-risk-summary-badges", "children"),
            Output("stockout-risk-table-container", "children"),
        ],
        Input("seasonality-stockout-risk-store", "data"),
        prevent_initial_call=True,
    )
    def update_stockout_risk_display(
        stockout_data: Optional[Dict],
    ) -> tuple:
        """
        Actualiza badges y tabla de riesgo de stock.

        Issue #500: Stock-out Risk Matrix display.
        """
        if not stockout_data:
            return (
                "Sin datos de inventario",
                create_empty_inventory_message(),
            )

        risk_items = stockout_data.get("risk_items", [])
        summary = stockout_data.get("summary", {})

        # Crear badges de resumen
        badges = create_stockout_risk_summary_badges(summary)

        # Crear tabla
        if risk_items:
            table = create_stockout_risk_table(risk_items)
        else:
            table = create_empty_inventory_message()

        return badges, table

    # =========================================================================
    # Issue #501: Anomaly Detection Callbacks
    # =========================================================================

    @app.callback(
        Output("seasonality-anomalies-store", "data"),
        [
            Input("url", "pathname"),
            Input("prescription-tabs", "active_tab"),
            Input("auth-ready", "data"),
            Input("seasonality-atc-filter", "value"),  # Issue #532: ATC filter
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_anomalies_data(
        pathname: str,
        active_tab: str,
        auth_ready: Optional[bool],
        atc_filter: Optional[str],  # Issue #532: ATC filter
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> Optional[Dict]:
        """
        Carga datos de anomalías cuando Tab 3 está activo.

        Issue #501: Detector de Anomalías + Alertas.
        Issue #532: ATC filter support.
        REGLA #7.6: Restaurar tokens multi-worker.
        """
        logger.debug(f"[load_anomalies] Triggered - active_tab: {active_tab}")

        # Guard 0: Solo ejecutar en /prescription
        if pathname != "/prescription":
            raise PreventUpdate

        # Guard 1: Solo ejecutar cuando Tab 3 (seasonality) está activo
        if active_tab != "tab-tendencias":
            logger.debug("[load_anomalies] Tab 3 not active - skipping")
            raise PreventUpdate

        # Guard 2: Esperar auth-ready
        if not auth_ready:
            raise PreventUpdate

        # Guard 3: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[load_anomalies] User not authenticated - skipping")
            raise PreventUpdate

        # Obtener headers de autenticación (REGLA #7.6)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[load_anomalies] No auth headers - skipping")
            raise PreventUpdate

        # Obtener pharmacy_id
        pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_anomalies] No pharmacy_id - skipping")
            raise PreventUpdate

        logger.info(f"[load_anomalies] Loading data for pharmacy {pharmacy_id}, atc_filter={atc_filter}")

        # Build params with optional ATC filter (Issue #532)
        anomalies_params = {
            "sensitivity": 2.0,
            "exclude_holidays": True,
            "exclude_bulk_sales": True,
        }
        if atc_filter:
            anomalies_params["atc_code"] = atc_filter

        try:
            anomalies_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/anomalies",
                method="GET",
                params=anomalies_params,
                timeout=60,
                auth_headers=auth_headers,
            )
            if anomalies_response and "error" not in anomalies_response:
                logger.info(
                    f"[load_anomalies] Loaded: "
                    f"{len(anomalies_response.get('anomalies', []))} anomalies"
                )
                return anomalies_response
        except Exception as e:
            logger.error(f"[load_anomalies] Error: {e}")

        return None

    @app.callback(
        [
            Output("anomaly-summary-badges", "children"),
            Output("anomaly-alerts-table-container", "children"),
        ],
        Input("seasonality-anomalies-store", "data"),
        prevent_initial_call=True,
    )
    def update_anomalies_display(
        anomalies_data: Optional[Dict],
    ) -> tuple:
        """
        Actualiza badges y tabla de anomalías.

        Issue #501: Anomaly Detection display.
        """
        if not anomalies_data:
            return (
                "Sin datos para análisis",
                create_no_data_message(),
            )

        anomalies = anomalies_data.get("anomalies", [])
        summary = anomalies_data.get("summary", {})
        data_quality = anomalies_data.get("data_quality", {})

        # Verificar si hay datos suficientes
        if data_quality.get("insufficient_data"):
            return (
                f"Se requieren más datos ({data_quality.get('days_analyzed', 0)} días disponibles)",
                create_no_data_message(),
            )

        # Merge summary con data_quality para badges
        merged_summary = {**summary, **data_quality}

        # Crear badges de resumen
        badges = create_anomaly_summary_badges(merged_summary)

        # Crear tabla
        table = create_anomaly_alerts_table(anomalies)

        return badges, table

    # =========================================================================
    # Issue #502: Export CSV + Badges Callbacks
    # =========================================================================

    @app.callback(
        Output("seasonality-badges-store", "data"),
        [
            Input("url", "pathname"),
            Input("prescription-tabs", "active_tab"),
            Input("auth-ready", "data"),
            Input("seasonality-atc-filter", "value"),  # Issue #532: ATC filter
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_badges_data(
        pathname: str,
        active_tab: str,
        auth_ready: Optional[bool],
        atc_filter: Optional[str],  # Issue #532: ATC filter
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> Optional[Dict]:
        """
        Carga badges para el header del Tab 3.

        Issue #502: Badges visuales (anomalías, próximo pico, stockout).
        Issue #532: ATC filter support.
        REGLA #7.6: Restaurar tokens multi-worker.
        """
        logger.debug(f"[load_badges] Triggered - active_tab: {active_tab}")

        # Guard 0: Solo ejecutar en /prescription
        if pathname != "/prescription":
            raise PreventUpdate

        # Guard 1: Solo ejecutar cuando Tab 3 (seasonality) está activo
        if active_tab != "tab-tendencias":
            logger.debug("[load_badges] Tab 3 not active - skipping")
            raise PreventUpdate

        # Guard 2: Esperar auth-ready
        if not auth_ready:
            raise PreventUpdate

        # Guard 3: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.debug("[load_badges] User not authenticated - skipping")
            raise PreventUpdate

        # Obtener headers de autenticación (REGLA #7.6)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[load_badges] No auth headers - skipping")
            raise PreventUpdate

        # Obtener pharmacy_id
        pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_badges] No pharmacy_id - skipping")
            raise PreventUpdate

        logger.info(f"[load_badges] Loading badges for pharmacy {pharmacy_id}, atc_filter={atc_filter}")

        # Build params with optional ATC filter (Issue #532)
        badges_params = {}
        if atc_filter:
            badges_params["atc_code"] = atc_filter

        try:
            badges_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/badges",
                method="GET",
                params=badges_params if badges_params else None,
                timeout=30,
                auth_headers=auth_headers,
            )
            if badges_response and "error" not in badges_response:
                logger.info(
                    f"[load_badges] Loaded: anomalies={badges_response.get('anomalies_count')}"
                )
                return badges_response
        except Exception as e:
            logger.error(f"[load_badges] Error: {e}")

        return None

    @app.callback(
        Output("seasonality-export-store", "data"),
        Input("seasonality-export-btn", "n_clicks"),
        [
            State("prescription-date-range", "start_date"),
            State("prescription-date-range", "end_date"),
            State("seasonality-atc-filter", "value"),  # Issue #532: ATC filter
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_export_data(
        n_clicks: Optional[int],
        start_date: str,
        end_date: str,
        atc_filter: Optional[str],  # Issue #532: ATC filter
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> Optional[Dict]:
        """
        Carga datos de export al hacer clic en el botón.

        Issue #502: Export CSV.
        Issue #532: ATC filter support.
        REGLA #7.6: Restaurar tokens multi-worker.
        """
        if not n_clicks:
            raise PreventUpdate

        logger.debug("[load_export] Export button clicked")

        # Guard: Verificar autenticación
        if not is_user_authenticated(auth_state):
            logger.warning("[load_export] User not authenticated - skipping")
            raise PreventUpdate

        # Guard: Verificar fechas
        if not start_date or not end_date:
            logger.warning("[load_export] No date range - skipping")
            raise PreventUpdate

        # Obtener headers de autenticación (REGLA #7.6)
        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[load_export] No auth headers - skipping")
            raise PreventUpdate

        # Obtener pharmacy_id
        pharmacy_id = auth_state.get("user", {}).get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_export] No pharmacy_id - skipping")
            raise PreventUpdate

        logger.info(f"[load_export] Loading export for pharmacy {pharmacy_id}, atc_filter={atc_filter}")

        # Build params with optional ATC filter (Issue #532)
        export_params = {
            "date_from": start_date,
            "date_to": end_date,
            "periods_ahead": 3,
            "top_categories": 10,
        }
        if atc_filter:
            export_params["atc_code"] = atc_filter

        try:
            export_response = request_coordinator.make_request(
                endpoint=f"/api/v1/prescription/{pharmacy_id}/seasonality/export",
                method="GET",
                params=export_params,
                timeout=60,
                auth_headers=auth_headers,
            )
            if export_response and "error" not in export_response:
                logger.info(
                    f"[load_export] Loaded: {len(export_response.get('data', []))} rows"
                )
                return export_response
        except Exception as e:
            logger.error(f"[load_export] Error: {e}")

        return None

    @app.callback(
        [
            Output("tab-seasonality-anomaly-badge", "children"),
            Output("tab-seasonality-peak-badge", "children"),
        ],
        Input("seasonality-badges-store", "data"),
        prevent_initial_call=True,
    )
    def update_tab_badges(badges_data: Optional[Dict]) -> tuple:
        """
        Actualiza badges en el tab header.

        Issue #502: Tab Badges.
        """
        from dash import html

        if not badges_data:
            return None, None

        # Badge de anomalías
        anomalies_count = badges_data.get("anomalies_count", 0)
        anomalies_color = badges_data.get("anomalies_badge_color", "secondary")

        if anomalies_count > 0:
            anomaly_badge = dbc.Badge(
                f"{anomalies_count} alertas",
                color=anomalies_color,
                pill=True,
                className="small",
            )
        else:
            anomaly_badge = None

        # Badge de próximo pico
        next_peak_label = badges_data.get("next_peak_label")
        next_peak_color = badges_data.get("next_peak_badge_color", "secondary")

        if next_peak_label:
            peak_badge = dbc.Badge(
                next_peak_label,
                color=next_peak_color,
                pill=True,
                className="small",
            )
        else:
            peak_badge = None

        return anomaly_badge, peak_badge

    @app.callback(
        Output("seasonality-csv-download", "data"),
        Input("seasonality-export-store", "data"),
        prevent_initial_call=True,
    )
    def download_csv(export_data: Optional[Dict]) -> Optional[Dict]:
        """
        Genera descarga CSV cuando hay datos de export.

        Issue #502: Download CSV.
        """
        if not export_data:
            raise PreventUpdate

        # Validar que no hay error en la respuesta
        if "error" in export_data:
            logger.warning(f"[download_csv] Export data has error: {export_data.get('error')}")
            raise PreventUpdate

        data = export_data.get("data", [])
        if not data:
            logger.debug("[download_csv] Export data is empty")
            raise PreventUpdate

        # Crear CSV content
        import csv
        from io import StringIO

        output = StringIO()
        writer = csv.DictWriter(
            output,
            fieldnames=[
                "category",
                "month",
                "month_num",
                "forecast",
                "lower_bound",
                "upper_bound",
                "seasonality_index",
            ],
        )
        writer.writeheader()
        writer.writerows(data)

        csv_content = output.getvalue()
        output.close()

        # Nombre del archivo
        export_info = export_data.get("export_info", {})
        date_from = export_info.get("date_from", "").replace("-", "")
        date_to = export_info.get("date_to", "").replace("-", "")
        filename = f"forecast_estacionalidad_{date_from}_{date_to}.csv"

        logger.info(f"[download_csv] Generating CSV: {filename}")

        return dict(
            content=csv_content,
            filename=filename,
            type="text/csv",
        )

    # =========================================================================
    # Issue #526: Period Display Callback
    # =========================================================================

    @app.callback(
        Output("seasonality-period-display", "children"),
        [
            Input("prescription-date-range", "start_date"),
            Input("prescription-date-range", "end_date"),
            Input("prescription-tabs", "active_tab"),
        ],
        prevent_initial_call=True,
    )
    def update_period_display(
        start_date: Optional[str],
        end_date: Optional[str],
        active_tab: str,
    ) -> list:
        """
        Actualiza el indicador de período en Tab 3.

        Issue #526: Mostrar claramente el período analizado.

        Args:
            start_date: Fecha inicio del rango seleccionado
            end_date: Fecha fin del rango seleccionado
            active_tab: Tab activo (solo actualizar si es tab-tendencias)

        Returns:
            list: Componentes con badges mostrando el período
        """
        # Guard: Solo actualizar cuando Tab 3 está activo
        if active_tab != "tab-tendencias":
            raise PreventUpdate

        # Guard: Verificar fechas
        if not start_date or not end_date:
            return [
                dbc.Badge(
                    "Selecciona un período en el filtro de fechas",
                    color="warning",
                    className="me-2",
                ),
            ]

        # Formatear fechas a español
        try:
            start_dt = datetime.fromisoformat(start_date.split("T")[0])
            end_dt = datetime.fromisoformat(end_date.split("T")[0])

            # Usar constantes centralizadas (DRY)
            start_formatted = f"{start_dt.day} {MESES_ESPANOL_CORTO.get(start_dt.month, '')} {start_dt.year}"
            end_formatted = f"{end_dt.day} {MESES_ESPANOL_CORTO.get(end_dt.month, '')} {end_dt.year}"

            # Calcular duración con precisión mejorada
            duration_days = (end_dt - start_dt).days
            if duration_days >= 365:
                years = duration_days / 365.25  # Cuenta años bisiestos
                years_int = int(years)
                if years == years_int:
                    duration_text = f"{years_int} {'año' if years_int == 1 else 'años'}"
                else:
                    duration_text = f"{years:.1f} años"
            elif duration_days >= 30:
                months = round(duration_days / 30.44)  # Longitud media del mes
                duration_text = f"{months} {'mes' if months == 1 else 'meses'}"
            else:
                duration_text = f"{duration_days} {'día' if duration_days == 1 else 'días'}"

            return [
                dbc.Badge(
                    [
                        html.I(className="fas fa-calendar-day me-1"),
                        f"{start_formatted}",
                    ],
                    color="info",
                    className="me-2",
                ),
                html.Span("→", className="text-muted mx-1"),
                dbc.Badge(
                    [
                        html.I(className="fas fa-calendar-check me-1"),
                        f"{end_formatted}",
                    ],
                    color="info",
                    className="me-2",
                ),
                dbc.Badge(
                    f"📊 {duration_text}",
                    color="light",
                    text_color="dark",
                    className="ms-2",
                ),
            ]

        except (ValueError, AttributeError) as e:
            logger.warning(f"[update_period_display] Error parsing dates: {e}")
            return [
                dbc.Badge(
                    f"{start_date} → {end_date}",
                    color="secondary",
                ),
            ]

    logger.info("[prescription/seasonality] 15 callbacks registered (Issue #489 + #498 + #500 + #501 + #502 + #526 + #532)")
