"""
Products callbacks for prescription dashboard.

Handles category accordion, ingredient products, and filter options.
"""

import logging
from typing import Optional, Dict, Any, Tuple

from dash import Input, Output, State, html, MATCH
from dash.exceptions import PreventUpdate
import dash_bootstrap_components as dbc

from utils.auth_helpers import is_user_authenticated, get_auth_headers_from_tokens
from utils.request_coordinator import request_coordinator
from utils.helpers import format_currency, format_percentage, format_number
from components.empty_state import create_analysis_empty_state, create_error_state
from dash_iconify import DashIconify
from styles.design_tokens import COLORS
from utils.constants import format_category_name, CATEGORY_NAME_TO_KEY

logger = logging.getLogger(__name__)

# Issue #510: Constantes de truncamiento para nombres largos (consistencia con contributors.py)
PRODUCT_DISPLAY_LENGTH = 28  # Productos en drill-down


def register_products_callbacks(app):
    """
    Registra callbacks de productos y filtros.

    Callbacks:
    1. update_products_accordion: Acordeón de categorías
    2. load_category_products: Carga lazy de productos por categoría
    3. render_category_products: Renderiza productos de categoría
    4. load_ingredient_products: Carga lazy de productos por principio activo
    5. render_ingredient_products: Renderiza productos de principio activo
    6. load_prescription_employee_options: Opciones de empleados (PRO)
    7. load_prescription_laboratory_options: Opciones de laboratorios
    """

    @app.callback(
        Output("prescription-products-table-container", "children"),
        [
            Input("prescription-products-trigger-store", "data"),
            Input("prescription-context-treemap", "clickData"),
        ],
        [
            State("prescription-overview-store", "data"),
            State("prescription-categories-filter-store", "data"),  # Issue #510: Cambio a State (REGLA #11 - evita Input duplicado con data_loading.py)
        ],
        prevent_initial_call=True,
    )
    def update_products_accordion(
        trigger_data: Optional[Dict],
        treemap_click: Optional[Dict],
        overview_data: Optional[Dict],
        selected_categories: Optional[list],  # Filtro de categorías (desde Store como State)
    ) -> Any:
        """
        Actualiza acordeón de categorías con drill-down a productos.

        Issue #441: Renderiza un acordeón donde cada categoría es un item expandible.
        """
        logger.debug("[update_products_accordion] Callback triggered")

        if not overview_data:
            logger.warning("[update_products_accordion] No overview data available")
            return create_analysis_empty_state(
                message="Aplica filtros para ver las categorías de prescripción",
                icon="mdi:view-list",
                suggestions=[
                    "Selecciona un período de análisis",
                    "Usa el botón 'Aplicar Filtros' para cargar datos",
                ],
                min_height="400px",
                section_name="Categorías por Ventas",
            )

        # Extraer datos de categorías del overview
        category_summary = overview_data.get("category_summary", [])

        # Filtrar por categoría si hay click en treemap
        selected_category = None
        if treemap_click and "points" in treemap_click:
            try:
                label = treemap_click["points"][0].get("label", "")
                selected_category = CATEGORY_NAME_TO_KEY.get(label)
                if selected_category:
                    logger.debug(f"[update_products_accordion] Filtering by category: {selected_category}")
            except (KeyError, IndexError):
                pass

        if not category_summary:
            return create_analysis_empty_state(
                message="No hay datos de categorías para el período seleccionado",
                icon="mdi:package-variant-closed",
                suggestions=[
                    "Verifica que existan ventas de prescripción",
                    "Amplía el rango de fechas",
                ],
                min_height="400px",
                section_name="Categorías por Ventas",
            )

        try:
            # Ordenar por ventas descendente
            sorted_categories = sorted(
                category_summary,
                key=lambda x: float(x.get("total_sales", 0)),
                reverse=True,
            )

            # Filtrar por categoría si está seleccionada (treemap click)
            if selected_category:
                sorted_categories = [c for c in sorted_categories if c.get("category") == selected_category]

            # Filtrar por categorías seleccionadas en dropdown (si hay selección)
            if selected_categories and len(selected_categories) > 0:
                sorted_categories = [c for c in sorted_categories if c.get("category") in selected_categories]
                logger.debug(f"[update_products_accordion] Filtering by dropdown: {selected_categories}")

            # Issue #443: Deduplicate by category (normalized key)
            seen_categories = {}
            deduplicated_categories = []
            for cat in sorted_categories:
                raw_key = cat.get("category", "unknown")
                # Normalizar: lowercase, strip, reemplazar espacios por guiones bajos
                cat_key = str(raw_key).lower().strip().replace(" ", "_").replace("-", "_")
                if cat_key not in seen_categories:
                    seen_categories[cat_key] = len(deduplicated_categories)
                    # Guardar con key normalizada para evitar duplicados en IDs
                    cat_copy = cat.copy()
                    cat_copy["category"] = cat_key
                    deduplicated_categories.append(cat_copy)
                else:
                    idx = seen_categories[cat_key]
                    existing = deduplicated_categories[idx]
                    existing["total_sales"] = float(existing.get("total_sales", 0)) + float(cat.get("total_sales", 0))
                    existing["total_units"] = int(existing.get("total_units", 0)) + int(cat.get("total_units", 0))
                    existing["percentage"] = float(existing.get("percentage", 0)) + float(cat.get("percentage", 0))
                    logger.warning(f"[update_products_accordion] Aggregated duplicate category: {raw_key} -> {cat_key}")

            sorted_categories = deduplicated_categories
            logger.info(f"[update_products_accordion] {len(sorted_categories)} unique categories after dedup")

            if not sorted_categories:
                return create_analysis_empty_state(
                    message="No hay categorías que coincidan con los filtros",
                    icon="mdi:filter-off",
                    suggestions=["Prueba con filtros menos restrictivos"],
                    min_height="400px",
                    section_name="Categorías por Ventas",
                )

            # Crear items del acordeón - deduplicar categorías
            # Issue #449: Evitar IDs duplicados usando set para tracking
            seen_categories = set()
            accordion_items = []
            for idx, cat in enumerate(sorted_categories[:15]):
                category_key = cat.get("category", "unknown")

                # Saltar categorías duplicadas
                if category_key in seen_categories:
                    logger.warning(f"[update_products_accordion] Skipping duplicate category: {category_key}")
                    continue
                seen_categories.add(category_key)

                category_name = format_category_name(category_key)
                sales = float(cat.get("total_sales", 0))
                percentage = float(cat.get("percentage", 0))

                header_content = dbc.Row(
                    [
                        dbc.Col(
                            [
                                DashIconify(icon="mdi:pill", width=18, color=COLORS["primary"], className="me-2"),
                                html.Span(category_name, style={"fontWeight": "600"}),
                            ],
                            width=6,
                        ),
                        dbc.Col(
                            html.Span(
                                format_currency(sales),
                                style={"fontWeight": "700", "color": COLORS["success"]},
                            ),
                            width=3,
                            className="text-end",
                        ),
                        dbc.Col(
                            html.Span(format_percentage(percentage / 100), className="text-muted"),
                            width=3,
                            className="text-end",
                        ),
                    ],
                    className="w-100 align-items-center",
                )

                body_content = html.Div(
                    [
                        html.Div(
                            id={"type": "category-products-container", "index": category_key},
                            children=[
                                html.Div(
                                    [
                                        dbc.Spinner(size="sm", color="primary", spinner_class_name="me-2"),
                                        html.Small("Cargando productos...", className="text-muted"),
                                    ],
                                    className="d-flex align-items-center justify-content-center py-3",
                                )
                            ],
                        ),
                    ],
                    style={"minHeight": "100px"},
                )

                accordion_items.append(
                    dbc.AccordionItem(
                        body_content,
                        title=header_content,
                        item_id=category_key,
                    )
                )

            # Total row fuera del acordeón
            total_sales = sum(float(c.get("total_sales", 0)) for c in sorted_categories[:15])
            total_row = dbc.Row(
                [
                    dbc.Col(html.Span("Total (Top 15)", style={"fontWeight": "700"}), width=6),
                    dbc.Col(
                        html.Span(
                            format_currency(total_sales),
                            style={"fontWeight": "700", "color": COLORS["primary"]},
                        ),
                        width=6,
                        className="text-end",
                    ),
                ],
                className="p-2 bg-light border-top mt-2",
            )

            return html.Div(
                [
                    dbc.Accordion(
                        accordion_items,
                        id="prescription-categories-accordion",
                        flush=True,
                        always_open=False,
                        start_collapsed=True,
                    ),
                    total_row,
                ]
            )

        except Exception as e:
            logger.error(f"[update_products_accordion] Error: {str(e)}", exc_info=True)
            return create_error_state(
                error_message=f"Error al renderizar acordeón: {str(e)}",
                min_height="400px",
            )

    @app.callback(
        [
            Output("prescription-category-products-store", "data"),
            Output("prescription-category-render-trigger", "data"),
        ],
        [Input("prescription-categories-accordion", "active_item")],
        [
            State("prescription-date-range", "start_date"),
            State("prescription-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("prescription-category-products-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_category_products(
        active_item: Optional[str],
        start_date: Optional[str],
        end_date: Optional[str],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
        existing_cache: Optional[Dict],
    ) -> Tuple[Dict, Optional[Dict]]:
        """
        Carga productos de una categoría cuando se expande en el acordeón.
        """
        logger.debug(f"[load_category_products] Accordion active_item: {active_item}")

        if not active_item:
            return existing_cache or {}, None

        cache = existing_cache or {}

        if active_item in cache and cache[active_item].get("loaded"):
            logger.debug(f"[load_category_products] Cache hit for {active_item}")
            trigger_data = {"category": active_item, "cached": True}
            return cache, trigger_data

        if not is_user_authenticated(auth_state):
            logger.debug("[load_category_products] User not authenticated")
            return cache, None

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        if not auth_headers or not start_date or not end_date:
            logger.warning("[load_category_products] Missing auth or dates")
            return cache, None

        user = auth_state.get("user") if auth_state else None
        if not user:
            logger.warning("[load_category_products] No user in auth_state")
            return cache, None

        pharmacy_id = user.get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_category_products] No pharmacy_id")
            return cache, None

        logger.info(f"[load_category_products] Fetching products for category: {active_item}")

        response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/products-by-category",
            method="GET",
            params={
                "date_from": start_date,
                "date_to": end_date,
                "category": active_item,
                "limit": 10,
            },
            timeout=30,
            auth_headers=auth_headers,
        )

        if response and "categories" in response:
            categories = response.get("categories", [])
            if categories and len(categories) > 0:
                products = categories[0].get("products", [])
                cache[active_item] = {
                    "products": products,
                    "loaded": True,
                    "category_name": format_category_name(active_item),
                }
                logger.info(f"[load_category_products] Loaded {len(products)} products for {active_item}")
            else:
                cache[active_item] = {"products": [], "loaded": True}
        else:
            error_msg = response.get("message", "Unknown") if response else "No response"
            logger.warning(f"[load_category_products] Failed: {error_msg}")
            cache[active_item] = {
                "products": [],
                "loaded": True,
                "error": error_msg,
            }

        trigger_data = {"category": active_item, "cached": False}
        return cache, trigger_data

    @app.callback(
        Output({"type": "category-products-container", "index": MATCH}, "children"),
        [Input("prescription-category-render-trigger", "data")],
        [
            State("prescription-category-products-store", "data"),
            State("prescription-categories-accordion", "active_item"),
            State({"type": "category-products-container", "index": MATCH}, "id"),
        ],
        prevent_initial_call=True,
    )
    def render_category_products(
        trigger_data: Optional[Dict],
        products_cache: Optional[Dict],
        active_item: Optional[str],
        container_id: Dict,
    ) -> Any:
        """
        Renderiza tabla de productos para la categoría expandida.
        """
        if not trigger_data:
            raise PreventUpdate

        category_key = container_id.get("index") if container_id else None
        logger.debug(f"[render_category_products] Category: {category_key}, Active: {active_item}")

        if category_key != active_item:
            return html.Div(
                html.Small("Expandir para ver productos", className="text-muted"),
                className="text-center py-3",
            )

        if not products_cache or category_key not in products_cache:
            return html.Div(
                [
                    dbc.Spinner(size="sm", color="primary", spinner_class_name="me-2"),
                    html.Small("Cargando productos...", className="text-muted"),
                ],
                className="d-flex align-items-center justify-content-center py-3",
            )

        category_data = products_cache.get(category_key, {})

        if category_data.get("error"):
            return html.Div(
                [
                    DashIconify(icon="mdi:alert-circle", width=20, color=COLORS["warning"], className="me-2"),
                    html.Small(f"Error: {category_data['error']}", className="text-muted"),
                ],
                className="text-center py-3",
            )

        products = category_data.get("products", [])

        if not products:
            return html.Div(
                [
                    DashIconify(
                        icon="mdi:package-variant-closed",
                        width=24,
                        color=COLORS["text_muted"],
                        className="me-2",
                    ),
                    html.Small("No hay productos en esta categoría", className="text-muted"),
                ],
                className="d-flex align-items-center justify-content-center py-3",
            )

        try:
            table_header = html.Thead(
                html.Tr(
                    [
                        html.Th("Producto", style={"width": "50%"}),
                        html.Th("Ventas (€)", className="text-end", style={"width": "25%"}),
                        html.Th("Unidades", className="text-end", style={"width": "25%"}),
                    ]
                )
            )

            table_rows = []
            for prod in products[:10]:
                product_name = prod.get("product_name", prod.get("name", "N/A"))
                sales = float(prod.get("sales", prod.get("total_sales", 0)))
                units = int(prod.get("units", prod.get("total_units", 0)))

                row = html.Tr(
                    [
                        html.Td(
                            product_name[:40] + "..." if len(product_name) > 40 else product_name,
                            title=product_name,
                            style={"fontSize": "0.85rem"},
                        ),
                        html.Td(
                            format_currency(sales),
                            className="text-end",
                            style={"fontWeight": "600", "color": COLORS["success"], "fontSize": "0.85rem"},
                        ),
                        html.Td(
                            format_number(units),
                            className="text-end text-muted",
                            style={"fontSize": "0.85rem"},
                        ),
                    ]
                )
                table_rows.append(row)

            table_body = html.Tbody(table_rows)

            return dbc.Table(
                [table_header, table_body],
                bordered=False,
                hover=True,
                responsive=True,
                striped=True,
                size="sm",
                className="mb-0",
                style={"fontSize": "0.85rem"},
            )

        except Exception as e:
            logger.error(f"[render_category_products] Error: {str(e)}", exc_info=True)
            return html.Div(
                html.Small(f"Error: {str(e)}", className="text-danger"),
                className="text-center py-3",
            )

    @app.callback(
        [
            Output("prescription-ingredient-products-store", "data"),
            Output("prescription-ingredient-render-trigger", "data"),
        ],
        [Input("prescription-contributors-accordion", "active_item")],
        [
            State("prescription-date-range", "start_date"),
            State("prescription-date-range", "end_date"),
            State("waterfall-period-comparison-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("prescription-ingredient-products-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def load_ingredient_products(
        active_item: Optional[str],
        start_date: Optional[str],
        end_date: Optional[str],
        comparison_store: Optional[Dict],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
        existing_cache: Optional[Dict],
    ) -> Tuple[Dict, Optional[Dict]]:
        """
        Carga productos de un grupo de análisis cuando se expande en el acordeón.

        Issue #451: Soporta drill-down adaptativo por tipo de agrupación.
        """
        logger.debug(f"[load_ingredient_products] Accordion active_item: {active_item}")

        if not active_item:
            return existing_cache or {}, None

        cache = existing_cache or {}

        if active_item in cache and cache[active_item].get("loaded"):
            logger.debug(f"[load_ingredient_products] Cache hit for {active_item}")
            trigger_data = {"ingredient": active_item, "cached": True}
            return cache, trigger_data

        if not is_user_authenticated(auth_state):
            logger.debug("[load_ingredient_products] User not authenticated")
            return cache, None

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        if not auth_headers or not start_date or not end_date:
            logger.warning("[load_ingredient_products] Missing auth or dates")
            return cache, None

        user = auth_state.get("user") if auth_state else None
        if not user:
            logger.warning("[load_ingredient_products] No user in auth_state")
            return cache, None

        pharmacy_id = user.get("pharmacy_id")
        if not pharmacy_id:
            logger.warning("[load_ingredient_products] No pharmacy_id")
            return cache, None

        # Issue #451: Parsear grouping_type del item_id
        # Formato: "grouping_type::active_ingredient" o solo "active_ingredient" (retrocompatibilidad)
        if "::" in active_item:
            grouping_type, ingredient_name = active_item.split("::", 1)
        else:
            grouping_type = None
            ingredient_name = active_item

        logger.info(
            f"[load_ingredient_products] Fetching products for {grouping_type or 'active_ingredient'}: {ingredient_name}"
        )

        params = {
            "date_from": start_date,
            "date_to": end_date,
            "active_ingredient": ingredient_name,
            "limit": 10,
        }

        # Issue #451: Añadir grouping_type al request
        if grouping_type:
            params["grouping_type"] = grouping_type

        if comparison_store and isinstance(comparison_store, dict):
            comp_from = comparison_store.get("comparison_date_from")
            comp_to = comparison_store.get("comparison_date_to")
            if comp_from and comp_to:
                params["comparison_date_from"] = comp_from
                params["comparison_date_to"] = comp_to
                logger.debug(f"[load_ingredient_products] YoY dates: {comp_from} to {comp_to}")

        response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/products-by-active-ingredient",
            method="GET",
            params=params,
            timeout=30,
            auth_headers=auth_headers,
        )

        if response and "products" in response:
            products = response["products"]
            cache[active_item] = {
                "products": products,
                "loaded": True,
            }
            logger.info(f"[load_ingredient_products] Loaded {len(products)} products for {active_item}")
        else:
            error_msg = response.get("message", "Unknown") if response else "No response"
            logger.warning(f"[load_ingredient_products] Failed: {error_msg}")
            cache[active_item] = {
                "products": [],
                "loaded": True,
                "error": error_msg,
            }

        trigger_data = {"ingredient": active_item, "cached": False}
        return cache, trigger_data

    @app.callback(
        Output({"type": "ingredient-products-container", "index": MATCH}, "children"),
        [Input("prescription-ingredient-render-trigger", "data")],
        [
            State("prescription-ingredient-products-store", "data"),
            State("prescription-contributors-accordion", "active_item"),
            State({"type": "ingredient-products-container", "index": MATCH}, "id"),
        ],
        prevent_initial_call=True,
    )
    def render_ingredient_products(
        trigger_data: Optional[Dict],
        products_cache: Optional[Dict],
        active_item: Optional[str],
        container_id: Dict,
    ) -> Any:
        """
        Renderiza tabla de productos para el grupo de análisis expandido.

        Issue #451: Soporta nuevo formato de item_id con grouping_type.
        """
        if not trigger_data:
            raise PreventUpdate

        raw_index = container_id.get("index") if container_id else None
        # El index del container usa formato "idx::grouping_type::active_ingredient"
        # Fix: Use :: as delimiter to properly parse grouping_type with "_" (e.g., homogeneous_group)
        if raw_index and "::" in raw_index:
            parts = raw_index.split("::")  # ["idx", "grouping_type", "active_ingredient"]
            ingredient_key = parts[2] if len(parts) > 2 else parts[-1]
        elif raw_index and "_" in raw_index:
            # Retrocompatibilidad con formato antiguo "idx_grouping_type_active_ingredient"
            parts = raw_index.split("_", 2)
            ingredient_key = parts[2] if len(parts) > 2 else parts[-1]
        else:
            ingredient_key = raw_index
        logger.debug(f"[render_ingredient_products] Container index ingredient: {ingredient_key}, Active: {active_item}")

        # Issue #451: active_item ahora puede tener formato "grouping_type::active_ingredient"
        # Extraer solo el nombre del ingrediente para comparación
        if active_item and "::" in active_item:
            _, active_ingredient_name = active_item.split("::", 1)
        else:
            active_ingredient_name = active_item

        if ingredient_key != active_ingredient_name:
            return html.Div(
                html.Small("Expandir para ver productos", className="text-muted"),
                className="text-center py-3",
            )

        # Issue #451: El cache usa el active_item completo (con grouping_type)
        cache_key = active_item  # Usar el item_id completo para el cache
        if not products_cache or cache_key not in products_cache:
            return html.Div(
                [
                    dbc.Spinner(size="sm", color="primary", spinner_class_name="me-2"),
                    html.Small("Cargando productos...", className="text-muted"),
                ],
                className="d-flex align-items-center justify-content-center py-3",
            )

        ingredient_data = products_cache.get(cache_key, {})

        if ingredient_data.get("error"):
            return html.Div(
                [
                    DashIconify(icon="mdi:alert-circle", width=20, color=COLORS["warning"], className="me-2"),
                    html.Small(f"Error: {ingredient_data['error']}", className="text-muted"),
                ],
                className="text-center py-3",
            )

        products = ingredient_data.get("products", [])

        if not products:
            # Issue #451: Mensaje genérico (puede ser grupo homogéneo o principio activo)
            return html.Div(
                [
                    DashIconify(
                        icon="mdi:package-variant-closed",
                        width=24,
                        color=COLORS["text_muted"],
                        className="me-2",
                    ),
                    html.Small("No hay productos para este grupo de análisis", className="text-muted"),
                ],
                className="d-flex align-items-center justify-content-center py-3",
            )

        try:
            # Issue #453: Añadir columna de unidades al drill-down
            # Issue #510: Renombrar "% Contrib" a "% Responsable" para claridad
            table_header = html.Thead(
                html.Tr(
                    [
                        html.Th("Producto", style={"width": "32%"}),
                        html.Th("Ventas", className="text-end", style={"width": "16%"}),
                        html.Th("Uds", className="text-end", style={"width": "12%"}),
                        html.Th("Var €", className="text-end", style={"width": "18%"}),
                        html.Th(
                            html.Span(
                                "% Responsable",
                                title="Positivo = contribuye al cambio | Negativo = contrarresta",
                            ),
                            className="text-end",
                            style={"width": "22%"},
                        ),
                    ]
                )
            )

            table_rows = []
            for prod in products[:10]:
                product_name = prod.get("product_name", "N/A")
                sales = float(prod.get("total_sales", 0))
                units = int(prod.get("total_units", 0))  # Issue #453: Unidades actuales
                prev_units = int(prod.get("prev_units", 0))  # Issue #453: Unidades anteriores
                units_variation = int(prod.get("variation_units", 0))  # Issue #453: Variación unidades
                variation = float(prod.get("variation_euros", 0))
                contribution = float(prod.get("contribution_percent", 0))

                # Color según dirección de variación
                var_color = (
                    COLORS["success"] if variation > 0 else COLORS["danger"] if variation < 0 else COLORS["text_muted"]
                )
                # Issue #510: Simplificado - format_currency preserva el signo natural
                var_sign = "+" if variation > 0 else ""

                # Issue #510: Indicador visual para % Responsable (consistencia con contributors.py)
                # Positivo = contribuye al cambio, Negativo = contrarresta, Cero = sin contribución
                if contribution > 0:
                    contrib_color = COLORS["success"]  # Contribuye al cambio (verde)
                    contrib_icon = "mdi:arrow-right-bold"
                    contrib_tooltip = "Contribuye al cambio total"
                elif contribution < 0:
                    contrib_color = COLORS["info"]  # Contrarresta (azul)
                    contrib_icon = "mdi:arrow-left-bold"
                    contrib_tooltip = "Contrarresta la tendencia"
                else:
                    contrib_color = COLORS["text_muted"]  # Sin contribución
                    contrib_icon = "mdi:minus"
                    contrib_tooltip = "Sin contribución neta"

                # Issue #453: Color y texto para variación de unidades
                units_var_color = (
                    COLORS["success"] if units_variation > 0 else COLORS["danger"] if units_variation < 0 else COLORS["text_muted"]
                )
                units_var_sign = "+" if units_variation > 0 else ""
                # Formato: "actual (variación)" ej: "136 (+10)" o "13 (-10)"
                units_display = html.Span([
                    html.Span(f"{units}", style={"fontWeight": "600"}),
                    html.Span(
                        f" ({units_var_sign}{units_variation})",
                        style={"color": units_var_color, "fontSize": "0.65rem"},
                    ) if units_variation != 0 else html.Span(""),
                ])

                # Issue #510: Usar constante de truncamiento
                truncated_name = product_name[:PRODUCT_DISPLAY_LENGTH] + "..." if len(product_name) > PRODUCT_DISPLAY_LENGTH else product_name

                row = html.Tr(
                    [
                        html.Td(
                            [
                                # Issue #510: Indicador visual de dirección del producto
                                DashIconify(
                                    icon="mdi:trending-up" if variation > 0 else "mdi:trending-down" if variation < 0 else "mdi:minus",
                                    width=14,
                                    color=var_color,
                                    className="me-1",
                                ),
                                html.Span(truncated_name, title=product_name),
                            ],
                            style={"fontSize": "0.75rem"},
                        ),
                        html.Td(
                            format_currency(sales),
                            className="text-end",
                            style={"fontWeight": "600", "fontSize": "0.75rem"},
                        ),
                        html.Td(
                            units_display,
                            className="text-end",
                            style={"fontSize": "0.75rem"},
                        ),
                        html.Td(
                            f"{var_sign}{format_currency(abs(variation))}",
                            className="text-end",
                            style={"fontWeight": "600", "color": var_color, "fontSize": "0.75rem"},
                        ),
                        html.Td(
                            html.Span(
                                [
                                    DashIconify(
                                        icon=contrib_icon,
                                        width=10,
                                        color=contrib_color,
                                        className="me-1",
                                    ),
                                    f"{abs(contribution):.1f}%",
                                ],
                                title=contrib_tooltip,
                            ),
                            className="text-end",
                            style={"fontWeight": "700", "color": contrib_color, "fontSize": "0.75rem"},
                        ),
                    ]
                )
                table_rows.append(row)

            table_body = html.Tbody(table_rows)

            return dbc.Table(
                [table_header, table_body],
                bordered=False,
                hover=True,
                responsive=True,
                striped=True,
                size="sm",
                className="mb-0",
                style={"fontSize": "0.8rem"},
            )

        except Exception as e:
            logger.error(f"[render_ingredient_products] Error: {str(e)}", exc_info=True)
            return html.Div(
                html.Small(f"Error: {str(e)}", className="text-danger"),
                className="text-center py-3",
            )

    @app.callback(
        [
            Output("prescription-employee-filter", "options"),
            Output("prescription-employee-filter", "disabled"),
            Output("prescription-employee-filter-badge", "style"),
        ],
        [Input("url", "pathname"), Input("auth-ready", "data")],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=False,
    )
    def load_prescription_employee_options(
        pathname: str,
        auth_ready: Optional[Dict],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> tuple:
        """
        Carga opciones de empleados para filtro de prescripción.
        Solo disponible para usuarios PRO+.
        """
        logger.debug(f"[load_prescription_employee_options] pathname: {pathname}, auth_ready: {bool(auth_ready)}")

        if pathname != "/prescription":
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        user = auth_state.get("user") if auth_state else None
        if not user:
            logger.warning("[load_prescription_employee_options] No user in auth_state")
            return [], True, {"display": "inline-block"}

        subscription_plan = user.get("subscription_plan", "").lower()
        pharmacy_id = user.get("pharmacy_id")

        if subscription_plan == "free":
            logger.debug(f"[load_prescription_employee_options] User plan '{subscription_plan}' - disabled")
            return [], True, {"display": "inline-block"}

        if not pharmacy_id:
            logger.warning("[load_prescription_employee_options] No pharmacy_id")
            return [], True, {"display": "inline-block"}

        logger.info(f"[load_prescription_employee_options] Fetching employees for pharmacy {pharmacy_id}")

        response = request_coordinator.make_request(
            endpoint=f"/api/v1/employees/{pharmacy_id}",
            method="GET",
            timeout=15,
            auth_headers=auth_headers,
        )

        if response and "employees" in response:
            employees = response["employees"]
            options = [{"label": "🚫 Sin empleado asignado", "value": "__sin_empleado__"}]
            options.extend(
                [
                    {
                        "label": f"{emp['name']} ({format_number(emp.get('sales_count', 0))} ventas)",
                        "value": emp["name"],
                    }
                    for emp in employees
                ]
            )
            logger.info(f"[load_prescription_employee_options] Loaded {len(options)} employees")
            return options, False, {"display": "none"}

        error_msg = response.get("message", "Unknown") if response else "No response"
        logger.warning(f"[load_prescription_employee_options] Failed: {error_msg}")
        return [], True, {"display": "inline-block"}

    @app.callback(
        Output("prescription-laboratories-filter", "options"),
        [
            Input("url", "pathname"),
            Input("auth-ready", "data"),
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # FIX: Moved from Input to State to break circular dependency
        ],
        prevent_initial_call=False,
    )
    def load_prescription_laboratory_options(
        pathname: str,
        auth_ready: Optional[Dict],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> list:
        """
        Carga opciones de laboratorios para filtro de prescripción.
        """
        logger.debug(f"[load_prescription_laboratory_options] pathname: {pathname}, auth_ready: {bool(auth_ready)}")

        if pathname != "/prescription":
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            raise PreventUpdate

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        if not auth_headers:
            logger.warning("[load_prescription_laboratory_options] No auth headers available")
            raise PreventUpdate

        logger.info(f"[load_prescription_laboratory_options] Fetching laboratories")

        response = request_coordinator.make_request(
            endpoint="/api/v1/laboratory-mapping/codes-to-names",
            method="GET",
            params={"page": 1, "per_page": 100, "envelope": True},
            timeout=15,
            auth_headers=auth_headers,
        )

        if response and isinstance(response, dict) and "items" in response:
            labs = response["items"]
            options = [{"label": name, "value": code} for code, name in labs.items() if code and name]
            options.sort(key=lambda x: x["label"])
            logger.info(f"[load_prescription_laboratory_options] Loaded {len(options)} laboratories (envelope format)")
            return options
        elif response and isinstance(response, dict) and "error" not in response:
            options = [{"label": name, "value": code} for code, name in response.items() if code and name]
            options.sort(key=lambda x: x["label"])
            logger.info(f"[load_prescription_laboratory_options] Loaded {len(options)} laboratories (direct format)")
            return options

        error_msg = response.get("message", "Unknown") if isinstance(response, dict) else "Invalid response"
        logger.warning(f"[load_prescription_laboratory_options] Failed: {error_msg}")
        return []

    # =========================================================================
    # Issue #537 FIX: Invalidar cache cuando cambian las fechas
    # =========================================================================
    @app.callback(
        [
            Output("prescription-category-products-store", "data", allow_duplicate=True),
            Output("prescription-ingredient-products-store", "data", allow_duplicate=True),
        ],
        [
            Input("prescription-date-range", "start_date"),
            Input("prescription-date-range", "end_date"),
        ],
        prevent_initial_call=True,
    )
    def invalidate_products_cache_on_date_change(start_date, end_date):
        """
        Invalida el cache de productos cuando cambian las fechas.

        Issue #537: Cuando el usuario cambia las fechas del filtro global,
        los productos cacheados en los acordeones deben recargarse.
        """
        logger.debug(f"[invalidate_products_cache] Date changed: {start_date} to {end_date}")
        return {}, {}  # Limpiar ambos caches

    logger.info("[prescription/products] 8 callbacks registered")
