"""
Overview callbacks for prescription dashboard.

Handles KPIs, context panel, and ATC distribution rendering.
"""

import logging
from collections import defaultdict
from datetime import datetime
from typing import Optional, Dict, Any, List

from dash import Input, Output, State, html, dcc
from dash.exceptions import PreventUpdate
import dash_bootstrap_components as dbc
import plotly.graph_objects as go

from utils.helpers import format_currency, format_percentage, format_number
from components.prescription.waterfall_analysis import create_waterfall_chart, create_category_chips
from components.skeleton_loader import create_chart_skeleton
from components.empty_state import create_analysis_empty_state, create_error_state
from dash_iconify import DashIconify
from styles.design_tokens import COLORS
from utils.constants import format_category_name

logger = logging.getLogger(__name__)


def register_overview_callbacks(app):
    """
    Registra callbacks de overview y KPIs.

    Callbacks:
    1. update_all_overview_components: KPIs, waterfall, context panel
    2. update_atc_distribution_chart: Gráfico de distribución ATC
    """

    @app.callback(
        [
            Output("prescription-total-sales-value", "children"),
            Output("prescription-total-units-value", "children"),
            Output("prescription-percentage-value", "children"),
            Output("prescription-atc-coverage-value", "children"),
            Output("atc-coverage-dynamic-text", "children"),
            Output("waterfall-analysis-container", "children"),
            Output("prescription-context-total-sales", "children"),
            Output("prescription-context-total-percentage", "children"),
            Output("prescription-context-total-units", "children"),
            Output("prescription-context-units-percentage", "children"),
            Output("prescription-products-trigger-store", "data"),
            Output("prescription-evolution-trigger-store", "data"),
        ],
        [
            Input("prescription-overview-store", "data"),
            Input("prescription-waterfall-store", "data"),
            Input("evolution-chart-type", "value"),
        ],
        prevent_initial_call=True,
    )
    def update_all_overview_components(
        overview_data: Optional[Dict],
        waterfall_data: Optional[Dict],
        chart_type: str,
    ) -> tuple:
        """
        Actualiza KPIs, visualizaciones Y métricas de contexto desde overview store.

        No requiere auth check (lee de store autenticado).
        REGLA #11: Solo este callback escucha prescription-overview-store.
        Nota: El treemap de contexto ahora se actualiza en callback separado.
        """
        logger.debug("[update_all_overview_components] Callback triggered")

        # Valores por defecto para KPIs
        default_kpis = ("--", "--", "--", "--", "Cobertura ATC no disponible")

        # Default context panel metrics
        default_context = ("--", "--", "--", "--")

        # Placeholder vacío para charts
        empty_chart = create_analysis_empty_state(
            message="No hay datos disponibles",
            icon="mdi:chart-line",
            suggestions=[
                "Aplica filtros para ver visualizaciones",
                "Verifica que existan ventas en el período seleccionado",
            ],
            min_height="500px",
            section_name="Gráficos de prescripción",
        )

        if not overview_data or "kpis" not in overview_data:
            logger.warning("[update_all_overview_components] No overview data available")
            return (*default_kpis, empty_chart, *default_context, None, None)

        # 1. Actualizar KPIs
        kpis = overview_data.get("kpis", {})

        total_sales = format_currency(kpis.get("total_sales", 0))
        total_units = format_number(int(kpis.get("total_units", 0)))
        prescription_percentage = format_percentage(kpis.get("prescription_percentage", 0))
        atc_coverage = format_percentage(kpis.get("atc_coverage", 0))

        # Texto dinámico de cobertura ATC
        coverage_value = kpis.get("atc_coverage", 0)
        if coverage_value > 0.20:
            dynamic_text = "Excelente cobertura de clasificación terapéutica"
        elif coverage_value > 0.15:
            dynamic_text = "Buena cobertura ATC para análisis terapéutico"
        elif coverage_value > 0.10:
            dynamic_text = "Cobertura ATC moderada"
        else:
            dynamic_text = "Cobertura ATC limitada - Revisar productos sin clasificar"

        logger.info(
            f"[update_all_overview_components] KPIs updated: sales={total_sales}, "
            f"units={total_units}, atc_coverage={atc_coverage}"
        )

        # 2. Waterfall Chart + Category Chips (Issue #454)
        if waterfall_data and waterfall_data.get("waterfall_data"):
            logger.info(
                f"[update_all_overview_components] Rendering waterfall chart - "
                f"{len(waterfall_data.get('waterfall_data', []))} category changes"
            )
            try:
                # Issue #454: Añadir chips clicables debajo del gráfico para mejor UX
                waterfall_chart = html.Div([
                    create_waterfall_chart(waterfall_data),
                    create_category_chips(waterfall_data),
                ])
            except Exception as e:
                logger.error(
                    f"[update_all_overview_components] Error rendering waterfall: {str(e)}",
                    exc_info=True,
                )
                waterfall_chart = create_error_state(
                    error_message="Error al renderizar análisis de crecimiento",
                    min_height="500px",
                )
        elif waterfall_data is None:
            waterfall_chart = create_chart_skeleton(height="500px")
        else:
            waterfall_chart = create_analysis_empty_state(
                message="No hay datos disponibles para el período seleccionado",
                icon="mdi:chart-waterfall",
                suggestions=[
                    "Ajusta el rango de fechas",
                    "Requiere datos del período actual Y anterior para comparar",
                    "Verifica cobertura ATC (banner superior)",
                ],
                min_height="500px",
                section_name="Variación",
            )

        # 3. Context Panel metrics (treemap actualizado en callback separado)
        context_sales = total_sales
        context_sales_pct = f"({format_percentage(kpis.get('prescription_percentage', 0))})"
        context_units = total_units
        context_units_pct = f"({format_percentage(kpis.get('prescription_percentage', 0))})"

        context_metrics = (context_sales, context_sales_pct, context_units, context_units_pct)

        # REGLA #11 Fix: Triggers separados para nuevos callbacks
        evolution_trigger = {
            "triggered": True,
            "timestamp": datetime.now().isoformat(),
            "overview_data": overview_data,
        }
        products_trigger = {"triggered": True, "timestamp": datetime.now().isoformat()}

        return (
            total_sales,
            total_units,
            prescription_percentage,
            atc_coverage,
            dynamic_text,
            waterfall_chart,
            *context_metrics,
            products_trigger,
            evolution_trigger,
        )

    @app.callback(
        Output("atc-distribution-container", "children"),
        Input("prescription-atc-distribution-store", "data"),
        prevent_initial_call=True,
    )
    def update_atc_distribution_chart(atc_data: Optional[Dict]) -> Any:
        """
        Renderiza gráfico de distribución ATC desde store.

        No requiere auth check (lee de store autenticado).
        REGLA #11: Solo este callback escucha prescription-atc-distribution-store.
        """
        logger.debug("[update_atc_distribution_chart] Callback triggered")

        if not atc_data:
            logger.warning("[update_atc_distribution_chart] No ATC data available")
            return create_analysis_empty_state(
                message="No hay datos de distribución ATC disponibles",
                icon="mdi:chart-tree",
                suggestions=[
                    "Aplica filtros para visualizar clasificación terapéutica",
                    "Verifica que el período tenga ventas de prescripción",
                ],
                min_height="600px",
                section_name="Distribución ATC",
            )

        nodes = atc_data.get("atc_distribution", [])

        if not nodes:
            logger.info("[update_atc_distribution_chart] Empty nodes")
            return create_analysis_empty_state(
                message="No hay productos con clasificación ATC en el período seleccionado",
                icon="mdi:pill",
                suggestions=[
                    "Los productos vendidos pueden no tener código ATC asignado",
                    "Revisa el banner de cobertura ATC en la parte superior",
                    "Productos como dermocosméticos o material sanitario no tienen ATC",
                ],
                min_height="600px",
                section_name="Distribución ATC",
            )

        logger.info(f"[update_atc_distribution_chart] Rendering ATC chart with {len(nodes)} top-level nodes")

        try:
            # Create horizontal bar chart for ATC distribution
            labels = []
            values = []
            percentages = []

            for node in nodes[:15]:
                labels.append(f"{node['atc_code']} - {node['atc_name'][:30]}")
                values.append(float(node["sales"]))
                percentages.append(node["percentage"])

            fig = go.Figure(
                data=[
                    go.Bar(
                        y=labels,
                        x=values,
                        orientation="h",
                        text=[
                            f"{format_currency(v)}<br>{format_percentage(p / 100)}"
                            for v, p in zip(values, percentages)
                        ],
                        textposition="auto",
                        marker=dict(
                            color=COLORS["primary"],
                            line=dict(color=COLORS["border_light"], width=1),
                        ),
                        hovertemplate="<b>%{y}</b><br>Ventas: %{x:,.2f}€<extra></extra>",
                    )
                ]
            )

            fig.update_layout(
                title={
                    "text": "Distribución de Ventas por Código ATC",
                    "x": 0.5,
                    "xanchor": "center",
                    "font": {"size": 16},
                },
                xaxis_title="Ventas (€)",
                yaxis_title="Código ATC",
                height=600,
                margin=dict(l=250, r=40, t=60, b=60),
                paper_bgcolor="white",
                plot_bgcolor="white",
                font=dict(size=12),
                xaxis=dict(gridcolor=COLORS["border_light"], showgrid=True),
                yaxis=dict(autorange="reversed"),
            )

            return dcc.Graph(figure=fig, config={"displayModeBar": False}, style={"height": "600px"})

        except Exception as e:
            logger.error(
                f"[update_atc_distribution_chart] Error rendering ATC chart: {str(e)}",
                exc_info=True,
            )
            return create_error_state(
                error_message=f"Error al renderizar distribución ATC: {str(e)}",
                min_height="600px",
            )

    # ============================================================================
    # 3. CALLBACK CONTEXT TREEMAP - Ventas Totales → Prescripción/Libre → Categorías
    # ============================================================================

    @app.callback(
        Output("prescription-context-treemap", "figure"),
        [
            Input("prescription-overview-store", "data"),
            Input("url", "pathname"),  # Issue #500: Trigger on page navigation
        ],
        [
            State("prescription-date-range", "start_date"),
            State("prescription-date-range", "end_date"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),  # Issue #540: Token restoration for multi-worker Render (REGLA #7.6)
        ],
        prevent_initial_call=False,  # Issue #500: Load on initial render (session store may have data)
    )
    def update_prescription_context_treemap(
        overview_data: Optional[Dict],
        pathname: Optional[str],
        start_date: Optional[str],
        end_date: Optional[str],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],  # Issue #540: Token restoration
    ) -> go.Figure:
        """
        Actualiza el treemap de contexto de ventas.

        Issue #500: Añadido pathname como Input para re-trigger en navegación.

        Jerarquía:
        - Ventas Totales
          - Prescripción (con categorías: Con Receta, Venta Libre, tiras, etc.)
          - Libre/OTC
        """
        from utils.auth_helpers import is_user_authenticated
        from utils.auth import auth_manager
        from utils.pharmacy_context import get_current_pharmacy_id
        from utils.config import BACKEND_URL
        import requests

        logger.debug(f"[update_prescription_context_treemap] Callback triggered - pathname: {pathname}")

        # Guard 0: Solo ejecutar en /prescription
        if pathname != "/prescription":
            raise PreventUpdate

        # Guard 1: Verificar autenticación
        if not is_user_authenticated(auth_state):
            return _create_empty_treemap("Sin sesión activa")

        # Guard 2: Obtener pharmacy_id
        try:
            pharmacy_id = get_current_pharmacy_id()
            if not pharmacy_id:
                return _create_empty_treemap("Farmacia no configurada")
        except ValueError:
            return _create_empty_treemap("Cargando datos...")

        # Guard 3: Si no hay overview_data
        if not overview_data:
            return _create_empty_treemap("Sin datos de ventas")

        try:
            # Issue #540: Restore tokens for multi-worker Render (REGLA #7.6)
            # En Gunicorn multi-worker, cada worker tiene su propio auth_manager singleton
            if auth_tokens and "tokens" in auth_tokens:
                auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

            token = auth_manager.get_access_token()
            if not token:
                return _create_empty_treemap("Sesión expirada")

            # 1. Obtener ventas totales de la farmacia desde context-treemap
            params = {}
            if start_date:
                params["start_date"] = start_date[:10] if isinstance(start_date, str) else start_date.strftime("%Y-%m-%d")
            if end_date:
                params["end_date"] = end_date[:10] if isinstance(end_date, str) else end_date.strftime("%Y-%m-%d")

            response = requests.get(
                f"{BACKEND_URL}/api/v1/analysis/context-treemap/{pharmacy_id}",
                params=params,
                headers={"Authorization": f"Bearer {token}"},
                timeout=30,
            )

            if response.status_code != 200:
                logger.warning(f"[update_prescription_context_treemap] API error: {response.status_code}")
                # Fallback: usar solo los datos de prescripción del overview
                category_summary = overview_data.get("category_summary", [])
                return _create_prescription_treemap(category_summary)

            treemap_api_data = response.json()
            hierarchy = treemap_api_data.get("hierarchy", {})
            total_sales = hierarchy.get("value", 0)

            # 2. Obtener categorías de prescripción del overview_data
            category_summary = overview_data.get("category_summary", [])
            prescription_sales = sum(float(cat.get("total_sales", 0)) for cat in category_summary)

            # 3. Calcular ventas libres (total - prescripción)
            libre_sales = max(0, total_sales - prescription_sales)

            logger.info(
                f"[update_prescription_context_treemap] Total: {total_sales:.2f}€, "
                f"Prescripción: {prescription_sales:.2f}€, Libre: {libre_sales:.2f}€"
            )

            return _create_hierarchical_treemap(
                total_sales=total_sales,
                prescription_sales=prescription_sales,
                libre_sales=libre_sales,
                category_summary=category_summary,
            )

        except requests.exceptions.Timeout:
            logger.error("[update_prescription_context_treemap] Timeout")
            category_summary = overview_data.get("category_summary", [])
            return _create_prescription_treemap(category_summary)
        except Exception as e:
            logger.error(f"[update_prescription_context_treemap] Error: {str(e)}", exc_info=True)
            return _create_empty_treemap("Error interno")

    logger.info("[prescription/overview] 3 callbacks registered")


def _create_empty_treemap(message: str) -> go.Figure:
    """Crear treemap vacío con mensaje."""
    fig = go.Figure()
    fig.add_annotation(
        text=message,
        showarrow=False,
        font={"size": 14, "color": "#6c757d"},
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
    )
    fig.update_layout(
        margin={"t": 10, "b": 10, "l": 10, "r": 10},
        height=280,
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )
    return fig


def _create_prescription_treemap(category_summary: List[Dict[str, Any]]) -> go.Figure:
    """
    Fallback: Crear figura treemap desde category_summary de prescripción (sin total farmacia).

    Muestra: Prescripción → Categorías (medicamentos, tiras, etc.)
    """
    if not category_summary:
        return _create_empty_treemap("Sin categorías de prescripción")

    # Calcular total
    total_sales = sum(float(cat.get("total_sales", 0)) for cat in category_summary)

    # Construir arrays para treemap
    labels = ["Prescripción"]
    parents = [""]
    values = [total_sales]
    texts = [format_currency(total_sales)]
    colors = [COLORS.get("primary", "#2c7be5")]

    # Colores semánticos por categoría
    category_colors = {
        "medicamentos": COLORS.get("primary", "#2c7be5"),
        "tiras_reactivas_glucosa": "#17a2b8",
        "absorbentes": "#6c757d",
        "alimentacion_especial": "#28a745",
        "optica": "#ffc107",
        "otros": "#6c757d",
    }

    for cat in category_summary:
        category_name = cat.get("category", "otros")
        cat_sales = float(cat.get("total_sales", 0))
        cat_pct = cat.get("percentage", 0)

        labels.append(format_category_name(category_name))
        parents.append("Prescripción")
        values.append(cat_sales)
        texts.append(f"{format_currency(cat_sales)}<br>{format_percentage(cat_pct / 100)}")
        colors.append(category_colors.get(category_name, "#6c757d"))

    fig = go.Figure(
        go.Treemap(
            labels=labels,
            parents=parents,
            values=values,
            text=texts,
            marker={"colors": colors},
            textinfo="label+text",
            textfont={"size": 11},
            hovertemplate="<b>%{label}</b><br>%{text}<extra></extra>",
            branchvalues="total",
            maxdepth=-1,
        )
    )

    fig.update_layout(
        margin={"t": 5, "b": 5, "l": 5, "r": 5},
        height=280,
        paper_bgcolor="rgba(0,0,0,0)",
        font={"family": "Arial, sans-serif"},
    )

    return fig


def _create_hierarchical_treemap(
    total_sales: float,
    prescription_sales: float,
    libre_sales: float,
    category_summary: List[Dict[str, Any]],
) -> go.Figure:
    """
    Crear treemap jerárquico completo.

    Jerarquía:
    - Ventas Totales
      - Prescripción
        - Con Receta
        - Venta Libre
        - Tiras Reactivas, etc.
      - Libre/OTC
    """
    if total_sales <= 0:
        return _create_empty_treemap("Sin datos de ventas")

    # Colores semánticos
    color_total = "#343a40"  # Gris oscuro para total
    color_prescription = COLORS.get("primary", "#2c7be5")  # Azul para prescripción
    color_libre = "#28a745"  # Verde para libre/OTC

    category_colors = {
        "Con receta": "#2c7be5",
        "Venta Libre": "#17a2b8",
        "No determinado": "#6c757d",
        "tiras_reactivas_glucosa": "#fd7e14",
        "dietoterapicos": "#28a745",
        "incontinencia_financiada": "#6610f2",
        "efectos_financiados": "#e83e8c",
        "ortopedia_financiada": "#20c997",
        "formulas_magistrales": "#6c757d",
        "vacunas_individualizadas": "#dc3545",
    }

    # Construir arrays para treemap
    labels = ["Ventas Totales", "Prescripción", "Libre/OTC"]
    parents = ["", "Ventas Totales", "Ventas Totales"]
    values = [total_sales, prescription_sales, libre_sales]

    # Calcular porcentajes
    pct_prescription = (prescription_sales / total_sales * 100) if total_sales > 0 else 0
    pct_libre = (libre_sales / total_sales * 100) if total_sales > 0 else 0

    texts = [
        format_currency(total_sales),
        f"{format_currency(prescription_sales)}<br>{format_percentage(pct_prescription / 100)}",
        f"{format_currency(libre_sales)}<br>{format_percentage(pct_libre / 100)}",
    ]
    colors = [color_total, color_prescription, color_libre]

    # Añadir categorías de prescripción como hijos de "Prescripción"
    for cat in category_summary:
        category_name = cat.get("category", "otros")
        cat_sales = float(cat.get("total_sales", 0))
        cat_pct = cat.get("percentage", 0)

        display_name = format_category_name(category_name)
        color = category_colors.get(category_name, "#6c757d")

        labels.append(display_name)
        parents.append("Prescripción")
        values.append(cat_sales)
        texts.append(f"{format_currency(cat_sales)}<br>{format_percentage(cat_pct / 100)}")
        colors.append(color)

    fig = go.Figure(
        go.Treemap(
            labels=labels,
            parents=parents,
            values=values,
            text=texts,
            marker={"colors": colors},
            textinfo="label+text",
            textfont={"size": 10},
            hovertemplate="<b>%{label}</b><br>%{text}<extra></extra>",
            branchvalues="total",
            maxdepth=-1,
        )
    )

    fig.update_layout(
        margin={"t": 5, "b": 5, "l": 5, "r": 5},
        height=280,
        paper_bgcolor="rgba(0,0,0,0)",
        font={"family": "Arial, sans-serif"},
    )

    return fig
