"""
Filter synchronization callbacks for prescription dashboard.

Handles date range, slider, YoY calculation, and filter dropdowns.

Issue #529 Fixes:
- Year boundary: Uses data year (max_date.year) instead of current year
- ISO format: Cleans T00:00:00 suffix from dates before parsing
- Infinite loop guards: Prevents redundant callback updates
"""

import logging
from datetime import datetime, timedelta
from typing import Optional, Dict, List

from dash import Input, Output, State, no_update
from dash.exceptions import PreventUpdate

from utils.auth_helpers import is_user_authenticated, get_auth_headers_from_tokens
from utils.helpers import format_month_year, MESES_ESPANOL_CORTO
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def clean_iso_date(date_str: Optional[str]) -> Optional[str]:
    """Remove ISO time suffix (T00:00:00) from date string."""
    return date_str.split("T")[0] if date_str else None

# Constantes de formato (usar MESES_ESPANOL_CORTO de helpers para consistencia)
MONTH_NAMES = {
    f"{i:02d}": name for i, name in MESES_ESPANOL_CORTO.items()
}

QUARTER_MONTHS = {
    "Q1": ["01", "02", "03"],
    "Q2": ["04", "05", "06"],
    "Q3": ["07", "08", "09"],
    "Q4": ["10", "11", "12"],
}


def get_quarter_from_month(month: str) -> str:
    """Obtiene el trimestre para un mes dado."""
    for quarter, months in QUARTER_MONTHS.items():
        if month in months:
            return quarter
    return "Q1"


def format_period_label(period: str, comparison_type: str = "mom") -> str:
    """Formatea un período según el tipo de comparación."""
    if "-Q" in period:
        return period.replace("-", " ")
    year, month = period.split("-")
    if comparison_type == "qoq":
        quarter = get_quarter_from_month(month)
        return f"{quarter} {year}"
    return f"{MONTH_NAMES.get(month, month)} {year}"


def register_filter_callbacks(app):
    """
    Registra callbacks de sincronización de filtros.

    Callbacks:
    1. populate_period_options_by_type: Opciones dropdown período
    2. apply_prescription_date_range: Store → DatePicker
    3. sync_slider_to_datepicker: Slider → DatePicker
    4. sync_datepicker_to_yoy: DatePicker → YoY store
    5. load_prescription_employee_options: Opciones empleados
    6. load_prescription_laboratory_options: Opciones laboratorios
    """

    @app.callback(
        [
            Output("waterfall-period-base", "options"),
            Output("waterfall-period-base", "value"),
            Output("waterfall-period-comparison", "options"),
            Output("waterfall-period-comparison", "value"),
        ],
        [
            Input("prescription-overview-store", "data"),
            Input("waterfall-comparison-type", "value"),
        ],
        [State("waterfall-period-base", "value")],
        prevent_initial_call=True,
    )
    def populate_period_options_by_type(
        overview_data: Optional[Dict],
        comparison_type: str,
        current_base_value: Optional[str],
    ) -> tuple:
        """Poblar opciones del dropdown de período según tipo de comparación."""
        if not overview_data:
            return [], no_update, [], None

        time_series = overview_data.get("time_series", [])
        if not time_series:
            return [], no_update, [], None

        periods = sorted(set(item.get("period") for item in time_series if item.get("period")))
        if not periods:
            return [], no_update, [], None

        comparison_type = comparison_type or "qoq"

        if comparison_type == "qoq":
            quarters_set = set()
            for period in periods:
                year, month = period.split("-")
                quarter = get_quarter_from_month(month)
                quarters_set.add(f"{year}-{quarter}")

            quarters = sorted(quarters_set)
            options = [{"label": q.replace("-", " "), "value": q} for q in quarters]
            default_value = quarters[-1] if quarters else None

            if current_base_value and current_base_value not in quarters:
                current_base_value = None

            base_value = current_base_value if current_base_value else default_value
        else:
            options = [
                {"label": format_period_label(p, comparison_type), "value": p}
                for p in periods
            ]
            default_value = periods[-1] if periods else None

            if current_base_value and current_base_value not in periods:
                current_base_value = None

            base_value = current_base_value if current_base_value else default_value

        logger.info(f"[populate_period_options_by_type] Type={comparison_type}, {len(options)} opciones")
        return options, base_value, [], None

    @app.callback(
        [
            Output("prescription-date-range", "start_date"),
            Output("prescription-date-range", "end_date"),
            Output("prescription-date-range", "min_date_allowed"),
            Output("prescription-date-range", "max_date_allowed"),
            Output("prescription-date-slider", "min"),
            Output("prescription-date-slider", "max"),
            Output("prescription-date-slider", "marks"),
            Output("prescription-date-slider", "value"),
            Output("waterfall-current-period-display", "children"),
            Output("waterfall-period-comparison-display", "children"),
            Output("waterfall-period-comparison-store", "data"),
        ],
        [Input("prescription-date-range-store", "data")],
        [
            State("prescription-date-range", "start_date"),
            State("prescription-date-range", "end_date"),
        ],
        prevent_initial_call=False,
    )
    def apply_prescription_date_range(
        store_data: Optional[Dict],
        current_start: Optional[str],
        current_end: Optional[str],
    ):
        """Aplicar rango de fechas al DatePicker desde Store."""
        if not store_data or not store_data.get("ready"):
            raise PreventUpdate

        min_date = store_data.get("min_date")
        max_date = store_data.get("max_date")

        if not min_date or not max_date:
            raise PreventUpdate

        # Calcular fechas por defecto (desde 1 de enero del año de los datos más recientes)
        max_dt = datetime.fromisoformat(clean_iso_date(max_date))
        min_dt = datetime.fromisoformat(clean_iso_date(min_date))

        # Issue #529 FIX: Usar el año de max_date, no datetime.now().year
        # Esto evita problemas cuando estamos en un año nuevo pero los datos son del año anterior
        data_year = max_dt.year
        default_start = datetime(data_year, 1, 1)

        # Asegurar que la fecha de inicio esté dentro del rango de datos disponibles
        if default_start < min_dt:
            default_start = min_dt
        if default_start > max_dt:
            # Fallback: usar 90 días antes de max_date, pero nunca antes de min_date
            default_start = max(min_dt, max_dt - timedelta(days=90))

        default_start_str = default_start.strftime("%Y-%m-%d")

        # Determinar fecha de inicio
        # Si hay fecha guardada, verificar si es del año actual
        if current_start:
            try:
                # Issue #529 FIX: Limpiar T00:00:00 antes de parsear
                current_start_clean = clean_iso_date(current_start)
                saved_start = datetime.fromisoformat(current_start_clean)
                # Si la fecha guardada es anterior al 1 de enero del año actual, usar default
                if saved_start < default_start:
                    start_date = default_start_str
                    logger.info(f"[apply_prescription_date_range] Reset start date from {current_start} to {default_start_str}")
                else:
                    start_date = current_start_clean  # Usar versión limpia
            except (ValueError, TypeError):
                start_date = default_start_str
        else:
            start_date = default_start_str

        # Issue #529 FIX: Limpiar formato de fecha (puede venir con T00:00:00)
        max_date_clean = clean_iso_date(max_date)
        end_date = clean_iso_date(current_end) or max_date_clean

        # Issue #529 FIX: Validar que start_date <= end_date
        try:
            start_dt_check = datetime.fromisoformat(start_date)
            end_dt_check = datetime.fromisoformat(end_date)
            if start_dt_check > end_dt_check:
                logger.warning(f"[apply_prescription_date_range] Invalid range: {start_date} > {end_date}, fixing...")
                # Issue #529 Review: Usar ventana de 30 días en lugar de mismo día
                start_date = max(min_dt, end_dt_check - timedelta(days=30)).strftime("%Y-%m-%d")
        except (ValueError, TypeError) as e:
            logger.warning(f"[apply_prescription_date_range] Date validation error: {e}")

        # Issue #529 Review: Guard movido aquí (antes de cálculos costosos)
        # Verificar si las fechas ya están correctamente establecidas
        current_start_clean = clean_iso_date(current_start)
        current_end_clean = clean_iso_date(current_end)

        if current_start_clean == start_date and current_end_clean == end_date:
            logger.debug(f"[apply_prescription_date_range] Fechas sin cambio: {start_date} to {end_date} - PreventUpdate")
            raise PreventUpdate

        # Issue #540: Optimizar cálculo de marks (sin iterar todos los días)
        # Calcula directamente las posiciones de los primeros de mes
        try:
            total_days = (max_dt - min_dt).days

            # Generar marks directamente para el primero de cada mes
            marks = {}
            # Empezar desde el primer día del mes siguiente a min_dt (o min_dt si es día 1)
            if min_dt.day == 1:
                current_month = min_dt
            else:
                # Avanzar al primer día del mes siguiente
                if min_dt.month == 12:
                    current_month = datetime(min_dt.year + 1, 1, 1)
                else:
                    current_month = datetime(min_dt.year, min_dt.month + 1, 1)

            while current_month <= max_dt:
                day_idx = (current_month - min_dt).days
                marks[day_idx] = format_month_year(current_month)
                # Avanzar al siguiente mes
                if current_month.month == 12:
                    current_month = datetime(current_month.year + 1, 1, 1)
                else:
                    current_month = datetime(current_month.year, current_month.month + 1, 1)

            # Issue #540: Limitar a ~6 marks para mejor UX y rendimiento
            if len(marks) > 6:
                mark_keys = sorted(marks.keys())
                step = len(mark_keys) // 6
                marks = {k: marks[k] for i, k in enumerate(mark_keys) if i % step == 0 or k == mark_keys[-1]}
        except (ValueError, TypeError, KeyError) as e:
            logger.debug(f"[apply_prescription_date_range] Slider marks fallback: {e}")
            total_days = 365
            marks = {}

        # Calcular YoY comparison period y valores del slider
        try:
            # Issue #529 FIX: Asegurar formato limpio (sin T00:00:00)
            start_dt = datetime.fromisoformat(clean_iso_date(start_date))
            end_dt = datetime.fromisoformat(clean_iso_date(end_date))
            comp_start = start_dt - timedelta(days=365)
            comp_end = end_dt - timedelta(days=365)

            current_display = f"{start_dt.strftime('%d/%m/%Y')} - {end_dt.strftime('%d/%m/%Y')}"
            comparison_display = f"{comp_start.strftime('%d/%m/%Y')} - {comp_end.strftime('%d/%m/%Y')}"

            comparison_store = {
                "comparison_date_from": comp_start.strftime("%Y-%m-%d"),
                "comparison_date_to": comp_end.strftime("%Y-%m-%d"),
                "type": "yoy",
            }

            # Calcular valores del slider correspondientes a start_date y end_date
            slider_start = (start_dt - min_dt).days
            slider_end = (end_dt - min_dt).days
            # Asegurar que estén dentro del rango
            slider_start = max(0, min(slider_start, total_days))
            slider_end = max(0, min(slider_end, total_days))
            slider_value = [slider_start, slider_end]
        except Exception as e:
            logger.debug(f"[apply_prescription_date_range] YoY calculation fallback: {e}")
            current_display = "(según filtros aplicados)"
            comparison_display = "(auto-calculado)"
            comparison_store = None
            slider_value = [0, total_days]

        logger.info(f"[apply_prescription_date_range] Applied: {start_date} to {end_date}, slider: {slider_value}")

        return (
            start_date,
            end_date,
            min_date,
            max_date,
            0,
            total_days,
            marks,
            slider_value,
            current_display,
            comparison_display,
            comparison_store,
        )

    @app.callback(
        [
            Output("prescription-date-range", "start_date", allow_duplicate=True),
            Output("prescription-date-range", "end_date", allow_duplicate=True),
        ],
        [Input("prescription-date-slider", "value")],
        [
            State("prescription-date-range-store", "data"),
            State("prescription-date-range", "start_date"),
            State("prescription-date-range", "end_date"),
        ],
        prevent_initial_call=True,
    )
    def sync_slider_to_datepicker(slider_value, store_data, current_start, current_end):
        """
        Sincronizar slider con DatePicker.

        Issue #529: Fix infinite loop - Solo actualizar si las fechas cambian.
        """
        if not slider_value or not store_data:
            raise PreventUpdate

        # Issue #529 FIX: Si las fechas actuales son None, significa que
        # apply_prescription_date_range aún no las ha establecido o acaba de hacerlo.
        # Dejar que ese callback maneje la inicialización.
        if not current_start or not current_end:
            logger.debug("[sync_slider_to_datepicker] Fechas actuales None - PreventUpdate")
            raise PreventUpdate

        min_date = store_data.get("min_date")
        if not min_date:
            raise PreventUpdate

        try:
            # Issue #529 FIX: Limpiar formato de min_date (puede tener T00:00:00)
            min_dt = datetime.fromisoformat(clean_iso_date(min_date))
            start_dt = min_dt + timedelta(days=slider_value[0])
            end_dt = min_dt + timedelta(days=slider_value[1])
            new_start = start_dt.strftime("%Y-%m-%d")
            new_end = end_dt.strftime("%Y-%m-%d")

            # Issue #529 FIX: Limpiar fechas actuales para comparación correcta
            # Las fechas pueden venir con T00:00:00 del DatePicker
            current_start_clean = clean_iso_date(current_start)
            current_end_clean = clean_iso_date(current_end)

            # Issue #529: Prevenir infinite loop - solo actualizar si hay cambio real
            if new_start == current_start_clean and new_end == current_end_clean:
                logger.debug("[sync_slider_to_datepicker] Fechas sin cambio - PreventUpdate")
                raise PreventUpdate

            logger.debug(f"[sync_slider_to_datepicker] Actualizando: {current_start_clean}->{new_start}, {current_end_clean}->{new_end}")
            return new_start, new_end
        except (ValueError, TypeError, KeyError) as e:
            # Issue #529 Review: Excepciones específicas en lugar de Exception genérico
            logger.debug(f"[sync_slider_to_datepicker] Slider sync error: {e}")
            raise PreventUpdate

    @app.callback(
        [
            Output("waterfall-current-period-display", "children", allow_duplicate=True),
            Output("waterfall-period-comparison-display", "children", allow_duplicate=True),
            Output("waterfall-period-comparison-store", "data", allow_duplicate=True),
        ],
        [
            Input("prescription-date-range", "start_date"),
            Input("prescription-date-range", "end_date"),
        ],
        [State("waterfall-period-comparison-store", "data")],
        prevent_initial_call=True,
    )
    def sync_datepicker_to_yoy(
        start_date: Optional[str],
        end_date: Optional[str],
        current_comparison_store: Optional[Dict],
    ):
        """
        Calcular período YoY cuando cambian las fechas.

        Issue #529: Fix infinite loop - Solo actualizar si el período cambia.
        """
        if not start_date or not end_date:
            raise PreventUpdate

        try:
            # Issue #529 FIX: Limpiar formato de fecha (puede venir con T00:00:00)
            start_clean = clean_iso_date(start_date)
            end_clean = clean_iso_date(end_date)

            if not start_clean or not end_clean:
                logger.debug("[sync_datepicker_to_yoy] Fechas vacías después de limpiar")
                raise PreventUpdate

            start_dt = datetime.fromisoformat(start_clean)
            end_dt = datetime.fromisoformat(end_clean)

            # Issue #529 FIX: Validar que start <= end
            if start_dt > end_dt:
                logger.warning(f"[sync_datepicker_to_yoy] Invalid range: {start_clean} > {end_clean}")
                raise PreventUpdate
            comp_start = start_dt - timedelta(days=365)
            comp_end = end_dt - timedelta(days=365)

            new_comp_from = comp_start.strftime("%Y-%m-%d")
            new_comp_to = comp_end.strftime("%Y-%m-%d")

            # Issue #529: Prevenir infinite loop - solo actualizar si hay cambio real
            if current_comparison_store:
                if (
                    current_comparison_store.get("comparison_date_from") == new_comp_from
                    and current_comparison_store.get("comparison_date_to") == new_comp_to
                ):
                    logger.debug("[sync_datepicker_to_yoy] Período sin cambio - PreventUpdate")
                    raise PreventUpdate

            current_display = f"{start_dt.strftime('%d/%m/%Y')} - {end_dt.strftime('%d/%m/%Y')}"
            comparison_display = f"{comp_start.strftime('%d/%m/%Y')} - {comp_end.strftime('%d/%m/%Y')}"

            comparison_store = {
                "comparison_date_from": new_comp_from,
                "comparison_date_to": new_comp_to,
                "type": "yoy",
            }

            return current_display, comparison_display, comparison_store
        except PreventUpdate:
            raise
        except Exception as e:
            logger.error(f"[sync_datepicker_to_yoy] Error parsing dates: start={start_date}, end={end_date}, error={e}")
            raise PreventUpdate

    # NOTE: Employee and laboratory options callbacks moved to products.py
    # They have more comprehensive implementations (PRO tier checks, better error handling)

    logger.info("[prescription/filters] 4 callbacks registered")
