"""
Evolution chart callbacks for prescription dashboard.

Handles temporal evolution with drill-down functionality.
"""

import logging
from collections import defaultdict
from typing import Optional, Dict

from dash import Input, Output, State, ctx
from dash.exceptions import PreventUpdate

from styles.design_tokens import COLORS
from utils.drill_down_helpers import (
    get_quarter_from_period,
    get_quarter_label,
    months_in_quarter,
    parse_drill_down_click,
    calculate_next_drill_level,
    calculate_previous_drill_level,
    get_drill_level_display_name,
    initialize_drill_state,
    DRILL_LEVEL_NAMES,
)
from utils.ribbon_chart_builder import build_ribbon_chart

logger = logging.getLogger(__name__)


def register_evolution_callbacks(app):
    """
    Registra callbacks de evolución temporal.

    Callbacks:
    1. update_evolution_chart: Gráfico de evolución con drill-down
    """

    @app.callback(
        [
            Output("prescription-evolution-chart", "figure"),
            Output("prescription-drill-level-badge", "children"),
            Output("prescription-drill-breadcrumb", "children"),
            Output("prescription-drill-up-btn", "disabled"),
            Output("prescription-temporal-drill-store", "data"),
            Output("prescription-categories-filter-store", "data"),  # Issue #449: Propagar filtro a Store
        ],
        [
            Input("prescription-evolution-trigger-store", "data"),
            Input("prescription-evolution-chart", "clickData"),
            Input("prescription-drill-up-btn", "n_clicks"),
            Input("evolution-chart-type", "value"),
            Input("ribbon-top-n", "value"),
            Input("prescription-evolution-categories-filter", "value"),  # Nuevo filtro de categorías
        ],
        [
            State("prescription-overview-store", "data"),
            State("prescription-temporal-drill-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def update_evolution_chart(
        trigger_data: Optional[Dict],
        click_data: Optional[Dict],
        drill_up_clicks: int,
        chart_type: str,
        top_n: str,
        selected_categories: Optional[list],  # Nuevo parámetro
        overview_data: Optional[Dict],
        drill_state: Optional[Dict],
    ) -> tuple:
        """
        Actualiza gráfico de evolución temporal con drill-down.

        Soporta niveles: Trimestre → Mes → Quincena
        Usa helpers de drill_down_helpers.py para consistencia con /generics.
        """
        logger.debug("[update_evolution_chart] Callback triggered")

        # Inicializar drill_state si es None
        if not drill_state:
            drill_state = initialize_drill_state()

        current_level = drill_state.get("level", "quarter")
        current_path = drill_state.get("path", [])

        # Determinar qué trigger activó el callback (REGLA #8: Dash 3.x API)
        trigger_id = ctx.triggered_id if ctx.triggered else None

        # FIX Issue #436: Preferir datos del trigger para evitar race condition
        if trigger_data and trigger_data.get("overview_data"):
            overview_data = trigger_data["overview_data"]
            logger.debug("[update_evolution_chart] Using overview_data from trigger")

        # Procesar drill-down (click en gráfico)
        if trigger_id == "prescription-evolution-chart" and click_data:
            try:
                point = click_data.get("points", [{}])[0]
                clicked_period = point.get("customdata")
                if isinstance(clicked_period, (list, tuple)) and len(clicked_period) > 0:
                    clicked_period = clicked_period[0]
                if not clicked_period:
                    clicked_period = point.get("x")
                logger.debug(f"[update_evolution_chart] Click detected: {clicked_period}")
            except (KeyError, IndexError, TypeError):
                clicked_period = None

            if clicked_period:
                new_level, new_path = calculate_next_drill_level(current_level, current_path, clicked_period)
                if new_level and new_path is not None:
                    current_level = new_level
                    current_path = new_path
                    drill_state = {"level": current_level, "path": current_path}

        # Procesar drill-up (click en botón)
        elif trigger_id == "prescription-drill-up-btn":
            new_level, new_path, _ = calculate_previous_drill_level(current_path)
            current_level = new_level
            current_path = new_path
            drill_state = {"level": current_level, "path": current_path}

        # Preparar outputs por defecto
        level_badge = get_drill_level_display_name(current_level)
        drill_up_disabled = len(current_path) == 0

        # Breadcrumb
        if current_path:
            flat_path = [str(p[0]) if isinstance(p, (list, tuple)) else str(p) for p in current_path]
            breadcrumb = " > ".join(flat_path)
        else:
            breadcrumb = "Vista general"

        # Empty figure si no hay datos
        if not overview_data:
            empty_figure = {
                "data": [],
                "layout": {
                    "margin": {"t": 20, "b": 40, "l": 60, "r": 20},
                    "annotations": [
                        {
                            "text": "Aplica filtros para ver la evolución temporal",
                            "showarrow": False,
                            "font": {"size": 14, "color": COLORS["text_secondary"]},
                        }
                    ],
                },
            }
            return empty_figure, level_badge, breadcrumb, drill_up_disabled, drill_state, selected_categories

        # Obtener time_series del overview
        time_series = overview_data.get("time_series", [])

        if not time_series:
            empty_figure = {
                "data": [],
                "layout": {
                    "margin": {"t": 20, "b": 40, "l": 60, "r": 20},
                    "annotations": [
                        {
                            "text": "No hay datos de serie temporal disponibles",
                            "showarrow": False,
                            "font": {"size": 14, "color": COLORS["text_secondary"]},
                        }
                    ],
                },
            }
            return empty_figure, level_badge, breadcrumb, drill_up_disabled, drill_state, selected_categories

        try:
            # Determinar si estamos en nivel trimestre o mes
            is_quarter_level = current_level == "quarter" and len(current_path) == 0
            selected_quarter = current_path[0] if len(current_path) > 0 else None

            # Agrupar datos por categoría (aplicar filtro si hay categorías seleccionadas)
            data_by_category = defaultdict(lambda: {"periods": [], "sales": []})
            all_periods = set()

            for record in time_series:
                period = record.get("period") or record.get("date")
                category = record.get("category", "unknown")
                total_sales = float(record.get("sales", 0) or record.get("total_sales", 0))

                # Filtrar por categorías seleccionadas (si hay selección activa)
                if selected_categories and len(selected_categories) > 0:
                    if category not in selected_categories:
                        continue  # Saltar categorías no seleccionadas

                if is_quarter_level:
                    aggregated_period = get_quarter_from_period(period)
                else:
                    aggregated_period = period

                all_periods.add(aggregated_period)
                data_by_category[category]["periods"].append(aggregated_period)
                data_by_category[category]["sales"].append(total_sales)

            # Consolidar valores duplicados (múltiples meses → mismo trimestre)
            if is_quarter_level:
                for category in data_by_category:
                    period_sales = defaultdict(float)
                    for period, sales in zip(
                        data_by_category[category]["periods"],
                        data_by_category[category]["sales"],
                    ):
                        period_sales[period] += sales
                    data_by_category[category]["periods"] = list(period_sales.keys())
                    data_by_category[category]["sales"] = list(period_sales.values())

            # Ordenar períodos cronológicamente
            sorted_periods = sorted(list(all_periods))

            # Etiquetas legibles para eje X
            if is_quarter_level:
                x_labels = [get_quarter_label(p) for p in sorted_periods]
                x_title = "Trimestre (click para ver detalle mensual)"
                chart_title = "Evolución Trimestral de Ventas por Categoría"
            else:
                month_names = ["Ene", "Feb", "Mar", "Abr", "May", "Jun", "Jul", "Ago", "Sep", "Oct", "Nov", "Dic"]
                x_labels = []
                for p in sorted_periods:
                    try:
                        year, month = p.split("-")
                        x_labels.append(f"{month_names[int(month) - 1]} {year}")
                    except (ValueError, IndexError):
                        x_labels.append(p)
                x_title = "Mes (vista anual completa)"
                chart_title = f"Detalle Mensual - {get_quarter_label(selected_quarter) if selected_quarter else ''}"

            # Usar ribbon chart builder con todas las categorías
            # Si hay filtro de categorías, top_n = número de categorías seleccionadas
            # Si no hay filtro, top_n = 100 para mostrar todas las categorías disponibles
            if selected_categories and len(selected_categories) > 0:
                top_n_value = len(selected_categories)
            else:
                top_n_value = 100  # Suficiente para mostrar todas las categorías (hay 14)

            fig = build_ribbon_chart(
                data_by_category=data_by_category,
                sorted_periods=sorted_periods,
                x_labels=x_labels,
                top_n=top_n_value,
                chart_title=chart_title,
            )

            # Ajustar layout adicional para drill-down
            fig.update_layout(
                xaxis_title=x_title,
                paper_bgcolor="white",
                plot_bgcolor="white",
                clickmode="event+select",
            )

            return fig, level_badge, breadcrumb, drill_up_disabled, drill_state, selected_categories

        except Exception as e:
            logger.error(f"[update_evolution_chart] Error: {str(e)}", exc_info=True)
            error_figure = {
                "data": [],
                "layout": {
                    "margin": {"t": 20, "b": 40, "l": 60, "r": 20},
                    "annotations": [
                        {
                            "text": f"Error: {str(e)}",
                            "showarrow": False,
                            "font": {"size": 14, "color": COLORS["danger"]},
                        }
                    ],
                },
            }
            return error_figure, level_badge, breadcrumb, drill_up_disabled, drill_state, selected_categories

    logger.info("[prescription/evolution] 1 callback registered")
