"""
Data loading callbacks for prescription dashboard.

Handles API calls to fetch prescription data from backend.
"""

import logging
from typing import List, Optional, Dict

from dash import Input, Output, State
from dash.exceptions import PreventUpdate

from utils.auth_helpers import is_user_authenticated, get_auth_headers_from_tokens
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)


def _get_default_date_range() -> dict:
    """Returns default 90-day date range as fallback."""
    from datetime import datetime, timedelta
    end_date = str(datetime.now().date())
    start_date = str((datetime.now() - timedelta(days=90)).date())
    return {"min_date": start_date, "max_date": end_date, "ready": True}


def register_data_loading_callbacks(app):
    """
    Registra callbacks de carga de datos.

    Callbacks:
    1. load_all_prescription_data: Carga overview, ATC, waterfall y top-contributors
    2. fetch_prescription_date_range: Obtiene rango de fechas disponible
    """

    @app.callback(
        [
            Output("prescription-overview-store", "data"),
            Output("prescription-atc-distribution-store", "data"),
            Output("prescription-waterfall-store", "data"),
            Output("prescription-top-contributors-store", "data"),
        ],
        [
            Input("url", "pathname"),
            Input("prescription-apply-filters-btn", "n_clicks"),
            Input("prescription-date-range-store", "data"),  # Issue #483: Broadcast store (shared with filters.py)
            Input("auth-ready", "data"),  # Issue #483: Wait for auth sync before API calls
            # Issue #537 FIX: Añadir fechas como Inputs para recargar cuando cambian
            Input("prescription-date-range", "start_date"),
            Input("prescription-date-range", "end_date"),
        ],
        [
            State("waterfall-period-base", "value"),
            State("waterfall-period-comparison-store", "data"),
            State("waterfall-comparison-type", "value"),
            State("prescription-categories-filter", "value"),
            State("prescription-atc-codes-filter", "value"),
            State("atc-level-filter", "value"),
            State("prescription-laboratories-filter", "value"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
            State("prescription-categories-filter-store", "data"),  # FIX: Moved from Input to State to break circular dependency
        ],
        prevent_initial_call=False,
    )
    def load_all_prescription_data(
        pathname: str,
        n_clicks: int,
        date_range_store: Optional[Dict],  # Issue #483: Trigger when dates ready
        auth_ready: Optional[bool],  # Issue #483: Wait for auth sync
        start_date: str,  # Issue #537 FIX: Ahora es Input para recargar cuando cambia
        end_date: str,  # Issue #537 FIX: Ahora es Input para recargar cuando cambia
        # States (read-only):
        period_base: Optional[str],
        period_comparison_store: Optional[Dict],
        comparison_type: Optional[str],
        categories: Optional[List[str]],
        atc_codes: Optional[List[str]],
        atc_level: int,
        laboratory_codes: Optional[List[str]],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
        categories_filter_store: Optional[List[str]],  # FIX: Moved to end (now State)
    ) -> tuple:
        """
        Carga TODOS los datos de prescripción en un solo callback.

        Issue #302: Auth proactiva antes de API calls.
        Issue #436: Integración con 4 backend endpoints.
        Issue #483: Fix race condition - añadido date_range_store y auth-ready como Inputs.
        REGLA #11: UN solo callback escucha prescription-apply-filters-btn.
        """
        logger.debug(f"[load_all_prescription_data] Callback triggered - pathname: {pathname}")

        # Guard 1: Solo ejecutar en /prescription
        if pathname != "/prescription":
            logger.debug("[load_all_prescription_data] Not on /prescription - skipping")
            raise PreventUpdate

        # Guard 2: Esperar auth-ready (como /generics)
        if not auth_ready:
            logger.debug("[load_all_prescription_data] Waiting for auth-ready")
            raise PreventUpdate

        # Guard 3: Auth check proactivo (Issue #302)
        if not is_user_authenticated(auth_state):
            logger.debug("[load_all_prescription_data] User not authenticated - skipping API calls")
            raise PreventUpdate

        # REGLA #7.6: Multi-worker token restoration
        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        # Guard 4: Esperar date_range_store listo (como /generics)
        if not date_range_store or not date_range_store.get("ready"):
            logger.debug("[load_all_prescription_data] date_range_store not ready - waiting")
            raise PreventUpdate

        # Guard 5: Usar fechas del DatePicker si están, sino del store
        if not start_date or not end_date:
            start_date = date_range_store.get("min_date")
            end_date = date_range_store.get("max_date")
            logger.debug(f"[load_all_prescription_data] Using store dates: {start_date} to {end_date}")

        if not start_date or not end_date:
            logger.debug("[load_all_prescription_data] Still no dates after fallback - waiting")
            raise PreventUpdate

        # Obtener pharmacy_id
        user = auth_state.get("user") if auth_state else None
        if not user:
            logger.error("[load_all_prescription_data] No user in auth_state")
            raise PreventUpdate  # Consistent with other guards

        pharmacy_id = user.get("pharmacy_id")
        if not pharmacy_id:
            logger.error("[load_all_prescription_data] No pharmacy_id in auth_state")
            raise PreventUpdate  # Consistent with other guards

        # Preparar parámetros comunes
        common_params = {"date_from": start_date, "date_to": end_date}

        # Issue #510: Priorizar filtro de categorías del store (desde gráfico evolución)
        # Si el usuario selecciona categorías en el gráfico de evolución, se propaga a waterfall/contributors
        if categories_filter_store:
            effective_categories = categories_filter_store
            logger.debug(f"[load_all_prescription_data] Using categories from evolution chart filter: {effective_categories}")
        elif categories:
            effective_categories = categories
            logger.debug(f"[load_all_prescription_data] Using categories from dropdown filter: {effective_categories}")
        else:
            effective_categories = None

        if effective_categories:
            common_params["categories"] = effective_categories
        if atc_codes:
            common_params["atc_codes"] = atc_codes
        if laboratory_codes:
            common_params["laboratory_codes"] = laboratory_codes

        # 1. Llamar a /overview
        logger.info(f"[load_all_prescription_data] Calling /prescription/{pharmacy_id}/overview")

        overview_response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/overview",
            method="GET",
            params=common_params,
            timeout=30,
            auth_headers=auth_headers,
        )

        overview_data = None
        if overview_response and "error" not in overview_response:
            overview_data = overview_response
            logger.info(
                f"[load_all_prescription_data] Overview success - "
                f"{len(overview_data.get('time_series', []))} time series records"
            )
        else:
            error_msg = overview_response.get("message", "Unknown error") if overview_response else "No response"
            logger.error(f"[load_all_prescription_data] Overview failed: {error_msg}")

        # 2. Llamar a /distribution-by-atc
        atc_params = common_params.copy()
        atc_params["atc_level"] = atc_level or 1

        atc_response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/distribution-by-atc",
            method="GET",
            params=atc_params,
            timeout=30,
            auth_headers=auth_headers,
        )

        atc_data = None
        if atc_response and "error" not in atc_response:
            atc_data = atc_response
            logger.info(
                f"[load_all_prescription_data] ATC success - "
                f"{len(atc_data.get('atc_distribution', []))} top-level nodes"
            )
        else:
            error_msg = atc_response.get("message", "Unknown error") if atc_response else "No response"
            logger.error(f"[load_all_prescription_data] ATC failed: {error_msg}")

        # 3. Llamar a /waterfall-analysis
        waterfall_params = common_params.copy()

        # Issue #540 FIX: Calcular fechas YoY directamente en lugar de leer del store
        # Esto elimina la race condition donde el store no está actualizado
        # cuando este callback se dispara por cambio en DatePicker
        from datetime import datetime, timedelta

        try:
            # Limpiar formato de fecha (puede venir con T00:00:00)
            start_clean = start_date.split("T")[0] if start_date else None
            end_clean = end_date.split("T")[0] if end_date else None

            if start_clean and end_clean:
                start_dt = datetime.fromisoformat(start_clean)
                end_dt = datetime.fromisoformat(end_clean)

                # Calcular período YoY (-365 días)
                comp_start = start_dt - timedelta(days=365)
                comp_end = end_dt - timedelta(days=365)

                waterfall_params["comparison_period"] = "custom"
                waterfall_params["comparison_date_from"] = comp_start.strftime("%Y-%m-%d")
                waterfall_params["comparison_date_to"] = comp_end.strftime("%Y-%m-%d")

                logger.debug(
                    f"[load_all_prescription_data] YoY calculated: "
                    f"{start_clean} to {end_clean} vs {comp_start.date()} to {comp_end.date()}"
                )
            else:
                waterfall_params["comparison_period"] = "yoy"
        except Exception as e:
            logger.warning(f"[load_all_prescription_data] Error calculating YoY dates: {e}")
            waterfall_params["comparison_period"] = "yoy"

        waterfall_response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/waterfall-analysis",
            method="GET",
            params=waterfall_params,
            timeout=45,
            auth_headers=auth_headers,
        )

        waterfall_data = None
        if waterfall_response and "error" not in waterfall_response:
            waterfall_data = waterfall_response
            waterfall_items = waterfall_data.get("waterfall_data", [])
            logger.info(f"[load_all_prescription_data] Waterfall success - {len(waterfall_items)} category changes")
        else:
            error_msg = waterfall_response.get("message", "Unknown error") if waterfall_response else "No response"
            logger.error(f"[load_all_prescription_data] Waterfall failed: {error_msg}")

        # 4. Llamar a /top-contributors
        top_contributors_params = common_params.copy()
        top_contributors_params["comparison_period"] = waterfall_params.get("comparison_period", "yoy")
        top_contributors_params["limit"] = 50
        top_contributors_params["direction"] = "all"

        if "comparison_date_from" in waterfall_params:
            top_contributors_params["comparison_date_from"] = waterfall_params["comparison_date_from"]
            top_contributors_params["comparison_date_to"] = waterfall_params["comparison_date_to"]

        top_contributors_response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/top-contributors",
            method="GET",
            params=top_contributors_params,
            timeout=45,
            auth_headers=auth_headers,
        )

        top_contributors_data = None
        if top_contributors_response and "error" not in top_contributors_response:
            top_contributors_data = top_contributors_response
            contributors_count = len(top_contributors_data.get("contributors", []))
            logger.info(f"[load_all_prescription_data] Top Contributors success - {contributors_count} contributors")
        else:
            error_msg = (
                top_contributors_response.get("message", "Unknown error") if top_contributors_response else "No response"
            )
            logger.error(f"[load_all_prescription_data] Top Contributors failed: {error_msg}")

        return overview_data, atc_data, waterfall_data, top_contributors_data

    # =========================================================================
    # CALLBACK: Fetch date range
    # =========================================================================
    @app.callback(
        Output("prescription-date-range-store", "data"),
        [
            Input("url", "pathname"),
            Input("auth-ready", "data"),  # Issue #483: Retry when auth syncs
        ],
        [
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def fetch_prescription_date_range(
        pathname: str,
        auth_ready: Optional[bool],  # Issue #483: Wait for auth sync
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ):
        """Obtiene el rango de fechas disponible y lo guarda en Store."""
        if pathname != "/prescription":
            raise PreventUpdate

        # Issue #483: Wait for auth-ready before fetching
        if not auth_ready:
            logger.debug("[fetch_prescription_date_range] Waiting for auth-ready")
            raise PreventUpdate

        if not is_user_authenticated(auth_state):
            logger.debug("[fetch_prescription_date_range] User not authenticated")
            raise PreventUpdate

        auth_headers = get_auth_headers_from_tokens(auth_tokens)

        if not auth_headers:
            logger.warning("[fetch_prescription_date_range] No access token available")
            raise PreventUpdate

        user = auth_state.get("user") if auth_state else None
        if not user:
            logger.error("[fetch_prescription_date_range] No user in auth_state")
            raise PreventUpdate

        pharmacy_id = user.get("pharmacy_id")
        if not pharmacy_id:
            logger.error("[fetch_prescription_date_range] No pharmacy_id in auth_state")
            raise PreventUpdate

        logger.info(f"[fetch_prescription_date_range] Fetching date range for pharmacy {pharmacy_id}")

        response = request_coordinator.make_request(
            endpoint=f"/api/v1/sales/date-range/{pharmacy_id}",
            method="GET",
            timeout=15,
            auth_headers=auth_headers,
        )

        if not response or "error" in response:
            return _get_default_date_range()

        min_date = response.get("min_date") or response.get("oldest_date")
        max_date = response.get("max_date") or response.get("latest_date")

        if not min_date or not max_date:
            return _get_default_date_range()

        logger.info(f"[fetch_prescription_date_range] Date range: {min_date} to {max_date}")
        return {"min_date": min_date, "max_date": max_date, "ready": True}

    logger.info("[prescription/data_loading] 2 callbacks registered")
