"""
Contributors callbacks for prescription dashboard.

Handles top contributors table and waterfall click-to-filter.
"""

import logging
from typing import Optional, Dict, Any

from dash import Input, Output, State, html, ctx, ALL
from dash.exceptions import PreventUpdate
import dash_bootstrap_components as dbc

from utils.helpers import format_currency, format_percentage
from components.empty_state import create_analysis_empty_state
from dash_iconify import DashIconify
from styles.design_tokens import COLORS
from utils.constants import format_category_name

logger = logging.getLogger(__name__)

# Issue #510: Constantes de truncamiento para nombres largos
INGREDIENT_DISPLAY_LENGTH = 35  # Grupos/principios activos en cabecera
PRODUCT_DISPLAY_LENGTH = 28     # Productos en drill-down


def register_contributors_callbacks(app):
    """
    Registra callbacks de top contributors.

    Callbacks:
    1. update_top_contributors_table: Tabla de top contributors
    2. handle_waterfall_click: Click-to-filter desde waterfall
    """

    @app.callback(
        [
            Output("top-contributors-container", "children"),
            # Issue #458: Combined outputs for product waterfall (REGLA #11)
            Output("product-waterfall-placeholder", "style"),
            Output("product-waterfall-chart", "figure"),
            Output("product-waterfall-chart", "style"),
            Output("product-waterfall-category-badge", "children"),
        ],
        [
            Input("prescription-top-contributors-store", "data"),
            Input("top-contributors-limit", "value"),
            Input("top-contributors-direction", "value"),
        ],
        prevent_initial_call=True,
    )
    def update_top_contributors_and_product_waterfall(
        top_contributors_data: Optional[Dict],
        limit: int,
        direction: str,
    ) -> tuple:
        """
        Renderiza tabla de Top Contributors y Product Waterfall con filtrado client-side.

        Issue #436 Fase 2: Top Contributors por Principio Activo.
        Issue #458: Product Waterfall combinado (REGLA #11).
        Feature: Click-to-filter desde waterfall chart por categoría.

        Returns:
            tuple: (contributors_container, placeholder_style, chart_figure, chart_style, badge)
        """
        from components.prescription.waterfall_analysis import create_product_waterfall_chart

        logger.debug("[update_top_contributors_and_product_waterfall] Callback triggered")

        # Default product waterfall outputs (placeholder visible, chart hidden)
        placeholder_visible = {"display": "block"}
        chart_hidden = {"height": "500px", "display": "none"}
        empty_figure = {"data": [], "layout": {"height": 500}}
        no_badge = ""

        if not top_contributors_data:
            logger.warning("[update_top_contributors_and_product_waterfall] No top contributors data available")
            empty_state = create_analysis_empty_state(
                message="No hay datos de top contributors disponibles",
                icon="mdi:format-list-bulleted-type",
                suggestions=[
                    "Aplica filtros para visualizar el análisis",
                    "Verifica que existan ventas en el período seleccionado",
                ],
                min_height="300px",
                section_name="Top Contributors",
            )
            return empty_state, placeholder_visible, empty_figure, chart_hidden, no_badge

        contributors = top_contributors_data.get("contributors", [])
        total_change = top_contributors_data.get("total_change", 0)
        comparison_period = top_contributors_data.get("comparison_period", "yoy")
        # Issue #457: Read category filter from API response metadata
        filters_applied = top_contributors_data.get("filters_applied", {})
        category_filter = filters_applied.get("categories")

        if not contributors:
            logger.info("[update_top_contributors_and_product_waterfall] Empty contributors list")
            empty_state = create_analysis_empty_state(
                message="No hay principios activos con cambios significativos en el período seleccionado",
                icon="mdi:trending-neutral",
                suggestions=[
                    "Las ventas pueden ser similares al período anterior",
                    "Prueba con un rango de fechas diferente",
                    "Verifica cobertura ATC de los productos vendidos",
                ],
                min_height="300px",
                section_name="Top Contributors",
            )
            return empty_state, placeholder_visible, empty_figure, chart_hidden, no_badge

        # Issue #457: Server-side filtering by category is done in the API
        # category_filter is now for display purposes only (showing filter badge)
        if category_filter:
            logger.info(f"[update_top_contributors_table] Data filtered by category: {category_filter}")

        # Filtrar por dirección (client-side)
        if direction == "up":
            contributors = [c for c in contributors if float(c.get("variation_euros", 0)) > 0]
        elif direction == "down":
            contributors = [c for c in contributors if float(c.get("variation_euros", 0)) < 0]

        # Issue #443: Deduplicate by active_ingredient
        seen_ingredients = {}
        deduplicated = []
        for c in contributors:
            ingredient = c.get("active_ingredient", "N/A")
            normalized = ingredient.strip().upper() if ingredient else "N/A"
            if normalized not in seen_ingredients:
                seen_ingredients[normalized] = len(deduplicated)
                deduplicated.append(c.copy())
            else:
                idx = seen_ingredients[normalized]
                existing = deduplicated[idx]
                existing["variation_euros"] = float(existing.get("variation_euros", 0)) + float(
                    c.get("variation_euros", 0)
                )
                existing["variation_percent"] = float(existing.get("variation_percent", 0)) + float(
                    c.get("variation_percent", 0)
                )
                existing["impact_percent"] = float(existing.get("impact_percent", 0)) + float(
                    c.get("impact_percent", 0)
                )
                logger.debug(f"[update_top_contributors_table] Aggregated duplicate ingredient: {ingredient}")

        contributors = deduplicated
        logger.debug(f"[update_top_contributors_table] After deduplication: {len(contributors)} unique ingredients")

        # Aplicar límite
        contributors = contributors[:limit]

        if not contributors:
            direction_text = "subidas" if direction == "up" else "bajadas" if direction == "down" else ""
            icon = (
                "mdi:trending-up" if direction == "up" else "mdi:trending-down" if direction == "down" else "mdi:filter-off"
            )

            if category_filter:
                category_name = format_category_name(category_filter)
                message = f"No hay principios activos en la categoría '{category_name}'"
                if direction_text:
                    message += f" con {direction_text}"
                suggestions = [
                    "Haz clic en otra barra del gráfico de variación para cambiar la categoría",
                    "Usa el botón 'Limpiar' para ver todos los contribuyentes",
                    "Prueba con el filtro de dirección 'Todos'",
                ]
                icon = "mdi:filter-off"
            else:
                message = f"No hay principios activos con {direction_text} en el período seleccionado"
                suggestions = [
                    "Prueba con el filtro 'Todos' para ver todos los cambios",
                    "Ajusta el período de comparación (YoY/MoM/QoQ)",
                ]

            empty_state = create_analysis_empty_state(
                message=message,
                icon=icon,
                suggestions=suggestions,
                min_height="300px",
                section_name="Top Contributors",
            )
            return empty_state, placeholder_visible, empty_figure, chart_hidden, no_badge

        category_info = f", category={category_filter}" if category_filter else ""
        logger.info(
            f"[update_top_contributors_and_product_waterfall] Rendering {len(contributors)} contributors "
            f"(limit={limit}, direction={direction}{category_info}) as accordion"
        )

        # Mapeo de período de comparación
        period_labels = {
            "yoy": "Año anterior",
            "mom": "Mes anterior",
            "qoq": "Trimestre anterior",
            "custom": "Período seleccionado",
        }
        period_label = period_labels.get(comparison_period, comparison_period.upper())

        # Issue #449: Mapeo de tipos de agrupación a iconos y tooltips
        GROUPING_TYPE_CONFIG = {
            "homogeneous_group": {
                "icon": "mdi:package-variant-closed",
                "tooltip": "Grupo Homogéneo",
                "color": COLORS["info"],
            },
            "active_ingredient": {
                "icon": "mdi:pill",
                "tooltip": "Principio Activo",
                "color": COLORS["primary"],
            },
            "product": {
                "icon": "mdi:cube-outline",
                "tooltip": "Producto",
                "color": COLORS["text_muted"],
            },
        }

        try:
            # Issue #441 + Issue #450: Crear lista mixta (acordeón + items simples)
            # - homogeneous_group / active_ingredient → Acordeón (desplegable)
            # - product → ListGroupItem simple (no desplegable, es producto único)
            accordion_items = []
            standalone_items = []  # Issue #450: Productos únicos sin desplegable

            # Issue #510: Cabecera de columnas para la tabla de contributors
            column_header = dbc.Row(
                [
                    dbc.Col(
                        html.Small("Producto / Grupo", className="text-muted fw-bold"),
                        width=4,
                    ),
                    dbc.Col(
                        html.Small("Ventas €", className="text-muted fw-bold"),
                        width=2,
                        className="text-end",
                    ),
                    dbc.Col(
                        html.Small("Var €", className="text-muted fw-bold"),
                        width=2,
                        className="text-end",
                    ),
                    dbc.Col(
                        html.Small("Var %", className="text-muted fw-bold"),
                        width=2,
                        className="text-end",
                    ),
                    dbc.Col(
                        html.Small("% Responsable", className="text-muted fw-bold"),
                        width=2,
                        className="text-end",
                    ),
                ],
                className="w-100 align-items-center py-2 px-3 border-bottom",
                style={"backgroundColor": COLORS["bg_tertiary"]},
            )

            for idx, c in enumerate(contributors):
                active_ingredient = c.get("active_ingredient", "N/A")
                variation_euros = float(c.get("variation_euros", 0))
                variation_percent = float(c.get("variation_percent", 0))
                impact_percent = float(c.get("impact_percent", 0))  # Issue #510: % Responsable
                current_sales = float(c.get("current_sales", 0))  # Issue #510: Ventas actuales
                grouping_type = c.get("grouping_type", "active_ingredient")  # Issue #449

                # Issue #449: Config del tipo de agrupación
                grouping_config = GROUPING_TYPE_CONFIG.get(grouping_type, GROUPING_TYPE_CONFIG["product"])

                # Color y icono según dirección de variación
                if variation_euros > 0:
                    var_color = COLORS["success"]
                    direction_icon = "mdi:trending-up"
                    var_sign = "+"
                elif variation_euros < 0:
                    var_color = COLORS["danger"]
                    direction_icon = "mdi:trending-down"
                    var_sign = ""
                else:
                    var_color = COLORS["text_muted"]
                    direction_icon = "mdi:minus"
                    var_sign = ""

                # Issue #510: Color para % Responsable
                # Positivo = contribuye en la misma dirección que el total
                # Negativo = contrarresta la tendencia
                # Cero = sin contribución neta
                if impact_percent > 0:
                    impact_color = COLORS["success"]  # Contribuye al cambio (verde = añade valor)
                    impact_icon = "mdi:arrow-right-bold"
                    impact_tooltip = "Contribuye al cambio total"
                elif impact_percent < 0:
                    impact_color = COLORS["info"]  # Contrarresta (azul = información neutral)
                    impact_icon = "mdi:arrow-left-bold"
                    impact_tooltip = "Contrarresta la tendencia"
                else:
                    impact_color = COLORS["text_muted"]  # Sin contribución
                    impact_icon = "mdi:minus"
                    impact_tooltip = "Sin contribución neta"

                header_content = dbc.Row(
                    [
                        # Columna 1: Producto (4)
                        dbc.Col(
                            [
                                # Issue #449: Icono de tipo de agrupación
                                DashIconify(
                                    icon=grouping_config["icon"],
                                    width=14,
                                    color=grouping_config["color"],
                                    className="me-1",
                                    style={"opacity": "0.7"},
                                ),
                                # Issue #510: Indicador visual de dirección
                                DashIconify(icon=direction_icon, width=16, color=var_color, className="me-1"),
                                html.Span(
                                    active_ingredient[:INGREDIENT_DISPLAY_LENGTH] + "..." if len(active_ingredient) > INGREDIENT_DISPLAY_LENGTH else active_ingredient,
                                    style={"fontWeight": "600", "fontSize": "0.85rem"},
                                    title=active_ingredient,  # Tooltip completo
                                ),
                            ],
                            width=4,
                        ),
                        # Columna 2: Ventas € (2)
                        dbc.Col(
                            html.Span(
                                format_currency(current_sales),
                                style={"fontSize": "0.85rem"},
                            ),
                            width=2,
                            className="text-end",
                        ),
                        # Columna 3: Var € (2) - Issue #510
                        dbc.Col(
                            html.Span(
                                f"{var_sign}{format_currency(abs(variation_euros))}",
                                style={"fontWeight": "600", "color": var_color, "fontSize": "0.85rem"},
                            ),
                            width=2,
                            className="text-end",
                        ),
                        # Columna 4: Var % (2)
                        dbc.Col(
                            html.Span(
                                f"{var_sign}{format_percentage(abs(variation_percent) / 100)}",
                                style={"color": var_color, "fontSize": "0.85rem"},
                            ),
                            width=2,
                            className="text-end",
                        ),
                        # Columna 5: % Responsable (2) - Issue #510
                        dbc.Col(
                            html.Span(
                                [
                                    DashIconify(
                                        icon=impact_icon,
                                        width=12,
                                        color=impact_color,
                                        className="me-1",
                                    ),
                                    f"{abs(impact_percent):.1f}%",
                                ],
                                style={"color": impact_color, "fontSize": "0.85rem"},
                                title=impact_tooltip,
                            ),
                            width=2,
                            className="text-end",
                        ),
                    ],
                    className="w-100 align-items-center",
                )

                # Issue #450: Productos únicos → ListGroupItem simple (sin desplegable)
                # Issue #457: Mostrar variación de unidades para productos
                if grouping_type == "product":
                    variation_units = c.get("variation_units")
                    units_display = None
                    if variation_units is not None:
                        units_sign = "+" if variation_units > 0 else ""
                        units_color = (
                            COLORS["success"] if variation_units > 0
                            else COLORS["danger"] if variation_units < 0
                            else COLORS["text_muted"]
                        )
                        units_display = html.Small(
                            f"({units_sign}{variation_units:,} uds.)",
                            style={"color": units_color, "marginLeft": "8px"},
                        )

                    # Combine header with units
                    product_content = html.Div([
                        header_content,
                        html.Div(
                            units_display if units_display else "",
                            className="text-end",
                            style={"marginTop": "-4px"},
                        ) if units_display else None,
                    ])

                    standalone_items.append(
                        dbc.ListGroupItem(
                            product_content if units_display else header_content,
                            className="py-2 px-3",
                            style={"backgroundColor": "#f8f9fa"},  # Ligeramente diferente para distinguir
                        )
                    )
                else:
                    # Issue #443: Use unique index to prevent duplicate IDs
                    # Include grouping_type to ensure uniqueness across different grouping types
                    # Fix: Use :: as delimiter to avoid conflicts with grouping_type containing "_"
                    unique_index = f"{idx}::{grouping_type}::{active_ingredient}"

                    body_content = html.Div(
                        id={"type": "ingredient-products-container", "index": unique_index},
                        children=[
                            html.Div(
                                [
                                    dbc.Spinner(size="sm", color="primary", spinner_class_name="me-2"),
                                    html.Small("Cargando productos...", className="text-muted"),
                                ],
                                className="d-flex align-items-center justify-content-center py-3",
                            )
                        ],
                        style={"minHeight": "80px"},
                    )

                    # Issue #451: Incluir grouping_type en item_id para drill-down correcto
                    # Formato: "grouping_type::active_ingredient" (:: como delimitador)
                    accordion_item_id = f"{grouping_type}::{active_ingredient}"

                    accordion_items.append(
                        dbc.AccordionItem(
                            body_content,
                            title=header_content,
                            item_id=accordion_item_id,
                        )
                    )

            # Crear resumen
            summary_text = f"Comparación: {period_label} | Cambio total: {format_currency(float(total_change))}"

            # Issue #450: Construir componentes según qué hay
            children = [html.Small(summary_text, className="text-muted d-block mb-2")]

            # Issue #510: Añadir cabecera de columnas
            children.append(column_header)

            if accordion_items:
                children.append(
                    html.Small(
                        " (click para ver productos)",
                        className="text-muted",
                        style={"fontSize": "0.75rem"},
                    )
                )
                children.append(
                    dbc.Accordion(
                        accordion_items,
                        id="prescription-contributors-accordion",
                        flush=True,
                        always_open=False,
                        start_collapsed=True,
                        className="mt-2",
                    )
                )

            if standalone_items:
                if accordion_items:
                    # Separador si hay ambos tipos
                    children.append(
                        html.Hr(className="my-2", style={"opacity": "0.3"})
                    )
                    children.append(
                        html.Small(
                            "Productos únicos:",
                            className="text-muted d-block mb-1",
                            style={"fontSize": "0.75rem"},
                        )
                    )
                children.append(
                    dbc.ListGroup(
                        standalone_items,
                        flush=True,
                        className="mt-1",
                    )
                )

            contributors_container = html.Div(children)

            # Issue #458: Generate product waterfall outputs
            # FIXED: Only show product waterfall when a SPECIFIC category is selected
            # When no category filter, show placeholder instead (avoid duplicate info with contributors table)
            waterfall_category_filter = category_filter
            if waterfall_category_filter:
                if isinstance(waterfall_category_filter, list):
                    waterfall_category_filter = waterfall_category_filter[0] if waterfall_category_filter else None

            if waterfall_category_filter:
                # Category selected -> Show product waterfall for that category
                category_name_waterfall = format_category_name(waterfall_category_filter)
                waterfall_badge = dbc.Badge(
                    category_name_waterfall,
                    color="info",
                    pill=True,
                    className="ms-2",
                )
                logger.info(f"[update_top_contributors_and_product_waterfall] Rendering product waterfall: {category_name_waterfall}")

                # Create the product waterfall chart
                chart = create_product_waterfall_chart(top_contributors_data, category_name_waterfall)

                # Hide placeholder, show chart
                placeholder_hidden = {"display": "none"}
                chart_visible = {"height": "500px", "display": "block"}

                return contributors_container, placeholder_hidden, chart.figure, chart_visible, waterfall_badge

            # No category filter -> Show placeholder, hide chart (avoid duplicate info with contributors table)
            logger.debug("[update_top_contributors_and_product_waterfall] No category filter - showing placeholder")
            return contributors_container, placeholder_visible, empty_figure, chart_hidden, no_badge

        except Exception as e:
            logger.error(
                f"[update_top_contributors_and_product_waterfall] Error rendering: {str(e)}",
                exc_info=True,
            )
            error_div = html.Div(
                html.P(
                    f"Error al renderizar acordeón de top contributors: {str(e)}",
                    className="text-danger text-center my-5",
                ),
                className="d-flex justify-content-center align-items-center",
                style={"minHeight": "300px"},
            )
            return error_div, placeholder_visible, empty_figure, chart_hidden, no_badge

    @app.callback(
        [
            Output("contributors-category-filter-store", "data"),
            Output("contributors-filter-category-name", "children"),
            Output("contributors-filter-row", "style"),
        ],
        [
            Input("waterfall-analysis-chart", "clickData"),
            Input({"type": "waterfall-category-chip", "index": ALL}, "n_clicks"),
            Input("contributors-clear-filter-btn", "n_clicks"),
        ],
        [
            State({"type": "waterfall-category-chip", "index": ALL}, "id"),
        ],
        prevent_initial_call=True,
    )
    def handle_waterfall_click(
        click_data: Optional[Dict],
        chip_clicks: list,
        clear_clicks: int,
        chip_ids: list,
    ) -> tuple:
        """
        Maneja clic en waterfall chart O chips para filtrar top contributors.

        Issue #454: Añade soporte para clicks en chips de categoría,
        mejorando la UX cuando las barras del waterfall son muy pequeñas.
        """
        trigger_id = ctx.triggered_id if ctx.triggered else None

        # Si se presionó el botón limpiar, resetear filtro
        if trigger_id == "contributors-clear-filter-btn":
            logger.debug("[handle_waterfall_click] Clear filter button clicked")
            return None, "", {"display": "none"}

        # Issue #454: Click en chip de categoría
        if isinstance(trigger_id, dict) and trigger_id.get("type") == "waterfall-category-chip":
            category_key = trigger_id.get("index")
            if category_key:
                category_name = format_category_name(category_key)
                logger.info(f"[handle_waterfall_click] Chip clicked: {category_key}")
                return category_key, category_name, {"display": "block"}
            raise PreventUpdate

        # Click en gráfico waterfall
        if trigger_id == "waterfall-analysis-chart":
            if not click_data or "points" not in click_data or not click_data["points"]:
                raise PreventUpdate

            point = click_data["points"][0]
            category_key = point.get("customdata")

            if not category_key:
                logger.debug("[handle_waterfall_click] Clicked on base/total period, clearing filter")
                return None, "", {"display": "none"}

            category_name = format_category_name(category_key)
            logger.info(f"[handle_waterfall_click] Waterfall bar clicked: {category_key}")
            return category_key, category_name, {"display": "block"}

        raise PreventUpdate

    @app.callback(
        Output("prescription-top-contributors-store", "data", allow_duplicate=True),
        Input("contributors-category-filter-store", "data"),
        [
            State("prescription-top-contributors-store", "data"),
            State("auth-state", "data"),
            State("auth-tokens-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def fetch_category_filtered_contributors(
        category_filter: Optional[str],
        current_data: Optional[Dict],
        auth_state: Optional[Dict],
        auth_tokens: Optional[Dict],
    ) -> Optional[Dict]:
        """
        Issue #457 + #458: Fetch top contributors, optionally filtered by category.

        - category_filter set: Fetch products for that specific category
        - category_filter None: Fetch ALL products (when filter is cleared)
        """
        from utils.auth_helpers import is_user_authenticated, get_auth_headers_from_tokens
        from utils.auth import auth_manager
        from utils.request_coordinator import request_coordinator

        if not is_user_authenticated(auth_state):
            logger.warning("[fetch_category_filtered_contributors] User not authenticated")
            raise PreventUpdate

        # REGLA #7.6: Restore tokens for multi-worker (Render)
        if auth_tokens and "tokens" in auth_tokens:
            auth_manager.restore_from_encrypted_tokens(auth_tokens["tokens"])

        auth_headers = get_auth_headers_from_tokens(auth_tokens)
        if not auth_headers:
            logger.warning("[fetch_category_filtered_contributors] No access token")
            raise PreventUpdate

        user = auth_state.get("user") if auth_state else None
        if not user:
            logger.error("[fetch_category_filtered_contributors] No user in auth_state")
            raise PreventUpdate

        pharmacy_id = user.get("pharmacy_id")
        if not pharmacy_id:
            logger.error("[fetch_category_filtered_contributors] No pharmacy_id")
            raise PreventUpdate

        # Get date params from current data (from initial load)
        if not current_data:
            logger.warning("[fetch_category_filtered_contributors] No current data")
            raise PreventUpdate

        filters_applied = current_data.get("filters_applied", {})
        date_from = filters_applied.get("date_from")
        date_to = filters_applied.get("date_to")
        comparison_period = current_data.get("comparison_period", "yoy")

        if not date_from or not date_to:
            logger.warning("[fetch_category_filtered_contributors] No dates in current data")
            raise PreventUpdate

        logger.info(
            f"[fetch_category_filtered_contributors] Fetching contributors for category: {category_filter}"
        )

        # Build params with categories filter
        params = {
            "date_from": date_from,
            "date_to": date_to,
            "comparison_period": comparison_period,
            "limit": 50,
            "direction": "all",
            "categories": category_filter,  # Issue #457: Server-side category filter
        }

        # Add custom comparison dates if present
        if "prev_from" in filters_applied and comparison_period == "custom":
            params["comparison_date_from"] = filters_applied["prev_from"]
            params["comparison_date_to"] = filters_applied["prev_to"]

        response = request_coordinator.make_request(
            endpoint=f"/api/v1/prescription/{pharmacy_id}/top-contributors",
            method="GET",
            params=params,
            timeout=30,
            auth_headers=auth_headers,
        )

        if response and "error" not in response:
            contributors_count = len(response.get("contributors", []))
            logger.info(
                f"[fetch_category_filtered_contributors] Success: {contributors_count} contributors for {category_filter}"
            )
            return response

        error_msg = response.get("message", "Unknown error") if response else "No response"
        logger.error(f"[fetch_category_filtered_contributors] Failed: {error_msg}")
        raise PreventUpdate

    # Note: render_product_waterfall merged into update_top_contributors_and_product_waterfall (REGLA #11)
    # Note: reset_product_waterfall_on_clear removed - handled by update_top_contributors_and_product_waterfall when data reloads

    logger.info("[prescription/contributors] 3 callbacks registered")
